package com.tenqube.visual_scraper.webviewhtmlloader

import android.webkit.WebView
import com.tenqube.visual_scraper.webviewhtmlloader.model.Response
import com.tenqube.visual_scraper.webviewhtmlloader.model.ScrapingUrl
import com.tenqube.visual_scraper.webviewhtmlloader.model.StatusCode
import com.tenqube.visual_scraper.webviewhtmlloader.model.WebViewResponse
import com.tenqube.visual_scraper.webviewhtmlloader.util.WebViewManager
import com.tenqube.visual_scraper.market.dto.Back
import kotlinx.coroutines.*
import timber.log.Timber
import com.tenqube.visual_scraper.shared.util.Result


class WebViewHtmlLoader(private val webViewManager: WebViewManager): WebViewLoader {
    private val scrapings : MutableList<Any> = mutableListOf()


    private var timeOutPeriod: Long = 30000
    private var retryCnt: Int = 1
    private var currentCnt: Int = 0
    private var isSkipDelay = false

    override fun getWebView(): WebView? {
        return webViewManager.webView
    }


    override fun skipDelay() = apply {
        isSkipDelay = true
    }

    override fun noCache() = apply {
//        webViewManager.webView?.clearCache(true)
    }

    override fun back(isInterceptUrl: Boolean) = apply {
        scrapings.add(Back(isInterceptUrl))
    }

    override fun onPageCallback(callbackJs: String, callback: OnPageLoadedCallback?) = apply {
        webViewManager.setOnPageCallback(callbackJs, callback)
    }

    // 크롤링할 페이지 지정하는 함수
    // 인자로 true 값을 준 함수 갯수만큼 페이지들을 크롤링
    override fun go(url: String, isScrappingPage: Boolean, onSuccessUrl: String?, timeOut: Long?) = apply {

        if(url.isNotEmpty()) {
            scrapings.add(ScrapingUrl(url, url.contains("javascript:"), isScrappingPage, onSuccessUrl, timeOut))

        }
    }

    override fun delayTime(time: Long) = apply {
        scrapings.add(time)
    }

    override suspend fun load(): Result<WebViewResponse> = withContext(Dispatchers.Main) {

        return@withContext try {
            webViewManager.initialize() // 기존 데이터 초기화 작업을 진행합니다.

            withTimeout(timeOutPeriod) {
                val response = loadUrls()
                Result.Success(
                    WebViewResponse(response.statusCode,
                    url = response.url,
                        callbackUrl = response.callbackUrl,
                    msg = response.msg,
                    alertJS = response.alertJS,
                    html = response.html)
                )
            }
        } catch (e: Exception) {
            Result.Error(e)
        } finally {
            clearWebViewHtmlLoader()
        }
    }

    private suspend fun loadUrls(): Response {

        var response = Response(StatusCode.Success, "", "")
        var requestUrl: String? = null
        loop@ for (any in scrapings) {

            when(any) {

                is Back -> {
                    webViewManager.goBack()

                    if (any.isInterceptUrl) {
                        requestUrl = webViewManager.getRequestUrl()

                        Timber.d(  "WEBTEST requestUrl $requestUrl")

                    }
                }
                is Long -> {
                    delay(any)
                }
                is ScrapingUrl -> {
                    response =
                            if (requestUrl != null)

                                call(any.copy(path = requestUrl))
                            else
                                call(any)

                    if(!isSkipDelay)
                        delay(DEFAULT_DELAY)

                    if(response.statusCode != StatusCode.Success) { // 하나라도 성공 아닌경우 더이상 진행 불가함
                        break@loop
                    }
                }
            }
        }

        return response
    }


    private suspend fun call(url: ScrapingUrl): Response {
        val response = webViewManager.call(url.path, url.isJs, url.hasHtml, url.onSuccessUrl, url.userTimeOut)

        if(response.statusCode == StatusCode.Retry) { // js 실행 후 에러난 경우 (웹페이지에 js 가 없는 경우, 다른 사이트로 redirect 된경우)

            return if(currentCnt < retryCnt) {

                if(!isSkipDelay)
                    delay(DEFAULT_DELAY)
                currentCnt += 1
                call(url)

            } else { // 이미 로그인됨 or 자바스크립트가 존재하지 않는 케이스
                response.copy(statusCode = StatusCode.Success)
            }
        }
        return response
    }

    private fun clearWebViewHtmlLoader() {
        webViewManager.webView?.clearCache(true)
        scrapings.clear()
    }

    companion object {
        const val DEFAULT_DELAY = 500L
    }

}
