package com.tenqube.visual_scraper.market

import android.content.Context
import android.text.TextUtils
import android.view.View
import com.google.gson.Gson
import com.google.gson.JsonArray
import com.google.gson.JsonObject
import com.tenqube.commerce.domain.vo.CommerceId
import com.tenqube.visual_scraper.MallViewHandler
import com.tenqube.visual_scraper.market.dto.GmarketRule
import com.tenqube.visual_scraper.market.dto.MarketResult
import com.tenqube.visual_scraper.market.dto.OrderItem
import com.tenqube.visual_scraper.market.util.Utils
import com.tenqube.visual_scraper.market.dto.ScrapingCode
import com.tenqube.visual_scraper.webviewhtmlloader.OnPageLoadedCallback
import com.tenqube.visual_scraper.webviewhtmlloader.WebViewLoader
import com.tenqube.visual_scraper.webviewhtmlloader.model.StatusCode
import com.tenqube.visual_scraper.webviewhtmlloader.model.WebViewResponse
import com.tenqube.visual_scraper.webviewhtmlloader.util.hide
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import org.jsoup.Jsoup
import timber.log.Timber
import java.text.SimpleDateFormat
import java.util.*
import com.tenqube.visual_scraper.webviewhtmlloader.util.WebLoader
import com.tenqube.visual_scraper.shared.util.getValue
import com.tenqube.visual_scraper.shared.util.Result
import kotlinx.coroutines.GlobalScope
import kotlinx.coroutines.launch

import java.util.regex.Pattern


class Gmarket(
        private val webViewHtmlLoader: WebViewLoader, override val key: MarketKey,
        override val context: Context, override val mallViewHandler: MallViewHandler?
) : Market {

    private var rule: GmarketRule

    private var userId: String = ""
    private var userPwd: String = ""

    init {
        val marketRule = Utils.getJsonDataFromAsset(context, "gmarket_rules.json")?:  ""
        rule = Gson().fromJson(marketRule, GmarketRule::class.java)
    }
    // 서버에서 받아올 값

    private var scrapingResult = MarketResult(ScrapingCode.UnknownError)

    // 콜백 방식으로 데이터 리턴
    override suspend fun load(
            id: String,
            password: String
    ): MarketResult = withContext(Dispatchers.Main) {

        scrapingResult = MarketResult(ScrapingCode.UnknownError)

        try {

            scrapingResult = login(id, password).run {
                order()
            }.run {
                parsing(this)
            }.run {
                filter(this)
            }.run {
                toResult(this)
            }

        } catch (e: Exception) {
            e.toString()
        }
        return@withContext scrapingResult.also {
            Timber.d("load 응답값: $it")
        }
    }

    override suspend fun signIn(id: String, password: String): MarketResult = withContext(Dispatchers.Main) {

        scrapingResult = MarketResult(ScrapingCode.UnknownError)

        try {
            login(id, password)
        } catch (e: Exception) {
            e.toString()
        }
        return@withContext scrapingResult.also {
            Timber.d("signIn 응답값: $it")
        }
    }

    override suspend fun scrapOrder(): MarketResult = withContext(Dispatchers.Main) {

        scrapingResult = MarketResult(ScrapingCode.UnknownError)

        try {

            scrapingResult = order().run {
                parsing(this)
            }.run {
                filter(this)
            }.run {
                toResult(this)
            }

        } catch (e: Exception) {
            e.toString()
        }
        return@withContext scrapingResult.also {
            Timber.d("load 응답값: $it")
        }
    }

    override suspend fun logOut() {

        Timber.i("LOGOUT")
        Timber.i("LOGOUT" + rule.logoutUrl)
        Timber.i("LOGOUT" + rule.logout)


        val loginResult = webViewHtmlLoader
            .go(rule.logoutUrl)
            .go(rule.logout, true)
            .load()
            .getValue()
    }

    private suspend fun login(id: String, password: String) {
        // html load
        userId = id
        userPwd = password


        val loginResult = webViewHtmlLoader
                .noCache()
                .go(rule.loginUrl)
                .go(getLoginUrl(id, password), true, "http://mobile.gmarket.co.kr/")
                .load()
                .getValue()

        val result = checkLoginHtml(loginResult)
        when(result.code == ScrapingCode.Success) {
            true -> {
                scrapingResult = MarketResult(ScrapingCode.Success)
            }
            false -> {
                scrapingResult = MarketResult(result.code, result.msg)
                throw Exception()
            }
        }
    }

    private suspend fun order(): String {
        val orderResult = webViewHtmlLoader
                .go(rule.orderUrl, true, null, userTimeOut = 10000)
                .load()
                .getValue()
                .also {
                    Timber.d("order ${it.isSuccess}")
                }
        when (orderResult.html == null) {

            true -> {
                scrapingResult = MarketResult(ScrapingCode.UnknownError, "html is empty")
                throw Exception()
            }
            false -> {
                return orderResult.html
            }
        }
    }

    private fun parsing(html: String): JsonArray {
        return loadJsoupHtmlParser(html).getValue()
    }

    private fun filter(jsonArray: JsonArray): List<OrderItem> {
        return loadParsedDataFilter(jsonArray)
    }

    private fun toResult(orderItems: List<OrderItem>): MarketResult {
        return MarketResult(
                ScrapingCode.Success, "success",
                results = orderItems).also {
            Timber.d("results ${it.results}")
        }
    }

    /**
     * url 성공 체크를 넣자
     * 디폴트는 성공 진행
     */
    private fun checkLoginHtml(response: WebViewResponse): MarketResult {
        Timber.d("checkLoginHtml rule ${response.isSuccess}")

        return when (response.isSuccess) {

            true -> {
                response.alertJS?.let {

                    Timber.d("checkLoginHtml alertJS ${response.alertJS}")

                    Timber.d("checkLoginHtml alertJS ${rule.loginErrorUrl}")
                    Timber.d("checkLoginHtml alertJS ${it.url}")

                    Timber.d("checkLoginHtml alertJS ${rule.loginErrorUrl == it.url}")

                    if (rule.loginErrorUrl == it.url) {
                        return MarketResult(ScrapingCode.IdOrPwdError, msg = it.msg ?: "")
                    }
                }
                // 한번더 로그인 페이지 호출해서 로그인페이지가 뜨는지 확인

                MarketResult(
                        code = ScrapingCode.Success,
                        msg = "",
                        results = emptyList()
                )
            }
            false -> {


                response.alertJS?.let {
                    if (rule.loginErrorUrl == it.url) {
                        return MarketResult(ScrapingCode.IdOrPwdError, msg = it.msg ?: "")
                    }
                }

                response.html?.let {

                    Timber.d("checkLoginHtml rule ${rule.captcha}")
                    Timber.d("checkLoginHtml ${it.contains(rule.captcha)}")

                    if (it.contains(rule.captcha)) {

                        onCaptchaResult()
                        return MarketResult(ScrapingCode.CaptchaError, webView = webViewHtmlLoader.getWebView())
                    }
                }

                Timber.d("checkLoginHtml response.callbackUrl ${response.callbackUrl}")
                Timber.d("checkLoginHtml response.callbackUrl ${rule.loginSuccessUrl}")

                MarketResult(
                    code = if (response.code == StatusCode.TimeOut && isSuccessUrl(response.callbackUrl) )
                        ScrapingCode.Success
                    else
                        ScrapingCode.NetworkError,
                    msg = response.msg ?: "",
                    results = emptyList()
                )
            }
        }
    }

    private fun onCaptchaResult() {
        mallViewHandler?.setWebView(CommerceId.Gmarket.onlineId!!, webViewHtmlLoader.getWebView(), View.VISIBLE)

        webViewHtmlLoader.onPageCallback(rule.captchaCallbackJs,
                object : OnPageLoadedCallback {
                    override fun onPageLoaded(url: String) {

                        Timber.d("WEBTEST onPageLoaded $url")
                        Timber.d("WEBTEST rule.loginSuccessUrl $rule.loginSuccessUrl")

                        Timber.d("WEBTEST rule.userId $userId")
                        Timber.d("WEBTEST rule.userPwd $userPwd")


                        if (isSuccessUrl(url)) {
                            if (!WebLoader.isTest)
                            mallViewHandler?.setWebView(CommerceId.Gmarket.onlineId!!, webViewHtmlLoader.getWebView(), View.GONE)
                            mallViewHandler?.retry(CommerceId.Gmarket.onlineId!!, userId, userPwd)
                            webViewHtmlLoader.onPageCallback() // 콘솔 제거해줍니다.
                        }
                    }

                    override fun onUserIdChanged(id: String) {
                        userId = id
                    }

                    override fun onUserPwdChanged(pwd: String) {
                        userPwd = pwd
                    }
                })
    }

    private fun isSuccessUrl(url: String): Boolean {
        return url.contains(rule.loginSuccessUrl) || url.contains(rule.loginSuccessUrl2)

    }
    private fun loadJsoupHtmlParser(html: String): Result<JsonArray> {

        return try {

            val body = Jsoup.parse(html).body()
            val jsonArray = JsonArray()
            val items = body.select(rule.container)

            Timber.d("items" + items.size)

            var recentOrderDate = ""
            var recentOrderLink = ""
            items.forEach { order ->


                val jsonObject = JsonObject()


                var orderDate = order.select(rule.orderDate).text()
                if(orderDate.isNullOrBlank()) {
                    orderDate = recentOrderDate
                } else {
                    recentOrderDate = orderDate
                }


                val orderNum = order.select(rule.orderNum).text()
                val orderPrice = order.select(rule.orderPrice).text()
                val orderState = order.select(rule.orderState).text()

                var orderLink = order.select(rule.orderLink).firstOrNull()?.attr("href")


                if(orderLink.isNullOrBlank()) {
                    orderLink = recentOrderLink
                } else {
                    recentOrderLink = orderLink
                }

                Timber.d(rule.orderLink + "orderLink" + orderLink)


                val seller = order.select(rule.seller).text()

                val productName = order.select(rule.productName).text()
                val productOption = order.select(rule.productOption).text()
                val productLink = order.select(rule.productLink).firstOrNull()?.attr("href") ?: ""
                val productImgUrl = order.select(rule.productImgUrl).attr("src")
                val productPrice = order.select(rule.productPrice).text()
                val productQuantity = order.select(rule.productQuantity).text()

                jsonObject.addProperty("orderDate", orderDate)
                jsonObject.addProperty("orderNum", orderNum)
                jsonObject.addProperty("orderPrice", orderPrice)
                jsonObject.addProperty("orderState", orderState)
                jsonObject.addProperty("orderLink", orderLink)
                jsonObject.addProperty("seller", seller)


                jsonObject.addProperty("productName", productName)
                jsonObject.addProperty("productOption", productOption)
                jsonObject.addProperty("productLink", productLink)
                jsonObject.addProperty("productImgUrl", productImgUrl)
                jsonObject.addProperty("productPrice", productPrice)
                jsonObject.addProperty("productQuantity", productQuantity)

                jsonArray.add(jsonObject)
            }

            return Result.Success(jsonArray).also {
                Timber.d("parsing$jsonArray")
                Timber.d("parsing" + jsonArray.size())

            }
        } catch (e: Exception) {
            e.printStackTrace()
            Result.Error(e)
        }
    }

    private fun loadParsedDataFilter(jsonArray: JsonArray): List<OrderItem>  {
        return try {
            jsonArray.mapNotNull {

                Timber.d("loadParsedDataFilter$it")

                val jsonObject = it.asJsonObject

                val orderDate = jsonObject["orderDate"].asString.toDate() //2020.05.24
                val orderNum = orderNumFilter(jsonObject["orderNum"].asString)// "(2020-05-24-F009B5)"
                val orderPrice = jsonObject["orderPrice"].asString.toPrice()
                val orderState = jsonObject["orderState"].asString
                val orderLink = jsonObject["orderLink"].asString
                val seller = jsonObject["seller"].asString

                val productName = jsonObject["productName"].asString
                val productOption = jsonObject["productOption"].asString
                val productLink = jsonObject["productLink"].asString
                val productImgUrl = jsonObject["productImgUrl"].asString
                val productPrice = jsonObject["productPrice"].asString.toPrice()
                val productQuantity = jsonObject["productQuantity"].asString.toQuantity()
                val productId = productLink.substring(productLink.lastIndexOf( "=") + 1)
                if(orderDate != null) {
                    OrderItem(key = key,
                            title = productName,
                            productId = productId,
                            orderDate = orderDate,
                            unitPrice = productPrice,
                            orderNum = orderNum,
                            orderState = orderState,
                            orderPrice = orderPrice,
                            imgUrl = productImgUrl,
                            option = productOption,
                            quantity = productQuantity,
                            orderLink = orderLink,
                            productLink = productLink)
                } else {
                    null
                }
            }.also {
                Timber.d("parseResults $it")
            }
        } catch (e: java.lang.Exception) {
            Timber.d("parseResults $e")

            emptyList()
        }
    }

    private fun orderNumFilter(orderNum: String): String {
        if (!TextUtils.isEmpty(orderNum)) {
            val pattern = Pattern.compile("\\s*주문번호\\s*(\\S+)\\s*", Pattern.CASE_INSENSITIVE or Pattern.DOTALL)
            val matcher = pattern.matcher(orderNum)
            if (matcher.matches()) {
                return matcher.group(1)
            }
        }
        return orderNum
    }

    private fun String.toDate(): String? {
        // 2020-07-14 -> 2020-07-14 00:00:00
        val format =
                SimpleDateFormat(rule.dateFormat, Locale.KOREA)

        val baseFormat =
                SimpleDateFormat("yyyy-MM-dd", Locale.KOREA)

        val date = format.parse(this)

        return date?.let {
            baseFormat.format(it) + " 00:00:00"
        }
    }

    private fun String.toQuantity(): Int {
        //1개 / 7,8000
        return try {
            val index = this.indexOf("/")

            val quantityStr = this.substring(0, index).replace("개", "").trim()

            val result =  Integer.parseInt(quantityStr)
            if(result == 0) {
                return 1
            } else {
                result
            }
        } catch (e: Exception) {
            1
        }
    }

    private fun String.toPrice(): Double {
        //1개 / 7,8000
        return try {
            val index = this.indexOf("/")

            val quantityStr = this.substring(index + 1)
                    .replace(",", "")
                    .replace("원", "")
                    .trim()

            return Integer.parseInt(quantityStr).toDouble()

        } catch (e: Exception) {
            0.0
        }
    }


    private fun getLoginUrl(id: String, password: String): String = String.format(rule.loginJs, id, password)
}
