package com.tenqube.visual_scraper.webviewhtmlloader.util

import android.annotation.SuppressLint
import android.content.Context
import android.os.Handler
import android.webkit.*
import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import com.tenqube.visual_scraper.webviewhtmlloader.OnPageLoadedCallback
import com.tenqube.visual_scraper.webviewhtmlloader.model.AlertJS
import com.tenqube.visual_scraper.webviewhtmlloader.model.Response
import com.tenqube.visual_scraper.webviewhtmlloader.model.StatusCode
import com.tenqube.visual_scraper.webviewhtmlloader.model.WebViewCallback
import kotlinx.coroutines.*
import timber.log.Timber
import kotlin.coroutines.resume


const val USER_AGENT_NORMAL = "Mozilla/5.0 (Windows NT 10.0; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/57.0.2987.133 Safari/537.36"


public object WebLoader {

    var isTest: Boolean = false
    val _web = MutableLiveData<Event<WebViewCallback>>()
    val web: LiveData<Event<WebViewCallback>> = _web

    val _statusMsg = MutableLiveData<String>()
    val statusMsg: LiveData<String> = _statusMsg
}

fun WebView.show(from: String = "") {
    val mainHandler = Handler(context.mainLooper)
    mainHandler.post(Runnable {
        WebLoader._web.value = Event(WebViewCallback(this, true, from))
    })
}


fun WebView.hide(from: String = "") {
    val mainHandler = Handler(context.mainLooper)
    mainHandler.post(Runnable {
        WebLoader._web.value = Event(WebViewCallback(this, false, from))
    })
}

class WebViewManager(val context: Context,
                     var webView: WebView? = null) {

    var callbackUrl = ""
    var isInit = false
    var calledUrl = ""
    var onSuccessUrl: String? = null
    var isJs = false
    var alertJS: AlertJS? = null
    var htmlCallback: SuspendCallback<String>? = null

    var callback: SuspendCallback<Response>? = null
    var backCallback: Callback<String>? = null
    var onPageLoadedCallback: OnPageLoadedCallback? = null

    fun destroy() {
        val mainHandler = Handler(context.mainLooper)
        mainHandler.post(Runnable {
            webView?.destroy()
        })
        webView = null
        initialize()
    }

    fun initialize() {
        alertJS = null
        isInit = false
        calledUrl = ""
        onSuccessUrl = "" // 성공 경로
        isJs = false
        callback = null
        htmlCallback = null
    }

    fun setOnPageCallback(callbackJs: String, onPageLoadedCallback: OnPageLoadedCallback?) {
        webView?.loadUrl(callbackJs) // 콘솔로그로 세팅되는 정보를 얻고자함.
        this.onPageLoadedCallback = onPageLoadedCallback
    }

    private suspend fun createWebView() = withContext(Dispatchers.Main) {
        if(webView == null) {
            webView = WebView(context)
            if(WebLoader.isTest)
                webView?.show("WebViewManager isTest true") //test 시만
        }
        setupWebView()
    }

    private suspend fun loadUrl(url: String) = withContext(Dispatchers.Main) {
        Timber.d("WEBTEST loadUrl $url")
        if(!isInit) createWebView()

        calledUrl = url

        if(isCoupang(url)) {
            webView?.postDelayed({
                webView?.loadUrl(url)
            }, 100)
        } else {
            webView?.loadUrl(url)
        }

    }

    fun goBack() {
        Timber.d("WEBTEST goBack")
        webView?.goBack()
    }

    suspend fun getRequestUrl(): String? {

        return suspendCoroutineWithTimeout(15000) { cont ->
            backCallback = object :
                    Callback<String> {
                override fun onDataLoaded(value: String) {
                    Timber.d("WEBTEST getRequestUrl $value")

                    backCallback = null
                    if(cont.isActive) {
                        cont.resume(value)
                    }
                }
            }
        }
    }

    private fun getScrappingUrl() =
        "javascript:window.Android.getHtml(document.getElementsByTagName('html')[0].innerHTML);"

    suspend fun call(url: String, isJs: Boolean = false, hasHtml: Boolean = false, onSuccessUrl: String? = null, userTimeOut: Long? = null): Response {

        this.isJs = isJs
        this.onSuccessUrl = onSuccessUrl
        val timeOut = userTimeOut
                ?: when(isJs) {
                    true -> JS_TIME
                    false -> LOADING_TIME
                }

         // 콜하고 응답없으면 html 조사 하기 캡챠 같은거일수있음
        return callUrlTimeOut(url, hasHtml, timeOut).run {

            when(this == null) {
                true -> { // time out 인경우
                    Timber.d("callUrlTimeOut timeout $url")
                    Response(if (isJs) {
                        StatusCode.TimeOut
                    } else {
                        StatusCode.Success
                    }, url = url, html = getHtmlOrNull(), alertJS = alertJS, callbackUrl = callbackUrl)
                }
                false -> {
                    this.copy(alertJS = alertJS)
                }
            }.also {
                Timber.d("callUrlTimeOut response $it")
            }
        }
    }

    private suspend inline fun <T> suspendCoroutineWithTimeout(timeout: Long, crossinline block: (CancellableContinuation<T>) -> Unit) : T? {
        var finalValue : T? = null
        withTimeoutOrNull(timeout) {
            finalValue = suspendCancellableCoroutine(block = block)
        }
        return finalValue
    }


    private suspend fun callUrlTimeOut(url: String, hasHtml: Boolean = false, timeOut: Long): Response? {

        return suspendCoroutineWithTimeout(
                timeOut
        ) { cont ->
            Timber.d("callUrlTimeOut url $url hasBody $hasHtml")

            loadUrl(url, callback = object :
                    SuspendCallback<Response> {
                override suspend fun onDataLoaded(value: Response) {
                    callback = null
                    val result = value.copy(alertJS = alertJS)

                    Timber.d("onDataLoaded result $result")

                    if (cont.isActive) {
                        if (hasHtml) { // html 값 파싱 해야할지 여부
                            val html = getHtmlOrNull()

                            cont.resume(result.copy(url = url, html = html))
                        } else {
                            cont.resume(result.copy(url = url))
                        }
                    }

                }
            })
        }
    }

    suspend fun getHtmlOrNull(): String? {
        delay(HTML_DELAY_TIME) // onPageLoaded 호출후 통신을 통해 추가적으로 데이터가 바인드 되는 시간 대기가 필요합니다.
        return withTimeoutOrNull(1000) {
            getHtml()
        }
    }

    private suspend fun getHtml(): String = suspendCancellableCoroutine { cont ->

        getHtml(callback = object :
                SuspendCallback<String> {
            override suspend fun onDataLoaded(html: String) {
                if (cont.isActive)
                    cont.resume(html)
            }
        })
    }


    private fun getHtml(callback: SuspendCallback<String>) {
        htmlCallback = callback
        GlobalScope.launch {
            loadUrl(getScrappingUrl())
        }
    }

    private fun loadUrl(url: String, callback: SuspendCallback<Response>) {
        this.callback = callback

        GlobalScope.launch {
            loadUrl(url)
        }
    }

    @SuppressLint("SetJavaScriptEnabled")
    private suspend fun setupWebView() = withContext(Dispatchers.Main) {
        webView?.apply {
            addJavascriptInterface(CustomBridge(
                    object : SuspendCallback<String> {
                        override suspend fun onDataLoaded(value: String) {
                            htmlCallback?.onDataLoaded(value)
                        }

                    }), "Android")

           webViewClient = object : WebViewClient() {

                override fun onPageFinished(view: WebView?, url: String?) {
                    super.onPageFinished(view, url)

                    callbackUrl = url ?: ""
                    Timber.d("WEBTEST onPageFinished $url")

                    Timber.d("onPageFinished url $url")
                    Timber.d("onPageFinished onPageLoadedCallback $onPageLoadedCallback")

                    url?.let { it ->
                        onPageLoadedCallback?.onPageLoaded(it)
                    }
/*
                    Login/Login == skip <- 콜백을함.
                    mobin/gmarket == onSUcces <- 호출대기함
*/
                    GlobalScope.launch {
                        // 로그인일때 타임아웃을 통해 특정페이지를 기다리는 로직을 추가합니다.

                        Timber.d("onPageFinished onSuccessUrl $onSuccessUrl")

                        onSuccessUrl?.let {
                            if(isJs) {
                                if(url?.contains(it) == true) {
                                    sendCallback(StatusCode.Success, "", "", url)
                                }
                            } else {
                                sendCallback(StatusCode.Success, "", "", url ?: "")
                            }
                        } ?: sendCallback(StatusCode.Success, "", "", url ?: "")

                    }
                }

               override fun onReceivedError(
                       view: WebView?,
                       errorCode: Int,
                       description: String?,
                       failingUrl: String?
               ) {
                   Timber.d("onReceivedError url $url description $description")

                   GlobalScope.launch {
                       sendCallback(StatusCode.Error, description, "", failingUrl ?: "")
                   }

                   super.onReceivedError(view, errorCode, description, failingUrl)
               }

               override fun shouldInterceptRequest(view: WebView?, request: WebResourceRequest?): WebResourceResponse? {

                   backCallback?.let {
                       Timber.d("WEBTEST shouldInterceptRequest ${request?.url.toString()}")

                       if (request != null && isCoupang(request.url.toString())) {
                           Timber.i("${request.url}")

                           it.onDataLoaded(request.url.toString())
                       }
                   }

                   return super.shouldInterceptRequest(view, request)
               }

           }

            webChromeClient = object : WebChromeClient() {

                override fun onJsAlert(
                        view: WebView?,
                        url: String?,
                        message: String?,
                        result: JsResult?
                ): Boolean {

                    Timber.d("WEBTEST onJsAlert url $url message $message")
                    result?.confirm()
                    alertJS = AlertJS(url, message)
                    return true
                }

                override fun onConsoleMessage(cm: ConsoleMessage?): Boolean {
                    //js 실행했을경우에만 체크합니다.

                    val message = cm?.message() ?: ""
                    Timber.d("WEBTEST onConsoleMessage url ${message}")

                    if(onPageLoadedCallback != null) {


                        Timber.d("WEBTEST onConsoleMessage message.indexOf(\"id:\") ${message.indexOf("id:")}")

                        Timber.d("WEBTEST onConsoleMessage message.indexOf(\"pw::\") ${message.indexOf("pw:")}")

                        if(message.indexOf("id:") == 0) {
                            onPageLoadedCallback?.onUserIdChanged(message.replace("id:", ""))
                        } else if(message.indexOf("pw:") == 0) {
                            onPageLoadedCallback?.onUserPwdChanged(message.replace("pw:", ""))
                        }
                    }
                    GlobalScope.launch {
                        Timber.d("onConsoleMessage url ${cm?.message()}")

                        if(isJs && cm?.message()?.contains("Uncaught") == true) {
                            // js 실행후 에러 콜백 떨어짐 원하는 페이지가 로드 안된경우 실제 없는경우
//                            sendCallback(StatusCode.Error, cm.message(), "")
                        }
                    }

                    return super.onConsoleMessage(cm)
                }
            }

            settings.run {
                javaScriptEnabled = true
                userAgentString = USER_AGENT_NORMAL
//                cacheMode =  WebSettings.LOAD_NO_CACHE
//                    setAppCacheEnabled(false)
            }

            isInit = true
        }

    }

    private fun isCoupang(url: String) : Boolean {
        return url.contains("https://mc.coupang.com/ssr/_next/data/")
    }
    private suspend fun sendCallback(statusCode: StatusCode, msg: String?, html: String, callbackUrl: String) {
        Timber.d("WEBTEST sendCallback $callback")


        callback?.onDataLoaded(
                Response(
                        statusCode = statusCode,
                        callbackUrl = callbackUrl,
                        msg = msg,
                        html = html,
                        url = "",
                        alertJS = alertJS
                )
        )
        callback = null
    }

    companion object {
        const val HTML_DELAY_TIME = 3000L // 웹페이지 로드후 html 정보 가져 오기전 대기시간.

        const val LOADING_TIME = 8000L // 웹페이지 로드후 html 정보 가져 오기전 대기시간.

        const val JS_TIME = 8000L // 웹페이지 로드후 html 정보 가져 오기전 대기시간.

    }

}