package com.tenqube.visual_scraper.thirdparty.market

import android.content.Context
import android.webkit.WebView
import com.tenqube.visual_scraper.ScrapService
import com.tenqube.visual_scraper.constants.Constants
import com.tenqube.visual_scraper.db.entity.OrderEntity
import com.tenqube.visual_scraper.db.entity.UserEntity
import com.tenqube.visual_scraper.market.MarketFactory
import com.tenqube.visual_scraper.market.MarketKey
import com.tenqube.visual_scraper.market.dto.MarketResult
import com.tenqube.visual_scraper.market.dto.ScrapingCode
import com.tenqube.visual_scraper.market.util.toDate
import com.tenqube.visual_scraper.webviewhtmlloader.WebViewHtmlLoader
import com.tenqube.visual_scraper.webviewhtmlloader.util.WebViewManager
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.GlobalScope
import kotlinx.coroutines.launch

class MarketScrapperImpl(val context: Context) : MarketScrapper {

    override fun startWebViewLogin(user: UserEntity, onResult: ScrapService.OnResult<List<OrderEntity>>) {

        GlobalScope.launch(Dispatchers.Default) {

            val key = getKey(user.mallId)
            if(key == null) {
                onResult.onFail(user.mallId, Constants.ERROR.PARSING_FAIL, "not exist mId")
                return@launch
            }

            val webViewManager = WebViewHtmlLoader(WebViewManager(context))
            val market = MarketFactory.createScrapper(key, webViewManager, context)
            if(market == null) {
                onResult.onFail(user.mallId, Constants.ERROR.PARSING_FAIL, "not exist market")
                return@launch
            } else {
                val result = market.load(user.userId, user.userPwd.value)

                if(result.code == ScrapingCode.Success) {
                    val results = result.results?.map {
                        OrderEntity(user.mallId, user.id, it.title, it.orderDate.toDate(), it.orderDate.toDate(),
                                it.option,
                                it.quantity,
                                it.unitPrice,
                                it.orderNum,
                                it.orderState,
                                it.imgUrl)
                    } ?: emptyList()

                    onResult.onSuccess(user.mallId, results, "success")
                } else {
                    getError(user.mallId, result, onResult, webViewManager.getWebView())
                }
            }
        }
    }

    override fun logout(mId: Int) {
        GlobalScope.launch(Dispatchers.Default) {

            val key = getKey(mId)
            if(key != null) {
                val webViewManager = WebViewHtmlLoader(WebViewManager(context))
                val market = MarketFactory.createScrapper(key, webViewManager, context)
                if(market != null) {
                    val result = market.logOut()
                }
            }
        }
    }

    private fun getError(mId: Int, result: MarketResult, onResult: ScrapService.OnResult<List<OrderEntity>>, webView: WebView?) {

       when(result.code) {
            ScrapingCode.CaptchaError -> {
                webView?.let {
                    onResult.onCaptcha(mId, webView)
                } ?: onResult.onFail(mId, Constants.ERROR.LOGIN_FAIL, result.msg)

            }
            ScrapingCode.IdOrPwdError -> {
                onResult.onFail(mId, Constants.ERROR.LOGIN_FAIL, result.msg)
            }
           ScrapingCode.NetworkError -> {
               onResult.onFail(mId, Constants.ERROR.NETWORK, result.msg)
           }
           else -> onResult.onFail(mId, Constants.ERROR.NONE, result.msg)
       }
    }

    private fun getKey(mId: Int): MarketKey? {

        return when (mId) {
            Constants.MALL.SSG.mallId -> {
                MarketKey.SSG
            }
            Constants.MALL.MarketKurly.mallId -> {
                MarketKey.MarketKurly
            }
            Constants.MALL.Coupang.mallId -> {
                MarketKey.COUPANG
            }
            Constants.MALL.Gmarket.mallId -> {
                MarketKey.Gmarket
            }

            else -> {
                null
            }
        }

    }
}