package com.tenqube.visual_scraper.market

import android.content.Context
import com.google.gson.Gson
import com.google.gson.JsonArray
import com.google.gson.JsonObject
import com.tenqube.visual_scraper.market.dto.MarketResult
import com.tenqube.visual_scraper.market.dto.OrderItem
import com.tenqube.visual_scraper.market.dto.SSGRule
import com.tenqube.visual_scraper.market.dto.ScrapingCode
import com.tenqube.visual_scraper.market.util.toPrice
import com.tenqube.visual_scraper.shared.util.Result
import com.tenqube.visual_scraper.shared.util.Utils
import com.tenqube.visual_scraper.shared.util.getValue
import com.tenqube.visual_scraper.webviewhtmlloader.WebViewLoader
import com.tenqube.visual_scraper.webviewhtmlloader.model.StatusCode
import com.tenqube.visual_scraper.webviewhtmlloader.model.WebViewResponse
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import org.jsoup.Jsoup
import timber.log.Timber
import java.text.SimpleDateFormat
import java.util.*

class SSG(
        private val webViewHtmlLoader: WebViewLoader, override val key: MarketKey,
        override val context: Context
) : Market {

    private var rule: SSGRule

    init {
        val marketRule = Utils.getJsonDataFromAsset(context, "ssg_rules.json")?:  ""
        rule = Gson().fromJson(marketRule, SSGRule::class.java)
    }
    // 서버에서 받아올 값

    private var scrapingResult = MarketResult(ScrapingCode.UnknownError)

    // 콜백 방식으로 데이터 리턴
    override suspend fun load(
        id: String,
        password: String
    ): MarketResult = withContext(Dispatchers.Main) {

        scrapingResult = MarketResult(ScrapingCode.UnknownError)

        try {

            scrapingResult = login(id, password).run {
                order()
            }.run {
                parsing(this)
            }.run {
                filter(this)
            }.run {
                toResult(this)
            }

        } catch (e: Exception) {
            e.toString()
        }
        return@withContext scrapingResult.also {
            Timber.d("load 응답값: $it")
        }
    }

    override suspend fun signIn(id: String, password: String): MarketResult = withContext(Dispatchers.Main) {

        scrapingResult = MarketResult(ScrapingCode.UnknownError)

        try {
            login(id, password)
        } catch (e: Exception) {
            e.toString()
        }
        return@withContext scrapingResult.also {
            Timber.d("signIn 응답값: $it")
        }
    }

    override suspend fun scrapOrder(): MarketResult = withContext(Dispatchers.Main) {

        scrapingResult = MarketResult(ScrapingCode.UnknownError)

        try {

            scrapingResult = order().run {
                parsing(this)
            }.run {
                filter(this)
            }.run {
                toResult(this)
            }

        } catch (e: Exception) {
            e.toString()
        }
        return@withContext scrapingResult.also {
            Timber.d("load 응답값: $it")
        }
    }

    override suspend fun logOut() {
        val loginResult = webViewHtmlLoader
                .go(rule.baseUrl)
                .go(rule.logout, true)
                .load()
                .getValue()
    }

    private suspend fun login(id: String, password: String) {
        // html load
        val loginResult = webViewHtmlLoader
            .noCache()
            .go(rule.loginUrl)
            .go(getLoginUrl(id, password), true)
            .load()
            .getValue()

        Timber.i("login result: $loginResult")

        val result = checkLoginHtml(loginResult)
        Timber.i("login result2 : $result")
        when(result.code == ScrapingCode.Success) {
            true -> {
                scrapingResult = MarketResult(ScrapingCode.Success)
            }
            false -> {
                scrapingResult = MarketResult(result.code, result.msg)
                throw Exception()
            }
        }
    }

    private suspend fun order(): String {
        val orderResult = webViewHtmlLoader
            .go(rule.orderUrl, true)
            .load()
            .getValue()
            .also {
                Timber.d("order ${it.isSuccess}")
            }
        when(orderResult.isSuccess) {
            true -> {
                when (orderResult.html == null) {

                    true -> {
                        scrapingResult = MarketResult(ScrapingCode.UnknownError, "html is empty")
                        throw Exception()
                    }
                    false -> {
                        return orderResult.html!!
                    }
                }
            }
            false -> {
                scrapingResult = MarketResult(ScrapingCode.NetworkError)
                throw Exception()
            }
        }
    }

    private fun parsing(html: String): JsonArray {
        return loadJsoupHtmlParser(html).getValue()
    }

    private fun filter(jsonArray: JsonArray): List<OrderItem> {
        return loadParsedDataFilter(jsonArray)
    }

    private fun toResult(orderItems: List<OrderItem>): MarketResult {
        return MarketResult(
            ScrapingCode.Success, "success",
            results = orderItems).also {
            Timber.d("results ${it.results}")
        }
    }

    private fun checkLoginHtml(response: WebViewResponse): MarketResult {
        Timber.d("checkLoginHtml rule ${response.isSuccess}")

        return when (response.isSuccess) {

            true -> {
                response.alertJS?.let {

                    Timber.d("checkLoginHtml alertJS ${response.alertJS}")

                    Timber.d("checkLoginHtml alertJS ${rule.loginErrorUrl}")
                    Timber.d("checkLoginHtml alertJS ${it.url}")

                    Timber.d("checkLoginHtml alertJS ${rule.loginErrorUrl == it.url}")
                    if(it.msg?.contains("아이디/비밀번호") == true) {
                        return MarketResult(ScrapingCode.IdOrPwdError, msg = it.msg ?: "")
                    }
                }

                MarketResult(
                    code = ScrapingCode.Success,
                    msg = "",
                    results = emptyList()
                )
            }
            false -> {

                response.html?.let {

                    Timber.d("checkLoginHtml rule ${rule.captcha}")
                    Timber.d("checkLoginHtml ${it.contains(rule.captcha)}")

                    if(it.contains(rule.captcha)) {
                        return MarketResult(ScrapingCode.CaptchaError)
                    }
                }

                response.alertJS?.let {
                    if(it.msg?.contains("아이디/비밀번호") == true) {
                        return MarketResult(ScrapingCode.IdOrPwdError, msg = it.msg ?: "")
                    }
//                    if (rule.loginErrorUrl == it.url) {
//                    }
                }

                MarketResult(
                    code = if(response.code == StatusCode.TimeOut)
                        ScrapingCode.Success
                    else
                        ScrapingCode.NetworkError,
                    msg = response.msg ?: "",
                    results = emptyList()
                )
            }
        }
    }

    private fun loadJsoupHtmlParser(html: String): Result<JsonArray> {

        return try {

            val body = Jsoup.parse(html).body()
            val jsonArray = JsonArray()
            val items = body.select(rule.container)

            Timber.d("items" + items.size)

            items.forEach { order ->

                val orderDate = order.selectFirst(rule.orderDate).text()
                val orderNum = order.selectFirst(rule.orderNum).text()
                val orderPrice = order.selectFirst(rule.orderPrice).text()

                val orderItems = order.select(rule.orderItemContainer)
                Timber.d("orderItems" + orderItems.size)

                orderItems.forEach { orderItem ->
                    Timber.d("orderItem" + orderItem)

                    val deliveryType = orderItem.selectFirst(rule.deliveryType).text()
                    val orderState = orderItem.selectFirst(rule.orderState).text()

//                {"orderDate":"2020.05.24","orderNum":"(2020-05-24-F009B5)","orderPrice":"0","deliveryType":"쓱배송","orderState":"반품완료","productLink":"","productOption":"신선보장","productName":"친환경 채소믹스 500g","productPrice":"5,045 5,900","productImgUrl":"https://sitem.ssgcdn.com/23/18/84/item/2097000841823_i1_140.jpg","productQuantity":"상품수량1개"},{"orderDate":"2020.05.24","orderNum":"(2020-05-24-F009B5)","orderPrice":"0 8,360 9,975 4,731 5,206 10,431 2,645 2,736 5,045 8,360 8,800 9,975 10,500 4,731 4,980 5,206 5,480 10,431 10,980 2,645 3,480 2,736 2,880 5,045 5,900","deliveryType":"쓱배송","orderState":"반품완료","productLink":"","productOption":"신선보장","productName":"친환경 채소믹스 500g","productPrice":"5,045 5,900","productImgUrl":"https://sitem.ssgcdn.com/23/18/84/item/2097000841823_i1_140.jpg","productQuantity":"상품수량1개"},{"orderDate":"2020.05.24","orderNum":"(2020-05-24-F009B5)","orderPrice":"0 8,360 9,975 4,731 5,206 10,431 2,645 2,736 5,045 8,360 8,800 9,975 10,500 4,731 4,980 5,206 5,480 10,431 10,980 2,645 3,480 2,736 2,880 5,045 5,900","deliveryType":"쓱배송","orderState":"반품완료","productLink":"","productOption":"신선보장","productName":"친환경 채소믹스 500g","productPrice":"5,045 5,900","productImgUrl":"https://sitem.ssgcdn.com/23/18/84/item/2097000841823_i1_140.jpg","productQuantity":"상품수량1개"},{"orderDate":"2020.05.24","orderNum":"(2020-05-24-F009B5)","orderPrice":"0 8,360 9,975 4,731 5,206 10,431 2,645 2,736 5,045 8,360 8,800 9,975 10,500 4,731 4,980 5,206 5,480 10,431 10,980 2,645 3,480 2,736 2,880 5,045 5,900","deliveryType":"쓱배송","orderState":"반품완료","productLink":"","productOption":"신선보장","productName":"친환경 채소믹스 500g","productPrice":"5,045 5,900","productImgUrl":"https://sitem.ssgcdn.com/23/18/84/item/2097000841823_i1_140.jpg","productQuantity":"상품수량1개"},{"orderDate":"2020.05.24","orderNum":"(2020-05-24-F009B5)","orderPrice":"0 8,360 9,975 4,731 5,206 10,431 2,645 2,736 5,045 8,360 8,800 9,975 10,500 4,731 4,980 5,206 5,480 10,431 10,980 2,645 3,480 2,736 2,880 5,045 5,900","deliveryType":"쓱배송","orderState":"반품완료","productLink":"","productOption":"신선보장","productName":"친환경 채소믹스 500g","productPrice":"5,045 5,900","productImgUrl":"https://sitem.ssgcdn.com/23/18/84/item/2097000841823_i1_140.jpg","productQuantity":"상품수량1개"},{"orderDate":"2020.05.24","orderNum":"(2020-05-24-F009B5)","orderPrice":"0 8,360 9,975 4,731 5,206 10,431 2,645 2,736 5,045 8,360 8,800 9,975 10,500 4,731 4,980 5,206 5,480 10,431 10,980 2,645 3,480 2,736 2,880 5,045 5,900","deliveryType":"쓱배송","orderState":"반품완료","productLink":"","productOption":"신선보장","productName":"친환경 채소믹스 500g","productPrice":"5,045 5,900","productImgUrl":"https://sitem.ssgcdn.com/23/18/84/item/2097000841823_i1_140.jpg","productQuantity":"상품수량1개"},{"orderDate":"2020.05.24","orderNum":"(2020-05-24-F009B5)","orderPrice":"0 8,360 9,975 4,731 5,206 10,431 2,645 2,736 5,045 8,360 8,800 9,975 10,500 4,731 4,980 5,206 5,480 10,431 10,980 2,645 3,480 2,736 2,880 5,045 5,900","deliveryType":"쓱배송","orderState":"반품완료","productLink":"","productOption":"신선보장","productName":"친환경 채소믹스 500g","productPrice":"5,045 5,900","productImgUrl":"https://sitem.ssgcdn.com/23/18/84/item/2097000841823_i1_140.jpg","productQuantity":"상품수량1개"},{"orderDate":"2020.05.24","orderNum":"(2020-05-24-F009B5)","orderPrice":"0 8,360 9,975 4,731 5,206 10,431 2,645 2,736 5,045 8,360 8,800 9,975 10,500 4,731 4,980 5,206 5,480 10,431 10,980 2,645 3,480 2,736 2,880 5,045 5,900","deliveryType":"쓱배송","orderState":"반품완료","productLink":"","productOption":"신선보장","productName":"친환경 채소믹스 500g","productPrice":"5,045 5,900","productImgUrl"



                    val productContainer = orderItem.select(rule.productContainer)
                    Timber.d("orderState" + orderState)

                    Timber.d("productContainer" + productContainer.size)

                    productContainer.forEach { product ->
                        Timber.d("product$product")


                        val jsonObject = JsonObject()

                        jsonObject.addProperty("orderDate", orderDate)
                        jsonObject.addProperty("orderNum", orderNum)
                        jsonObject.addProperty("orderPrice", orderPrice)
                        jsonObject.addProperty("deliveryType", deliveryType)
                        jsonObject.addProperty("orderState", orderState)

                        val productLink = product.select(rule.productLink).first().attr("href")
                        val productOption = product.select(rule.productOption).text()
                        val productName = product.select(rule.productName).text()
                        val productPrice = product.select(rule.productPrice).text()
                        val productQuantity = product.select(rule.productQuantity).text()
                        val productImgUrl = product.select(rule.productImgUrl).attr("src")


                        jsonObject.addProperty("productLink", productLink)
                        jsonObject.addProperty("productOption", productOption)
                        jsonObject.addProperty("productName", productName)
                        jsonObject.addProperty("productPrice", productPrice)
                        jsonObject.addProperty("productImgUrl", productImgUrl)
                        jsonObject.addProperty("productQuantity", productQuantity)


                        jsonArray.add(jsonObject)
                    }
                }

            }

            return Result.Success(jsonArray).also {
                Timber.d("parsing$jsonArray")
                Timber.d("parsing" + jsonArray.size())

            }
        } catch (e: Exception) {
            Result.Error(e)
        }
    }

    private fun loadParsedDataFilter(jsonArray: JsonArray): List<OrderItem>  {
        return try {
            jsonArray.mapNotNull {

                Timber.d("loadParsedDataFilter$it")

                val jsonObject = it.asJsonObject

                val productName = jsonObject["productName"].asString //친환경 채소믹스 500g
                val orderDate = jsonObject["orderDate"].asString.toDate() //2020.05.24
                val productPrice = jsonObject["productPrice"].asString.toPrice()// 5,045
                val orderPrice = jsonObject["orderPrice"].asString.toPrice()
                val orderNum = jsonObject["orderNum"].asString.toOrderNum()// "(2020-05-24-F009B5)"
                val orderState = (jsonObject["orderState"].asString.split(" ").firstOrNull() ?: "").toOrderState()
                val productImgUrl = jsonObject["productImgUrl"].asString //https://sitem.ssgcdn.com/23/18/84/item/2097000841823_i1_140.jpg
//                val orderOption = jsonObject["deliveryType"].asString //쓱배송
                val productOption = jsonObject["productOption"].asString//신선보장
                val productQuantity = jsonObject["productQuantity"].asString.toQuantity() //상품수량1개
                val productLink = jsonObject["productLink"].asString

                if(orderDate != null) {
                    OrderItem(key = key,
                        title = productName,
                        orderDate = orderDate,
                        price = productPrice,
                        orderNum = orderNum,
                        orderState = orderState,
                        orderPrice = orderPrice,
                        imgUrl = productImgUrl,
                        option = productOption,
                        quantity = productQuantity,
                        orderLink = "",
                        productLink = productLink)
                } else {
                    null
                }
            }.also {
                Timber.d("parseResults $it")
            }
        } catch (e: java.lang.Exception) {
            Timber.d("parseResults $e")

            emptyList()
        }
    }

    private fun String.toDate(): String? {
        // 2020.07.14 -> 2020-07-14 00:00:00
        val format =
            SimpleDateFormat(rule.dateFormat, Locale.KOREA)

        val baseFormat =
            SimpleDateFormat("yyyy-MM-dd", Locale.KOREA)

        val date = format.parse(this)

        return date?.let {
            baseFormat.format(it) + " 00:00:00"
        }
    }

    private fun String.toOrderNum(): String {
//        "(2020-05-24-F009B5)" -> 20200524F009B5
        return this.replace("(", "")
            .replace(")", "")
            .replace("-", "")
    }

    private fun String.toQuantity(): Int {
        //상품수량1개 -> 1
        return try {
            Integer.parseInt(this.replace("상품수량", "")
                .replace("개", ""))
        } catch (e: Exception) {
            1
        }
    }

    private fun String.toOrderState(): String{

        when {
            this.contains("배송완료") -> {
                return "배송완료"
            }
            this.contains("결제완료") -> {
                return "결제완료"
            }
            this.contains("상품준비중") -> {
                return "상품준비중"
            }
            this.contains("출발대기") -> {
                return "출발대기"
            }
            this.contains("점포출발") -> {
                return "점포출발"
            }
            this.contains("배송중") -> {
                return "배송중"
            }
            this.contains("주문취소완료") -> {
                return "주문취소완료"
            }
            else -> {
                return ""
            }
        }
    }

    private fun getLoginUrl(id: String, password: String): String = rule.loginJs.replace("\$id", id).replace("\$password", password)
}
