package com.tenqube.visual_scraper;

import android.content.Context;
import android.content.SharedPreferences;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.os.Message;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import android.telecom.Call;
import android.text.TextUtils;
import android.util.SparseArray;
import android.webkit.WebView;

import com.tenqube.visual_scraper.constants.Constants.MALL;
import com.tenqube.visual_scraper.db.entity.UserEntity;
import com.tenqube.visual_scraper.manager.WebViewManager;
import com.tenqube.visual_scraper.model.LoadingState;
import com.tenqube.visual_scraper.model.query.MallWithUser;
import com.tenqube.visual_scraper.model.query.Password;
import com.tenqube.visual_scraper.model.view.MallInfo;
import com.tenqube.visual_scraper.model.view.OrderInfo;
import com.tenqube.visual_scraper.model.view.UserMall;
import com.tenqube.visual_scraper.model.view.ViewMoreInfo;
import com.tenqube.visual_scraper.repository.ScrapRepository;
import com.tenqube.visual_scraper.thirdparty.market.MarketScrapper;
import com.tenqube.visual_scraper.thirdparty.market.MarketScrapperImpl;
import com.tenqube.visual_scraper.thirdparty.mart.MartScrapper;
import com.tenqube.visual_scraper.thirdparty.mart.MartScrapperImpl;
import com.tenqube.visual_scraper.utils.DateUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import timber.log.Timber;

import static com.tenqube.visual_scraper.ScrapManager.HANDLER_MSG;
import static com.tenqube.visual_scraper.ScrapManager.HANDLER_MSG_ERROR;
import static com.tenqube.visual_scraper.ScrapManager.HANDLER_MSG_MALL_ID;
import static com.tenqube.visual_scraper.constants.Constants.*;
import static com.tenqube.visual_scraper.constants.Constants.MALL.*;

public class ScrapServiceImpl implements ScrapService {

    public Context context;

    private ScrapRepository repository;
    private SharedPreferences preferences;

    private WebViewManager mWebViewManager;

    private DataInteraction dataInteraction;
    public static Locale locale;

    private ScrapManager scrapManager;

    private MartScrapper martScrapper;

    private MarketScrapper marketScrapper;

    private SparseArray<LoadingState> progressStateArray = new SparseArray<>();

    private Handler handler;

    public ScrapServiceImpl(@NonNull Context context,
                            @NonNull String apiKey,
                            @NonNull String qualifier, /* @Nullable MallApiService api,*/
                            @NonNull DataInteraction dataInteraction) throws ParameterException {

        if(context == null) throw new ParameterException("context is null");
        if(TextUtils.isEmpty(apiKey)) throw new ParameterException("apiKey is empty");
        if(TextUtils.isEmpty(qualifier)) throw new ParameterException("qualifier is empty");
        if(dataInteraction == null) throw new ParameterException("dataInteraction is empty");

        this.context = context;
        this.dataInteraction = dataInteraction;

        initLocale();

        preferences = Injection.provideSharedPreference(context);
        preferences.edit().putString(PREF_KEY.LAYER, qualifier).apply();

        repository = Injection.provideScrapRepository(context.getApplicationContext(), preferences);
        repository.saveKey(apiKey);

        handler = createHandler();

        scrapManager = ScrapManager.getInstance(context, repository, handler);

        martScrapper = MartScrapperImpl.getInstance(context);

        marketScrapper = new MarketScrapperImpl(context);

        if (qualifier.equals(LAYER.DEV)) {
            Timber.plant(new Timber.DebugTree());
        }

        singUp();

        syncRule(null);

    }

    /**
     *  사용자 등록
     */
    private void singUp() {
        if (!preferences.getString(PREF_KEY.UUID, "").equals(dataInteraction.getUid())) {

            dataInteraction.getAdId(adId -> {

                String appName = dataInteraction.getServiceAppName();
                if(TextUtils.isEmpty(appName))return;

                String uid = dataInteraction.getUid();
                if(TextUtils.isEmpty(uid))return;

                repository.signUpUser(appName, uid, adId, items -> {

                    Timber.i("appName :%s", appName);
                    Timber.i("uid: %s", uid);
                    Timber.i("adId: %s", adId);
                });
            });
        }
    }

    private void initLocale(){
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N){
            locale = context.getResources().getConfiguration().getLocales().get(0);
        } else{
            //noinspection deprecation
            locale =  context.getResources().getConfiguration().locale;
        }
    }

    @Override
    public void initialize() {
        repository.initialize();
    }

    @Override
    public void destroy() {
        progressStateArray.clear();
    }

    @Override
    public WebView getWebView() {
        return mWebViewManager.getWebView();
    }

    @Override
    public void requestMobileAuthCode(String phoneNumber, String birth, String name, OnResultCallback<Boolean> onResult) throws Exception {
            martScrapper.init(items -> {

                try {
                    martScrapper.requestMobileAuthCode(phoneNumber, birth, name, success -> onResult.onDataLoaded(success));
                } catch (Exception e) {
                    e.printStackTrace();
                    onResult.onDataLoaded(false);

                }
            });
    }

    @Override
    public void getUserMalls(onResultInterface<List<UserMall>> onResult) throws Exception {
        try {

            repository.getMallWithActiveUsers(items -> {

                ArrayList<UserMall> users = new ArrayList<>();
                for(MallWithUser mallUser : items) {

                    MallInfo mall = new MallInfo(mallUser.mall.getId(), mallUser.mall.getDisplayName(), mallUser.mall.getIconUrl());
                    users.add(new UserMall(mall,
                            mallUser.user.getUserId(), mallUser.user.getUserPwd().getValue()));
                }
                onResult.onDataLoaded(users);

            });

        } catch (Exception e) {
            onResult.onFail();
        }
    }

    @Override
    public void startHomeplusLogin(String authCode, OnResultCallback<Boolean> onResult) throws Exception {
        try {
            martScrapper.startHomeplusLogin(authCode, onResult::onDataLoaded);
        } catch (Exception e) {
            e.printStackTrace();
            onResult.onDataLoaded(false);
        }
    }

    @Override
    public void startLoginAndParsing(String userId, String userPwd, int mId, MallViewHandler mallViewService, OnResultCallback<Integer> onResult) throws ParameterException {
        Timber.i("startLoginAndParsing = " + userId+"/"+mId + "/" + userPwd);
        if (TextUtils.isEmpty(userId) || TextUtils.isEmpty(userPwd) || mId == 0) throw new ParameterException("empty parameter");

        progressStateArray.put(mId, new LoadingState(true));

        // 몰 활성화 여부 조회
        repository.checkIsActiveMall(mId, (mall) -> {
            Timber.i("checkIsActiveMall : %s %s", mId, mall);
            // mall Active 하지 않거나, 해당 몰정보 없음
            if (mall == null) {
                onResult.onFail(mId, ERROR.EMPTY_MALL, "empty mall");
                return;
            }

            Timber.i("checkIsActiveMall : !!");
            scrapManager.setCallback(new OnResultCallback<Integer>() {
                @Override
                public void onDataLoaded(Integer item) {
                    progressStateArray.put(mId, new LoadingState(false));
                    onResult.onDataLoaded(item);
                }

                @Override
                public void onFail(int mId, ERROR error, String msg) {
                    progressStateArray.put(mId, new LoadingState(false));
                    onResult.onFail(mId, error, msg);

                }
            });
            // 유저 (& 몰정보) 가져오기

            repository.getMallWithUser(mId, userId, new onResultInterface<MallWithUser>() {
                @Override
                public void onDataLoaded(MallWithUser item) {

                    // 유저 정보 존재 -> 로그인 성공시 비번 다를경우, 업데이트
                    // login
                    Timber.i("item : %s", item);

                    MallWithUser mallWithUser = new MallWithUser(item.mall, new UserEntity(mId, userId, new Password(userPwd)));
                    startScrap(mId, mallWithUser, mallViewService, martScrapper, marketScrapper, false);

                }

                @Override
                public void onFail() {
                    // 유저 및 몰정보 없음 -> 해당 mall 의 신규 user id 등록
                    Timber.i("mId : %s", mId);
                    MallWithUser mallWithUser = new MallWithUser(mall, new UserEntity(mId, userId, new Password(userPwd)));
                    startScrap(mId, mallWithUser, mallViewService, martScrapper, marketScrapper, false);

                }
            });
        });
    }

    private void startScrap(int mId,
                            MallWithUser mallWithUser,
                            MallViewHandler mallViewService,
                            MartScrapper martScrapper,
                            MarketScrapper marketScrapper,
                            boolean isLoginSkip) {

        MALL mall = fromMallId(mId);

        if(mall != null) {
            switch (mall) {
                case Emart:
                case LotteMart:
                case Homeplus:
                    startScrap(mallWithUser, martScrapper, isLoginSkip);
                    break;

                case SSG:
                case MarketKurly:
                    startScrap(mallWithUser, marketScrapper);

                    break;
                default:
                    startScrap(mallWithUser, createWebViewManager(mallViewService), !isLoginSkip);
                    break;
            }
        }

    }

    private WebViewManager createWebViewManager(@Nullable MallViewHandler mallViewService) {
        mWebViewManager = new WebViewManager(context, mallViewService);
        return mWebViewManager;
    }

    private void startScrap(MallWithUser mallWithUser, MartScrapper martScrapper, boolean isLoginSkip) {
        scrapManager.startScrap(mallWithUser, martScrapper, isLoginSkip);
    }

    private void startScrap(MallWithUser mallWithUser, MarketScrapper marketScrapper) {
        scrapManager.startScrap(mallWithUser, marketScrapper);
    }

    private void startScrap(MallWithUser mallWithUser, WebViewManager webViewManager, boolean shouldInsert) {
        scrapManager.startScrap(mallWithUser, webViewManager, shouldInsert);
    }

    private Handler createHandler() {
        return new Handler(Looper.getMainLooper()) {

            @Override
            public void handleMessage(Message msg) {

                ScrapState value = ScrapState.values()[msg.what];
                Bundle bundle = msg.getData();
                String errMsg = bundle.getString(HANDLER_MSG);
                int mallId = bundle.getInt(HANDLER_MSG_MALL_ID);
                int errorCode = bundle.getInt(HANDLER_MSG_ERROR);
                ERROR error = ERROR.values()[errorCode];

                progressStateArray.put(mallId, new LoadingState(false));

            }
        };
    }

    @Override
    public void startParsingMalls() {

        if (DateUtils.checkDiffTime(preferences.getLong(PREF_KEY.CHECK_PARSING_TIME, 0),
                preferences.getInt(PREF_KEY.PARSING_FREQUENCY, 10))) {

            syncRule(null);

            Timber.i("parsing checkDiffTime true");
            preferences.edit().putLong(PREF_KEY.CHECK_PARSING_TIME, System.currentTimeMillis()).apply();
            // 몰 & 유저정보 가져오기
            repository.getMallWithActiveUsers((items -> {
                Timber.i("getMallsAndParsing"+"item : "+items);

                for (MallWithUser mallWithUser : items) {
                    startScrap(mallWithUser.mall.getId(),
                            mallWithUser,
                            null,
                            martScrapper,
                            marketScrapper,
                            true);

                    Timber.i("getMallsAndParsing"+"mallWithUser : "+mallWithUser);
                }
            }));
        }
    }

    private void syncRule(@Nullable ScrapService.onResultInterface<Void> callback) {
        repository.syncRule(callback);
    }

    @Override
    public void setEnabled(boolean enabled) {
        preferences.edit().putBoolean(PREF_KEY.SDK_IS_ENABLE, enabled).apply();
    }

    @Override
    public void getOrderList(int limit, ViewMoreInfo viewMoreInfo, OnResultCallback<List<OrderInfo>> callback) {
        repository.getOrderList(limit, viewMoreInfo, callback::onDataLoaded);
    }

    @Override
    public void getMallWithUsers(Callback<List<MallWithUser>> callback) {

        syncRule(new onResultInterface<Void>() {
            @Override
            public void onDataLoaded(Void item) {
                loadMalls(callback);
            }

            @Override
            public void onFail() {
                loadMalls(callback);
            }
        });
    }

    private void loadMalls(Callback<List<MallWithUser>> callback) {

        repository.getMallWithUser(items -> {
            for(MallWithUser mallWithUser : items) {

                LoadingState loadingState = progressStateArray.get(mallWithUser.mall.getId());
                boolean isLoading = loadingState != null && loadingState.isLoading();
                mallWithUser.setProgress(isLoading);
            }
            callback.onDataLoaded(items);
        });
    }

    @Override
    public void signOut(int mId, OnResultCallback<Void> callback) {
        // 해당 몰 유저, order data 삭제
        logout(mId);

        repository.deleteOrderByMallIds(mId);
        repository.deleteUsersByMallId(mId);
        if (callback != null) callback.onDataLoaded(null);
    }

    private void logout(int mId) {

        MALL mall = fromMallId(mId);

        if(mall != null) {
            switch (mall) {
                case Emart:
                case LotteMart:
                case Homeplus:
                    repository.getMallWithActiveUser(mId, new Callback<MallWithUser>() {
                        @Override
                        public void onDataLoaded(MallWithUser item) {
                            if(item != null && item.user != null)
                                martScrapper.logout(mId, item.user.getUserId());
                        }
                    });
                    break;
                case SSG:
                case MarketKurly:
                    marketScrapper.logout(mId);
                    break;
                default:
                    scrapManager.logout(mId);
                    break;
            }
        }
    }

    @Override
    public void hideOrder(int orderId) {
        repository.hideOrder(orderId);
    }

    @Override
    public void deleteOrder(int orderId) {
        repository.deleteOrder(orderId);
    }

    @Override
    public void deleteOrdersByOrderIds(int... orderIds) {
        repository.deleteOrders(orderIds);
    }

    @Override
    public void deleteAllOrders() {
        repository.deleteAllOrders();
    }

    @Override
    public void deleteOrderByMallIds(int... mIds) {
        repository.deleteOrderByMallIds(mIds);
    }

    @Override
    public void insertUserEntity(UserEntity userEntity) {
        repository.insertUsers(userEntity, null);
    }

    @Override
    public void setParsingFrequency(int frequency) {
        preferences.edit().putInt(PREF_KEY.PARSING_FREQUENCY, frequency).apply();
    }

    @Override
    public void getLastOrderId(ScrapService.Callback<Integer> callback) {
        repository.getLastOrderId(callback);
    }

}
