package com.tenqube.visual_scraper.webviewhtmlloader.util

import android.annotation.SuppressLint
import android.content.Context
import android.webkit.*
import com.tenqube.visual_scraper.webviewhtmlloader.model.AlertJS
import com.tenqube.visual_scraper.webviewhtmlloader.model.Response
import com.tenqube.visual_scraper.webviewhtmlloader.model.StatusCode
import kotlinx.coroutines.*
import timber.log.Timber
import kotlin.coroutines.Continuation
import kotlin.coroutines.resume
import kotlin.coroutines.suspendCoroutine

class WebViewManager(val context: Context,
                     var webView: WebView? = null) {

    var isInit = false
    var calledUrl = ""
    var isJs = false
    var alertJS: AlertJS? = null
    var htmlCallback: SuspendCallback<String>? = null

    var callback: SuspendCallback<Response>? = null

    fun initialize() {
        alertJS = null
        isInit = false
        calledUrl = ""
        isJs = false
        callback = null
        htmlCallback = null
    }

    private suspend fun createWebView() = withContext(Dispatchers.Main) {
        if(webView == null) webView = WebView(context)
        setupWebView()
    }

    private suspend fun loadUrl(url: String) = withContext(Dispatchers.Main) {
        Timber.d(  "loadUrl $url")

        if(!isInit) createWebView()
        calledUrl = url
        webView!!.loadUrl(url)

    }

    private fun getScrappingUrl() =
        "javascript:window.Android.getHtml(document.getElementsByTagName('html')[0].innerHTML);"

    suspend fun call(url: String, isJs: Boolean = false, hasHtml: Boolean = false): Response {

        this.isJs = isJs
        val timeOut = when(isJs) {
            true -> JS_TIME
            false -> LOADING_TIME
        }
         // 콜하고 응답없으면 html 조사 하기 캡챠 같은거일수있음
        return callUrlTimeOut(url, hasHtml, timeOut).run {

            when(this == null) {
                true -> { // time out 인경우
                    Timber.d(  "callUrlTimeOut timeout $url")
                    Response(StatusCode.TimeOut, url = url,  html = getHtmlOrNull(), alertJS = alertJS)
                }
                false -> {
                    this.copy(alertJS = alertJS)
                }
            }.also {
                Timber.d(  "callUrlTimeOut response $it")
            }
        }
    }

    private suspend inline fun <T> suspendCoroutineWithTimeout(timeout: Long, crossinline block: (Continuation<T>) -> Unit ) : T? {
        var finalValue : T? = null
        withTimeoutOrNull(timeout) {
            finalValue = suspendCancellableCoroutine(block = block)
        }
        return finalValue
    }


    private suspend fun callUrlTimeOut(url: String, hasHtml: Boolean = false, timeOut: Long): Response? {

        return suspendCoroutineWithTimeout(
            timeOut
        ) { cont ->
            Timber.d(  "callUrlTimeOut url $url hasBody $hasHtml")

            loadUrl(url, callback = object :
                    SuspendCallback<Response> {
                override suspend fun onDataLoaded(value: Response) {
                    callback = null
                    val result = value.copy(alertJS = alertJS)

                    Timber.d(  "onDataLoaded result $result")

                    if (hasHtml) { // html 값 파싱 해야할지 여부
                        val html = getHtmlOrNull()
                        cont.resume(result.copy(url = url, html = html))
                    } else {
                        cont.resume(result.copy(url = url))
                    }
                }
            })
        }
    }

    suspend fun getHtmlOrNull(): String? {
        delay(HTML_DELAY_TIME) // onPageLoaded 호출후 통신을 통해 추가적으로 데이터가 바인드 되는 시간 대기가 필요합니다.
        return withTimeoutOrNull(1000) {
            getHtml()
        }
    }

    private suspend fun getHtml(): String = suspendCoroutine { cont ->

        getHtml(callback = object :
                SuspendCallback<String> {
            override suspend fun onDataLoaded(html: String) {
                cont.resume(html)
            }
        })
    }


    private fun getHtml(callback: SuspendCallback<String>) {
        htmlCallback = callback
        GlobalScope.launch {
            loadUrl(getScrappingUrl())
        }
    }

    private fun loadUrl(url: String, callback: SuspendCallback<Response>) {
        this.callback = callback

        GlobalScope.launch {
            loadUrl(url)
        }
    }

    @SuppressLint("SetJavaScriptEnabled")
    private suspend fun setupWebView() = withContext(Dispatchers.Main) {
        webView?.apply {
            addJavascriptInterface(CustomBridge(
                object : SuspendCallback<String> {
                    override suspend fun onDataLoaded(value: String) {
                        htmlCallback?.onDataLoaded(value)
                    }

                }), "Android")

           webViewClient = object : WebViewClient() {
                override fun onPageFinished(view: WebView?, url: String?) {
                    super.onPageFinished(view, url)

                    Timber.d(  "onPageFinished url $url")

                    GlobalScope.launch {
                        sendCallback(StatusCode.Success, "", "")
                    }
                }

               override fun onReceivedError(
                   view: WebView?,
                   errorCode: Int,
                   description: String?,
                   failingUrl: String?
               ) {
                   Timber.d(  "onReceivedError url $url description $description")

                   GlobalScope.launch {
                       sendCallback(StatusCode.Error, description, "")
                   }

                   super.onReceivedError(view, errorCode, description, failingUrl)
               }

           }

            webChromeClient = object : WebChromeClient() {

                override fun onJsAlert(
                    view: WebView?,
                    url: String?,
                    message: String?,
                    result: JsResult?
                ): Boolean {

                    Timber.d(  "onJsAlert url $url message $message")
                    result?.confirm()
                    alertJS = AlertJS(url, message)
                    return true
                }
                
                override fun onConsoleMessage(cm: ConsoleMessage?): Boolean {
                    //js 실행했을경우에만 체크합니다.

                    GlobalScope.launch {
                        Timber.d(  "onConsoleMessage url ${cm?.message()}")

                        if(isJs && cm?.message()?.contains("Uncaught") == true) {
                            // js 실행후 에러 콜백 떨어짐 원하는 페이지가 로드 안된경우 실제 없는경우
//                            sendCallback(StatusCode.Error, cm.message(), "")
                        }
                    }

                    return super.onConsoleMessage(cm)
                }
            }

            settings.run {
                javaScriptEnabled = true
                userAgentString = "1"
                cacheMode = WebSettings.LOAD_NO_CACHE
                setAppCacheEnabled(false)
            }

            isInit = true
        }

    }

    private suspend fun sendCallback(statusCode: StatusCode, msg: String?, html: String) {
        callback?.onDataLoaded(
            Response(
                statusCode = statusCode,
                msg = msg,
                html = html,
                url = "",
                alertJS = alertJS
            )
        )
        callback = null
    }

    companion object {
        const val HTML_DELAY_TIME = 3000L // 웹페이지 로드후 html 정보 가져 오기전 대기시간.

        const val LOADING_TIME = 8000L // 웹페이지 로드후 html 정보 가져 오기전 대기시간.

        const val JS_TIME = 5000L // 웹페이지 로드후 html 정보 가져 오기전 대기시간.

    }

}