package com.tenqube.visual_scraper;

import android.content.Context;
import android.content.SharedPreferences;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.os.Message;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import android.util.SparseArray;
import android.webkit.WebView;

import com.tenqube.visual_scraper.constants.Constants;
import com.tenqube.visual_scraper.db.entity.UserEntity;
import com.tenqube.visual_scraper.manager.WebViewManager;
import com.tenqube.visual_scraper.model.LoadingState;
import com.tenqube.visual_scraper.model.query.MallWithUser;
import com.tenqube.visual_scraper.model.query.Password;
import com.tenqube.visual_scraper.model.view.OrderInfo;
import com.tenqube.visual_scraper.model.view.ViewMoreInfo;
import com.tenqube.visual_scraper.repository.ScrapRepository;
import com.tenqube.visual_scraper.third_party.mart.MartScrapper;
import com.tenqube.visual_scraper.third_party.mart.MartScrapperImpl;
import com.tenqube.visual_scraper.utils.DateUtils;

import java.util.List;
import java.util.Locale;

import timber.log.Timber;

import static com.tenqube.visual_scraper.ScrapManager.HANDLER_MSG;
import static com.tenqube.visual_scraper.ScrapManager.HANDLER_MSG_ERROR;
import static com.tenqube.visual_scraper.ScrapManager.HANDLER_MSG_MALL_ID;

public class ScrapServiceImpl implements ScrapService {

    public Context context;

    private ScrapRepository repository;
    private SharedPreferences preferences;

    private WebViewManager mWebViewManager;

    private DataInteraction dataInteraction;
    public static Locale locale;

    private ScrapManager scrapManager;

    private MartScrapper martScrapper;

    private SparseArray<LoadingState> progressStateArray = new SparseArray<>();

    private Handler handler;

    public ScrapServiceImpl(@NonNull Context context,
                            @NonNull String apiKey,
                            @NonNull String qualifier, /* @Nullable MallApiService api,*/
                            @NonNull DataInteraction dataInteraction) throws ParameterException {

        if(context == null) throw new ParameterException("context is null");
        if(TextUtils.isEmpty(apiKey)) throw new ParameterException("apiKey is empty");
        if(TextUtils.isEmpty(qualifier)) throw new ParameterException("qualifier is empty");
        if(dataInteraction == null) throw new ParameterException("dataInteraction is empty");

        this.context = context;
        this.dataInteraction = dataInteraction;

        initLocale();

        preferences = Injection.provideSharedPreference(context);
        preferences.edit().putString(Constants.PREF_KEY.LAYER, qualifier).apply();

        repository = Injection.provideScrapRepository(context.getApplicationContext(), preferences);
        repository.saveKey(apiKey);

        handler = createHandler();

        scrapManager = ScrapManager.getInstance(repository, handler);

        martScrapper = MartScrapperImpl.getInstance(context);

        if (qualifier.equals(Constants.LAYER.DEV)) {
            Timber.plant(new Timber.DebugTree());
        }

        singUp();

        syncRule(null);

    }

    /**
     *  사용자 등록
     */
    private void singUp() {
        if (!preferences.getString(Constants.PREF_KEY.UUID, "").equals(dataInteraction.getUid())) {

            dataInteraction.getAdId(adId -> {

                String appName = dataInteraction.getServiceAppName();
                if(TextUtils.isEmpty(appName))return;

                String uid = dataInteraction.getUid();
                if(TextUtils.isEmpty(uid))return;

                repository.signUpUser(appName, uid, adId, items -> {

                    Timber.i("appName :%s", appName);
                    Timber.i("uid: %s", uid);
                    Timber.i("adId: %s", adId);
                });
            });
        }
    }

    private void initLocale(){
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N){
            locale = context.getResources().getConfiguration().getLocales().get(0);
        } else{
            //noinspection deprecation
            locale =  context.getResources().getConfiguration().locale;
        }
    }

    @Override
    public void initialize() {
        repository.initialize();
    }

    @Override
    public void destroy() {
        progressStateArray.clear();
    }

    @Override
    public WebView getWebView() {
        return mWebViewManager.getWebView();
    }

    @Override
    public void startLoginAndParsing(String userId, String userPwd, int mId, MallViewHandler mallViewService, OnResultCallback<Integer> onResult) throws ParameterException {
        Timber.i("startLoginAndParsing = " + userId+"/"+mId + "/" + userPwd);
        if (TextUtils.isEmpty(userId) || TextUtils.isEmpty(userPwd) || mId == 0) throw new ParameterException("empty parameter");

        progressStateArray.put(mId, new LoadingState(true));

        // 몰 활성화 여부 조회
        repository.checkIsActiveMall(mId, (mall) -> {
            Timber.i("checkIsActiveMall : %s %s", mId, mall);
            // mall Active 하지 않거나, 해당 몰정보 없음
            if (mall == null) {
                onResult.onFail(mId, Constants.ERROR.EMPTY_MALL, "empty mall");
                return;
            }

            Timber.i("checkIsActiveMall : !!");
            scrapManager.setCallback(new OnResultCallback<Integer>() {
                @Override
                public void onDataLoaded(Integer item) {
                    progressStateArray.put(mId, new LoadingState(false));
                    onResult.onDataLoaded(item);
                }

                @Override
                public void onFail(int mId, Constants.ERROR error, String msg) {
                    progressStateArray.put(mId, new LoadingState(false));
                    onResult.onFail(mId, error, msg);

                }
            });
            // 유저 (& 몰정보) 가져오기

            repository.getMallWithUser(mId, userId, new onResultInterface<MallWithUser>() {
                @Override
                public void onDataLoaded(MallWithUser item) {

                    // 유저 정보 존재 -> 로그인 성공시 비번 다를경우, 업데이트
                    // login
                    Timber.i("item : %s", item);

                    MallWithUser mallWithUser = new MallWithUser(item.mall, new UserEntity(mId, userId, new Password(userPwd)));
                    startScrap(mId, mallWithUser, mallViewService, martScrapper);

                }

                @Override
                public void onFail() {
                    // 유저 및 몰정보 없음 -> 해당 mall 의 신규 user id 등록
                    Timber.i("mId : %s", mId);
                    MallWithUser mallWithUser = new MallWithUser(mall, new UserEntity(mId, userId, new Password(userPwd)));
                    startScrap(mId, mallWithUser, mallViewService, martScrapper);

                }
            });
        });
    }

    private void startScrap(int mId, MallWithUser mallWithUser, MallViewHandler mallViewService, MartScrapper martScrapper) {
        if(mId == Constants.MALL.Emart.getMallId() || mId == Constants.MALL.LotteMart.getMallId()) {
            startScrap(mallWithUser, martScrapper, true);
        } else {
            startScrap(mallWithUser, createWebViewManager(mallViewService), true);
        }
    }

    private WebViewManager createWebViewManager(@Nullable MallViewHandler mallViewService) {
        mWebViewManager = new WebViewManager(context, mallViewService);
        return mWebViewManager;
    }

    private void startScrap(MallWithUser mallWithUser, MartScrapper martScrapper, boolean shouldInsert) {
        scrapManager.startScrap(mallWithUser, martScrapper, shouldInsert);
    }

    private void startScrap(MallWithUser mallWithUser, WebViewManager webViewManager, boolean shouldInsert) {
        scrapManager.startScrap(mallWithUser, webViewManager, shouldInsert);
    }

    private Handler createHandler() {
        return new Handler(Looper.getMainLooper()) {

            @Override
            public void handleMessage(Message msg) {

                Constants.ScrapState value = Constants.ScrapState.values()[msg.what];
                Bundle bundle = msg.getData();
                String errMsg = bundle.getString(HANDLER_MSG);
                int mallId = bundle.getInt(HANDLER_MSG_MALL_ID);
                int errorCode = bundle.getInt(HANDLER_MSG_ERROR);
                Constants.ERROR error = Constants.ERROR.values()[errorCode];

                progressStateArray.put(mallId, new LoadingState(false));

//                switch (value) {
//                    case SUCCESS:
//                        if(mCallback != null) mCallback.onDataLoaded(mallId);
//                        break;
//                    default:
//                        if(mCallback != null) mCallback.onFail(mallId, error, errMsg);
//                        break;
//                }
            }
        };
    }

    @Override
    public void startParsingMalls() {

        if (DateUtils.checkDiffTime(preferences.getLong(Constants.PREF_KEY.CHECK_PARSING_TIME, 0),
                preferences.getInt(Constants.PREF_KEY.PARSING_FREQUENCY, 10))) {

            syncRule(null);

            Timber.i("parsing checkDiffTime true");
            preferences.edit().putLong(Constants.PREF_KEY.CHECK_PARSING_TIME, System.currentTimeMillis()).apply();
            // 몰 & 유저정보 가져오기
            repository.getMallWithActiveUsers((items -> {
                Timber.i("getMallsAndParsing"+"item : "+items);

                for (MallWithUser mallWithUser : items) {
                    Timber.i("getMallsAndParsing"+"mallWithUser : "+mallWithUser);
                    startScrap(mallWithUser, createWebViewManager(null), false);
                }
            }));
        }
    }

    private void syncRule(@Nullable ScrapService.onResultInterface<Void> callback) {

        repository.syncRule(callback);

//        repository.syncRule(new onResultInterface<Void>() {
//            @Override
//            public void onDataLoaded(Void item) {
//                repository.syncMartRule(callback);
//            }
//
//            @Override
//            public void onFail() {
//
//                callback.onFail();
//            }
//        });


    }

    @Override
    public void setEnabled(boolean enabled) {
        preferences.edit().putBoolean(Constants.PREF_KEY.SDK_IS_ENABLE, enabled).apply();
    }

    @Override
    public void getOrderList(int limit, ViewMoreInfo viewMoreInfo, OnResultCallback<List<OrderInfo>> callback) {
        repository.getOrderList(limit, viewMoreInfo, callback::onDataLoaded);
    }

    @Override
    public void getMallWithUsers(Callback<List<MallWithUser>> callback) {

        syncRule(new onResultInterface<Void>() {
            @Override
            public void onDataLoaded(Void item) {
                loadMalls(callback);
            }

            @Override
            public void onFail() {
                loadMalls(callback);
            }
        });
    }

    private void loadMalls(Callback<List<MallWithUser>> callback) {

        repository.getMallWithUser(items -> {
            for(MallWithUser mallWithUser : items) {

                LoadingState loadingState = progressStateArray.get(mallWithUser.mall.getId());
                boolean isLoading = loadingState != null && loadingState.isLoading();
                mallWithUser.setProgress(isLoading);
            }
            callback.onDataLoaded(items);
        });
    }

    @Override
    public void signOut(int mId, OnResultCallback<Void> callback) {
        // 해당 몰 유저, order data 삭제
        repository.deleteOrderByMallIds(mId);
        repository.deleteUsersByMallId(mId);
        if (callback != null) callback.onDataLoaded(null);
    }

    @Override
    public void hideOrder(int orderId) {
        repository.hideOrder(orderId);
    }

    @Override
    public void deleteOrder(int orderId) {
        repository.deleteOrder(orderId);
    }

    @Override
    public void deleteOrdersByOrderIds(int... orderIds) {
        repository.deleteOrders(orderIds);
    }

    @Override
    public void deleteAllOrders() {
        repository.deleteAllOrders();
    }

    @Override
    public void deleteOrderByMallIds(int... mIds) {
        repository.deleteOrderByMallIds(mIds);
    }

    @Override
    public void insertUserEntity(UserEntity userEntity) {
        repository.insertUsers(userEntity, null);
    }

    @Override
    public void setParsingFrequency(int frequency) {
        preferences.edit().putInt(Constants.PREF_KEY.PARSING_FREQUENCY, frequency).apply();
    }

    @Override
    public void getLastOrderId(ScrapService.Callback<Integer> callback) {
        repository.getLastOrderId(callback);
    }

}
