package com.tenqube.visual_scraper;

import android.os.Handler;
import android.support.annotation.Nullable;
import android.text.TextUtils;

import com.tenqube.visual_scraper.constants.Constants;
import com.tenqube.visual_scraper.db.entity.OrderEntity;
import com.tenqube.visual_scraper.mall.FactoryMall;
import com.tenqube.visual_scraper.mall.Mall;
import com.tenqube.visual_scraper.mall.data.MallData;
import com.tenqube.visual_scraper.manager.WebViewManager;
import com.tenqube.visual_scraper.model.query.MallWithUser;
import com.tenqube.visual_scraper.repository.ScrapRepository;
import com.tenqube.visual_scraper.third_party.mart.MartData;
import com.tenqube.visual_scraper.third_party.mart.MartScrapper;
import com.tenqube.visual_scraper.utils.Utils;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import timber.log.Timber;

public class ScrapManager {

    private static ScrapManager sInstance;


    public static final String HANDLER_MSG_ERROR = "error";
    public static final String HANDLER_MSG = "msg";
    public static final String HANDLER_MSG_MALL_ID = "mallId";

    // Sets the initial threadpool size to 8
    private static final int CORE_POOL_SIZE = 2;

    // Sets the maximum threadpool size to 8
    private static final int MAXIMUM_POOL_SIZE = 4;


    // Sets the amount of time an idle thread will wait for a task before terminating
    private static final int KEEP_ALIVE_TIME = 10;

    // Sets the Time Unit to seconds
    private static final TimeUnit KEEP_ALIVE_TIME_UNIT = TimeUnit.SECONDS;


    private BlockingQueue<Runnable> mParserWorkQueue;

    private ThreadPoolExecutor mParserThreadPool;

    private final ScrapRepository repository;

    private final Handler handler;

    private ScrapService.OnResultCallback<Integer> callback;



    public static ScrapManager getInstance(final ScrapRepository repository, final Handler handler) {
        if (sInstance == null) {
            synchronized (ScrapManager.class) {
                if (sInstance == null) {
                    sInstance = new ScrapManager(repository, handler);
                }
            }
        }
        return sInstance;
    }

    public void setCallback(ScrapService.OnResultCallback<Integer> callback) {
        this.callback = callback;
    }

    private ScrapManager(ScrapRepository repository, Handler handler) {

        this.repository = repository;
        this.handler = handler;

        mParserWorkQueue = new LinkedBlockingQueue<>();
        mParserThreadPool = new ThreadPoolExecutor(CORE_POOL_SIZE, MAXIMUM_POOL_SIZE,
                KEEP_ALIVE_TIME, KEEP_ALIVE_TIME_UNIT, mParserWorkQueue);

    }

    private void sendMsg(@Nullable Handler handler, Constants.ScrapState state, Constants.ERROR error, String msg, int mallId) {

        if(callback != null) {
            if(state == Constants.ScrapState.SUCCESS) {
                callback.onDataLoaded(mallId);
            } else {
                callback.onFail(mallId, error, msg);
            }
        }

        Thread.interrupted();

//        if(handler != null) {
//            Message message = new Message();
//            message.what = state.ordinal();
//            Bundle bundle = new Bundle();
//            bundle.putInt(HANDLER_MSG_ERROR, error.ordinal());
//            bundle.putString(HANDLER_MSG, msg);
//            bundle.putInt(HANDLER_MSG_MALL_ID, mallId);
//            message.setData(bundle);
//            handler.sendMessage(message);
//        }

    }

    void startScrap(MallWithUser mallWithUser, MartScrapper martScrapper, boolean shouldInsert) {
        mParserThreadPool.execute(() -> {
            try {

                if (Thread.interrupted()) {
                    throw new InterruptedException();
                }

                martScrapper.init(msg -> {
                    if(!TextUtils.isEmpty(msg)) {
                        sendMsg(handler, Constants.ScrapState.FAIL, Constants.ERROR.NONE,  msg, mallWithUser.mall.getId());
                        return;
                    }
                    martScrapper.addUser(mallWithUser.mall.getId(),
                            mallWithUser.user.getUserId(),
                            mallWithUser.user.getUserPwd().getValue(), userMsg -> {

                        if(!TextUtils.isEmpty(userMsg)) {
                            sendMsg(handler,Constants.ScrapState.FAIL, Constants.ERROR.NONE,  userMsg, mallWithUser.mall.getId());
                            return;
                        }

                        mallWithUser.user.setLogin(true);
                        repository.insertUsers(mallWithUser.user, uId -> {
                            mallWithUser.user.setId(uId);

                            martScrapper.start(mallWithUser.mall.getId(), scrapMsg -> {

                                if(!TextUtils.isEmpty(scrapMsg)) {
                                    sendMsg(handler, Constants.ScrapState.FAIL, Constants.ERROR.NONE,  scrapMsg, mallWithUser.mall.getId());
                                    return;
                                }

                                martScrapper.getMartItems(mallWithUser.mall.getId(), martItems -> {

                                    //martItems 변경하기
                                    List<OrderEntity> results = new ArrayList<>();
                                    for(MartData mart : martItems) {
                                        results.add(mart.toEntity(mallWithUser.mall.getId(), mallWithUser.user.getId()));
                                    }

                                    repository.insertOrders(results, (insertedList -> {

                                        long lastScrapAt = Utils.getLastScrapAt();
                                        repository.updateUserLastScrapAt(lastScrapAt, mallWithUser.mall.getId(), mallWithUser.user.getUserId());

                                        repository.callInsertOrderList();

                                        sendMsg(handler, Constants.ScrapState.SUCCESS, Constants.ERROR.NONE,  "success", mallWithUser.mall.getId());

                                    }));
                                });
                            });
                        });
                    });

                });
                    //sdk에서 받아서


            }catch (Exception e) {
                sendMsg(handler, Constants.ScrapState.FAIL, Constants.ERROR.NONE, e.toString(), mallWithUser.mall.getId());
            }
        });
}

    void startScrap(MallWithUser mallWithUser, WebViewManager webViewManager, boolean shouldInsert) {
        mParserThreadPool.execute(() -> {
            try {

                if (Thread.interrupted()) {
                    throw new InterruptedException();
                }

                Timber.i("login" + "MallWithUser : "+mallWithUser);
                // login rule 가져오기
                repository.getLoginRule(mallWithUser.mall.getId(), (loginRule) -> {
                    Timber.i("login" + "loginRule : "+loginRule);

                    Mall m = FactoryMall.create(new MallData(mallWithUser.mall.getId(), mallWithUser.mall.getName(), mallWithUser.mall.getDisplayName()), repository, webViewManager);
                    Timber.i("login" + "Mall : "+m);
                    if (m == null) {
                        sendMsg(handler, Constants.ScrapState.FAIL, Constants.ERROR.EMPTY_MALL,  "empty factory mall", mallWithUser.mall.getId());
                        return;
                    }

                    webViewManager.setMall(m);

                    if (loginRule != null && loginRule.loginWebRule != null) {
                        // web login
                        Timber.i("login" +"로그인 룰정보 존재, startWebViewLogin");
                        m.startWebViewLogin(loginRule.loginWebRule, mallWithUser.user, shouldInsert, new ScrapService.OnResult<List<OrderEntity>>() {
                            @Override
                            public void onSuccess(int mId, List<OrderEntity> items, String msg) {
                                repository.callInsertOrderList();
                                sendMsg(handler, Constants.ScrapState.SUCCESS, Constants.ERROR.NONE,  "success", mId);
                            }

                            @Override
                            public void onFail(int mId, Constants.ERROR error, String msg) {
                                sendMsg(handler, Constants.ScrapState.FAIL, error,  msg, mId);
                            }
                        });

                    } else {
                        // login rule 정보 없음
                        Timber.i("login" +"로그인 룰정보 없음 ");
                        sendMsg(handler, Constants.ScrapState.FAIL, Constants.ERROR.NOT_FOUND_LOGIN_RULE, "로그인 룰정보 없음", mallWithUser.mall.getId());
                    }
                });

            } catch (Exception e) {
                sendMsg(handler, Constants.ScrapState.FAIL, Constants.ERROR.NOT_FOUND_LOGIN_RULE, e.toString(), mallWithUser.mall.getId());
            }

        });
    }


}
