package com.tenqube.visual_scraper;

import android.content.Context;
import android.content.SharedPreferences;
import android.os.Build;
import android.support.annotation.NonNull;
import android.text.TextUtils;
import android.util.Log;
import android.util.SparseBooleanArray;
import android.webkit.WebView;

import com.tenqube.visual_scraper.constants.Constants;
import com.tenqube.visual_scraper.db.entity.OrderEntity;
import com.tenqube.visual_scraper.db.entity.UserEntity;
import com.tenqube.visual_scraper.manager.WebViewManager;
import com.tenqube.visual_scraper.model.query.MallWithUser;
import com.tenqube.visual_scraper.model.query.Password;
import com.tenqube.visual_scraper.model.view.MallInfo;
import com.tenqube.visual_scraper.model.view.OrderInfo;
import com.tenqube.visual_scraper.model.view.ViewMoreInfo;
import com.tenqube.visual_scraper.repository.ScrapRepository;
import com.tenqube.visual_scraper.scrap.mall.FactoryMall;
import com.tenqube.visual_scraper.scrap.mall.Mall;
import com.tenqube.visual_scraper.utils.DateUtils;

import java.util.List;
import java.util.Locale;

import timber.log.Timber;

public class ScrapServiceImpl implements ScrapService/*, ScrapService.OnResultForMall*/ {

    public Context context;
    private final String apiKey;
    private final String qualifier;
    ScrapRepository repository;
    private SharedPreferences preferences;
    private WebViewManager mWebViewManager;

    private String uuid;
    private String serviceAppName;
    private MallViewService mallViewService;
    public static Locale locale;
    public SparseBooleanArray progressStateArray = new SparseBooleanArray();

    public ScrapServiceImpl(@NonNull Context context, @NonNull MallViewService mallViewService, @NonNull String apiKey, @NonNull String qualifier, /* @Nullable MallApiService api,*/
                            @NonNull String uuid, @NonNull String serviceAppName) throws ParameterException {
        this.context = context;
        this.apiKey = apiKey;
        this.qualifier = qualifier;

        initLocale();

        preferences = Injection.provideSharedPreference(context);

        repository = Injection.provideScrapRepository(context.getApplicationContext(), preferences);
        repository.saveKey(apiKey);
        repository.saveLayer(qualifier);

        this.mallViewService = mallViewService;
        this.uuid = uuid;
        this.serviceAppName = serviceAppName;
    }

    private void initLocale(){
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N){
            locale = context.getResources().getConfiguration().getLocales().get(0);
        } else{
            //noinspection deprecation
            locale =  context.getResources().getConfiguration().locale;
        }
    }

//    private void init(String serviceAppName) {
//        // uuid 등록
//        if (!preferences.getString(Constants.PREF_KEY.UUID, "").equals(uuid)) {
//            Log.i("UUID","uuid 저장 및 사용자 등록");
//            repository.signUpUser(serviceAppName, uuid, (Void) -> {
//                //success get token
//                // 룰 동기화
//                repository.syncRule((callback) -> {
//                    // 동기화 완료
//
//                });
//            });
//
//        }
//    }

    @Override
    public WebView getWebView() {
        return mWebViewManager.getWebView();
    }

    @Override
    public void startLoginAndParsing(String userId, String userPwd, int mId, OnResultCallback<List<OrderEntity>> onResult) throws ParameterException {
        Log.i("startLoginAndParsing",userId+"/"+mId + "/" + userPwd);
        if (TextUtils.isEmpty(userId) || TextUtils.isEmpty(userPwd) || mId == 0) throw new ParameterException("empty parameter");

        if (!preferences.getString(Constants.PREF_KEY.UUID, "").equals(uuid)) {
            repository.signUpUser(serviceAppName, uuid, (Void) -> logInAndParsing(userId, userPwd, mId, onResult));
        } else {
            logInAndParsing(userId, userPwd, mId, onResult);
        }
    }


    private void logInAndParsing(String userId, String userPwd, int mId, OnResultCallback<List<OrderEntity>> onResult) {
        // sdk 활성화 여부 체크
        if(isEnabledSDK()) {
            return;
        }

        // 룰 동기화
        repository.syncRule((callback) -> {
            // 동기화 완료
            Log.i("logInAndParsing", "syncRule success");
            new Thread(() -> {

                progressStateArray.put(mId, true);
                start(userId, userPwd, mId, new OnResult<List<OrderEntity>>() {
                    @Override
                    public void onSuccess(int mId, List<OrderEntity> itemList, String msg) {
                        // itemList 바로 리턴 가능
                        // But, 중복 체크 (DB 에서 unique key 로?)

                        // 해당 몰Id data 조회해보고 리턴
                        progressStateArray.put(mId, false);
                        repository.getOrderList(items -> {

                            onResult.onDataLoaded(itemList);

                        }, mId);
                    }

                    @Override
                    public void onFail(int mId, Constants.ERROR error, String msg) {
                        progressStateArray.put(mId, false);

                        Log.i("logInAndParsing", "error" + error.name() + "onFail" + msg);
                        onResult.onFail(mId, error, msg);
                    }
                });
            }).start();
        });

    }


    public void start(String userId, String userPwd, int mId, OnResult callback) {

        // 몰 활성화 여부 조회
        repository.checkIsActiveMall(mId, (mall) -> {
            // mall Active 하지 않거나, 해당 몰정보 없음
            if (mall == null) return;

            // 유저 (& 몰정보) 가져오기
            repository.getMallWithUser(mId, userId, new onResultInterface<MallWithUser>() {
                @Override
                public void onDataLoaded(MallWithUser item) {
                    // 유저 정보 존재 -> 로그인 성공시 비번 다를경우, 업데이트
                    // login
                    Timber.i("item : %s", item);

                    login(new MallWithUser(item.mall, new UserEntity(mId, userId, new Password(userPwd))), Constants.CRUD.UPDATE, callback);
                }

                @Override
                public void onFail() {
                    // 유저 및 몰정보 없음 -> 해당 mall 의 신규 user id 등록
                    Timber.i("mId : %s", mId);
                    login(new MallWithUser(mall, new UserEntity(mId, userId, new Password(userPwd))), Constants.CRUD.INSERT, callback);
                }
            });
        });
    }

    private void login(MallWithUser mallWithUser, int crud, OnResult onResult) {

        Timber.i("login" + "MallWithUser : "+mallWithUser);
        // login rule 가져오기
        repository.getLoginRule(mallWithUser.mall.getId(), (loginRule) -> {
            Log.i("login","loginRule : "+loginRule);

            WebViewManager webViewManager = new WebViewManager(context, mallViewService/*, webView*/);
            mWebViewManager = webViewManager;
            Mall m = FactoryMall.create(mallWithUser.mall.getName(), repository, webViewManager);
            Log.i("login","Mall : "+m);
            if (m == null) return;
            webViewManager.setMall(m);

//            isDoing = true;
            /*if (loginRule != null && loginRule.loginApiRule != null) {
                // api login
                m.startApiLogin(loginRule, mallWithUser.user, crud, this, onResult);

            } else*/
            if (loginRule != null && loginRule.loginWebRule != null) {
                // web login
                Log.i("login","로그인 룰정보 존재, startWebViewLogin");
                m.startWebViewLogin(loginRule.loginWebRule, mallWithUser.user, crud, onResult);

            } else {
                // login rule 정보 없음
                Log.i("login","로그인 룰정보 없음 ");
                onResult.onFail(mallWithUser.mall.getId(), Constants.ERROR.NOT_FOUND_LOGIN_RULE, mallWithUser.mall.getDisplayName() +" login rule does not exist");
            }
        });
    }

    @Override
    public void startParsingMalls(OnResultCallback<List<OrderEntity>> onResult) {
        if (!preferences.getString(Constants.PREF_KEY.UUID, "").equals(uuid)) {
            repository.signUpUser(serviceAppName, uuid, (Void) -> {
                repository.syncRule((callback) -> {
                    getMallsAndParsing(onResult);
                });
            });
        } else {
            repository.syncRule((callback) -> {
                getMallsAndParsing(onResult);
            });
        }
    }

    private void getMallsAndParsing(OnResultCallback<List<OrderEntity>> onResult) {

        if (DateUtils.checkDiffTime(preferences.getLong(Constants.PREF_KEY.CHECK_PARSING_TIME, 0), preferences.getInt(Constants.PREF_KEY.PARSING_FREQUENCY, 10))) {
            Log.i("parsing","checkDiffTime true");
            preferences.edit().putLong(Constants.PREF_KEY.CHECK_PARSING_TIME, System.currentTimeMillis()).apply();
            // 몰 & 유저정보 가져오기
            repository.getMallWithActiveUsers((items -> {
                Log.i("getMallsAndParsing","item : "+items);
                for (MallWithUser mallWithUser : items) {
                    Log.i("getMallsAndParsing","mallWithUser : "+mallWithUser);
                    new Thread(() ->{

                        login(mallWithUser, 0, new OnResult<List<OrderEntity>>() {
                            @Override
                            public void onSuccess(int mId, List<OrderEntity> itemList, String msg) {
                                Log.i("onSuccess", "mId : "+mId + "Update size : "+itemList);
                                onResult.onDataLoaded(itemList);
                            }

                            @Override
                            public void onFail(int mId, Constants.ERROR error, String msg) {
                                Log.i("onFail", "mId : "+mId + "update error : "+error+msg);

                            }
                        });
                    }).start();
                }
            }));

        } else {
            Log.i("getMallsAndParsing","request later");
            onResult.onFail(-1, Constants.ERROR.REQUEST_FREQUENT,"request later");
        }
    }

    @Override
    public void startParsingMall(int mallId, OnResultCallback<List<OrderEntity>> onResult) {
        if (!preferences.getString(Constants.PREF_KEY.UUID, "").equals(uuid)) {
            repository.signUpUser(serviceAppName, uuid, (Void) -> {
                getInfoAndMallParsing(mallId, onResult);
            });
        } else {
            getInfoAndMallParsing(mallId, onResult);
        }
    }

    private void getInfoAndMallParsing(int mallId, OnResultCallback<List<OrderEntity>> onResult) {

        // 몰 & 유저정보 가져오기
        repository.getMallWithActiveUser(mallId, (mallWithUser) -> {
            login(mallWithUser, 0, new OnResult<List<OrderEntity>>() {
                @Override
                public void onSuccess(int mId, List<OrderEntity> itemList, String msg) {
                    Log.i("onSuccess", "mId : "+mId + "Update size : "+itemList);
                    onResult.onDataLoaded(itemList);
                }

                @Override
                public void onFail(int mId, Constants.ERROR error, String msg) {
                    Log.i("onFail", "mId : "+mId + "update error : "+error+msg);
                }
            });
        });
    }

    @Override
    public void setEnabled(boolean enabled) {
        preferences.edit().putBoolean(Constants.PREF_KEY.SDK_IS_ENABLE, enabled).apply();
    }

    private boolean isEnabledSDK() {
        return preferences.getBoolean(Constants.PREF_KEY.SDK_IS_ENABLE, false);
    }

    /**
     * view list
     * @param callback
     */
    @Override
    public void getOrderList(int limit, ViewMoreInfo viewMoreInfo, OnResultCallback<List<OrderInfo>> callback) {
        repository.getOrderList(limit, viewMoreInfo, callback::onDataLoaded);
    }

    @Override
    public void getMallWithUsers(Callback<List<MallWithUser>> callback) {
        if (!preferences.getString(Constants.PREF_KEY.UUID, "").equals(uuid)) {
            repository.signUpUser(serviceAppName, uuid, (Void) -> getMalls(callback));
        } else {
            getMalls(callback);
        }
    }

    private void getMalls(Callback<List<MallWithUser>> callback) {
        repository.syncRule((mCallback) ->
                repository.getMallWithUser(items -> {
                    for(MallWithUser mallWithUser : items) {
                        mallWithUser.setProgress(progressStateArray.get(mallWithUser.mall.getId()));
                    }
                    callback.onDataLoaded(items);
                }));
    }

    @Override
    public void signOut(int mId, OnResultCallback<Void> callback) {
        repository.updateUserForSignOut(mId, callback);
    }

    @Override
    public void hideOrder(int orderId) {
        repository.hideOrder(orderId);
    }

    @Override
    public void deleteOrder(int orderId) {
        repository.deleteOrder(orderId);
    }

    @Override
    public void deleteOrders(int... orderIds) {
        repository.deleteOrder(orderIds);
    }

    @Override
    public void deleteAll(int... mIds) {
        if (mIds != null) {
            repository.deleteForMallIds();
        } else {
            // 전체 삭제
            repository.deleteAll();
        }
    }

    @Override
    public void deleteForMallId(int mid) {
        repository.deleteForMallId(mid);
    }

    @Override
    public void insertUserEntity(UserEntity userEntity) {
        repository.insertUsers(userEntity);
    }

    @Override
    public void getMallList(Callback<List<MallInfo>> callback) {
        repository.getMallList(callback);
    }

    @Override
    public void setParsingFrequency(int frequency) {
        preferences.edit().putInt(Constants.PREF_KEY.PARSING_FREQUENCY, frequency).apply();
    }

    //    @Override
//    public void getMalls(Callback<List<MallInfo>> callback) {
//        if (isEnabledSDK()) return;
//
//
//    }
//
//    @Override
//    public void getOrders(Callback<List<OrderEntity>> callback, int... mIds) {
//        if (isEnabledSDK()) return;
//
//        // uuid 로 조회
//
//
//        // 파싱 & 저장
////        startParsingMalls();
//
//        // 몰리스트 조회
//
//    }
//
//    @Override
//    public void deleteAll(int... mIds) {
//
//    }
//
//    @Override
//    public void exportCSV(Callback<File> callback, int... mIds) {
//
//    }

//
//    @Override
//    public void deleteOrder(int orderId) {
//
//    }
//
//
//    @Override
//    public void updateOrder(OrderEntity order) {
//
//    }


//    @Override
//    public void getProducts(Callback<List<OrderEntity>> callback, Constants.MALL... malls) {
//
//    }
//
//    @Override
//    public void exportCSV() {
//
//    }

}
