/*
 * Copyright 2017, The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.tenqube.visual_scraper.db;

import android.arch.lifecycle.LiveData;
import android.arch.lifecycle.MutableLiveData;
import android.arch.persistence.db.SupportSQLiteDatabase;
import android.arch.persistence.room.Database;
import android.arch.persistence.room.Room;
import android.arch.persistence.room.RoomDatabase;
import android.content.Context;
import android.support.annotation.NonNull;

import com.tenqube.visual_scraper.db.dao.OrderDao;
import com.tenqube.visual_scraper.db.dao.ScrapDao;
import com.tenqube.visual_scraper.db.dao.UserDao;
import com.tenqube.visual_scraper.db.dao.UserWithOrdersDao;
import com.tenqube.visual_scraper.db.entity.OrderEntity;
import com.tenqube.visual_scraper.db.entity.ScrapEntity;
import com.tenqube.visual_scraper.db.entity.UserEntity;
import com.tenqube.visual_scraper.utils.AppExecutors;

import java.util.List;

/**
 * The Room Database that contains the ScreenImage table.
 */
@Database(entities = {
        ScrapEntity.class, UserEntity.class, OrderEntity.class}, version = 1, exportSchema = false)
//@TypeConverters({Converters.class})
public abstract class ScraperDatabase extends RoomDatabase {

    public static final String DATABASE_NAME = "online_malls";

    private static ScraperDatabase sInstance;

    public abstract ScrapDao scrapDao();

    public abstract UserDao userDao();

    public abstract OrderDao orderDao();

    public abstract UserWithOrdersDao userWithOrdersDao();


    private final MutableLiveData<Boolean> mIsDatabaseCreated = new MutableLiveData<>();


    public static ScraperDatabase getInstance(final Context context, final AppExecutors executors) {
        if (sInstance == null) {
            synchronized (ScraperDatabase.class) {
                if (sInstance == null) {
                    sInstance = buildDatabase(context.getApplicationContext(), executors);
                    sInstance.updateDatabaseCreated(context.getApplicationContext());
                }
            }
        }
        return sInstance;
    }
    /**
     * Build the database. {@link Builder#build()} only sets up the database configuration and
     * creates a new instance of the database.
     * The SQLite database is only created when it's accessed for the first time.
     */
    private static ScraperDatabase buildDatabase(final Context appContext,
                                             final AppExecutors executors) {

        return Room.databaseBuilder(appContext, ScraperDatabase.class, DATABASE_NAME)
                .addCallback(new Callback() {
                    @Override
                    public void onCreate(@NonNull SupportSQLiteDatabase db) {
                        super.onCreate(db);
                        executors.diskIO().execute(() -> {
                            // Add a delay to simulate a long-running operation
                            addDelay();
                            // Generate the data for pre-population
                            ScraperDatabase database = ScraperDatabase.getInstance(appContext, executors);
                            List<ScrapEntity> scrapEntities = DataGenerator.generateScraps();

                            insertData(database, scrapEntities);
                            // notify that the database was created and it's ready to be used
                            database.setDatabaseCreated();
                        });
                    }
                })
                .build();
    }

    /**
     * Check whether the database already exists and expose it via {@link #getDatabaseCreated()}
     */
    private void updateDatabaseCreated(final Context context) {
        if (context.getDatabasePath(DATABASE_NAME).exists()) {
            setDatabaseCreated();
        }
    }

    private void setDatabaseCreated(){
        mIsDatabaseCreated.postValue(true);
    }

    private static void insertData(final ScraperDatabase database, final List<ScrapEntity> scrapEntities) {
        database.runInTransaction(() -> {
            database.scrapDao().inserts(scrapEntities);
        });
    }

    private static void addDelay() {
        try {
            Thread.sleep(4000);
        } catch (InterruptedException ignored) {
        }
    }

    public LiveData<Boolean> getDatabaseCreated() {
        return mIsDatabaseCreated;
    }

}
