package com.tenqube.visual_scraper.repository;

import android.content.SharedPreferences;

import com.tenqube.visual_scraper.ScrapService;
import com.tenqube.visual_scraper.api.ScrapApiService;
import com.tenqube.visual_scraper.constants.Constants;
import com.tenqube.visual_scraper.db.ScraperDatabase;
import com.tenqube.visual_scraper.db.dao.OrderDao;
import com.tenqube.visual_scraper.db.dao.ScrapDao;
import com.tenqube.visual_scraper.db.dao.UserDao;
import com.tenqube.visual_scraper.db.dao.UserWithOrdersDao;
import com.tenqube.visual_scraper.db.entity.OrderEntity;
import com.tenqube.visual_scraper.db.entity.ScrapEntity;
import com.tenqube.visual_scraper.db.entity.UserEntity;
import com.tenqube.visual_scraper.utils.AppExecutors;
import com.tenqube.visual_scraper.vo.Mall;
import com.tenqube.visual_scraper.vo.api.Order;
import com.tenqube.visual_scraper.vo.api.Orders;
import com.tenqube.visual_scraper.vo.api.RequestDocumentData;
import com.tenqube.visual_scraper.vo.api.RuleData;

import java.util.ArrayList;
import java.util.List;

import retrofit2.Response;

public class ScrapRepository {

    private static ScrapRepository sInstance;

    private AppExecutors appExecutors;
    private final ScraperDatabase mDatabase;
    private final UserDao userDao;
    private final ScrapDao scrapDao;
    private final OrderDao orderDao;
    private final UserWithOrdersDao userWithOrdersDao;

    private final ScrapApiService mApi;
    private SharedPreferences preferences;

    private ScrapRepository(final ScraperDatabase database,
                            final ScrapApiService scrapApi,
                            final AppExecutors appExecutors,
                            final SharedPreferences preferences) {

        // local
        mDatabase = database;
        userDao = database.userDao();
        scrapDao = database.scrapDao();
        orderDao = database.orderDao();
        userWithOrdersDao = database.userWithOrdersDao();
        this.preferences = preferences;

        // remote
        mApi = scrapApi;
        this.appExecutors = appExecutors;
    }

    public static ScrapRepository getInstance(final ScraperDatabase database, final ScrapApiService scrapApi, final AppExecutors appExecutors, final SharedPreferences preferences) {
        if (sInstance == null) {
            synchronized (ScrapRepository.class) {
                if (sInstance == null) {
                    sInstance = new ScrapRepository(database, scrapApi, appExecutors, preferences);
                }
            }
        }
        return sInstance;
    }

    public void saveKey(String apiKey) {
        preferences.edit().putString(Constants.PREF_KEY.X_API_KEY, apiKey).apply();
    }

    public void saveLayer(String layer) {
        preferences.edit().putString(Constants.PREF_KEY.LAYER, layer).apply();
    }

    public void syncRule(ScrapService.Callback<Void> callback) {

        Runnable runnable = () -> {

            Response<RuleData> response;
            try {
                response = mApi.callScrapingRule(preferences.getInt(Constants.PREF_KEY.RULE_VERSION, 0)).execute();
                if(response.isSuccessful()) {
                    RuleData result = response.body();
                    if(result != null) {
                        preferences.edit().putInt(Constants.PREF_KEY.RULE_VERSION, result.getVersion()).apply();
                        scrapDao.inserts(result.getOnlineMalls());
                    }
                }
            } catch (Exception e) {
                e.printStackTrace();
            } finally {
                callback.onDataLoaded(null);

            }
        };
        appExecutors.networkIO().execute(runnable);
    }

    public void getUser(String userId, String pwd, Constants.MALL mall, ScrapService.Callback<UserEntity> callback) {

        Runnable runnable = () -> {

            UserEntity userEntity = null;
            try {
                userEntity = userDao.getUser(userId, pwd, mall.name());

                if(userEntity == null) {
                    userEntity = new UserEntity(0, mall.name(), userId, pwd, "");
                    userEntity.setId((int) userDao.insert(userEntity));
                }
            } catch (Exception e) {
                e.printStackTrace();
            } finally {
                UserEntity finalUserEntity = userEntity;
                appExecutors.mainThread().execute(() -> callback.onDataLoaded(finalUserEntity));
            }
        };

        appExecutors.diskIO().execute(runnable);

    }

    public void getUser(Constants.MALL mall, ScrapService.Callback<UserEntity> callback) {

        Runnable runnable = () -> {

            UserEntity userEntity = null;
            try {
                userEntity = userDao.getUser(mall.name());
            } catch (Exception e) {
                e.printStackTrace();
            } finally {
                UserEntity finalUserEntity = userEntity;
                appExecutors.mainThread().execute(() -> callback.onDataLoaded(finalUserEntity));
            }
        };

        appExecutors.diskIO().execute(runnable);

    }

    public void deleteUser(Constants.MALL mall) {
        Runnable runnable = () -> {
            try {
                userDao.deleteUser(mall.name());
            } catch (Exception e) {
                e.printStackTrace();
            }
        };

        appExecutors.diskIO().execute(runnable);
    }

    public void deleteUser(UserEntity userEntity) {
        Runnable runnable = () -> {
            try {
                userDao.deleteUser(userEntity.getId());
            } catch (Exception e) {
                e.printStackTrace();
            }
        };

        appExecutors.diskIO().execute(runnable);
    }

    public void getMalls(ScrapService.Callback<List<Mall>> callback) {
        Runnable runnable = () -> {
            List<Mall> malls = new ArrayList<>();
            try {
                List<ScrapEntity> scrapEntities = scrapDao.getScraps();
                List<UserEntity> userEntities = userDao.getUsers();

                for(ScrapEntity entity : scrapEntities) {
                    Mall mall = entity.toMall();
                    for(UserEntity userEntity : userEntities) {
                        if(entity.getName().equals(userEntity.getMallName())) {
                            mall.setSignIn(true);
                        }
                    }
                    malls.add(mall);
                }
            } catch (Exception e) {
                e.printStackTrace();
            } finally {
                appExecutors.mainThread().execute(() -> callback.onDataLoaded(malls));
            }
        };

        appExecutors.diskIO().execute(runnable);
    }

    public void geScrapRule(Constants.MALL mall, ScrapService.Callback<ScrapEntity> callback) {

        Runnable runnable = () -> {

            ScrapEntity scrapEntity = null;
            try {
                scrapEntity = scrapDao.getScrap(mall.name());
            } catch (Exception e) {
                e.printStackTrace();
            } finally {
                ScrapEntity finalScrapEntity = scrapEntity;
                appExecutors.mainThread().execute(() -> callback.onDataLoaded(finalScrapEntity));
            }
        };

        appExecutors.diskIO().execute(runnable);

    }

    public void requestMallParsing(int uid, RequestDocumentData html, ScrapService.Callback<Void> callback) {

        Runnable runnable = () -> {

            Response<Orders> response;
            try {
                response = mApi.callPurchaseParser(html).execute();

                if(response.isSuccessful()) {

                    Orders result = response.body();
                    if(result != null) {

                        for(Order order : result.getOrders()) {
                            orderDao.insert(order.toEntity(uid));
                        }
                    }
                }

            } catch (Exception e) {
                e.printStackTrace();
            } finally {
                appExecutors.mainThread().execute(() -> callback.onDataLoaded(null));

            }
        };
        appExecutors.networkIO().execute(runnable);

    }

    public void getAllOrders(ScrapService.Callback<List<OrderEntity>> callback) {
        Runnable runnable = () -> {

            List<OrderEntity> orderEntities = new ArrayList<>();
            try {
                orderEntities = orderDao.getOrders();
            } catch (Exception e) {
                e.printStackTrace();
            } finally {
                List<OrderEntity> finalOrderEntities = orderEntities;
                appExecutors.mainThread().execute(() -> callback.onDataLoaded(finalOrderEntities));
            }
        };

        appExecutors.diskIO().execute(runnable);
    }

}
