package com.tenqube.visual_scraper.request_interface.visualserver;

import android.support.annotation.NonNull;

import java.lang.annotation.Annotation;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.concurrent.Executor;

import retrofit2.Call;
import retrofit2.CallAdapter;
import retrofit2.Callback;
import retrofit2.Response;
import retrofit2.Retrofit;

import static com.tenqube.visual_scraper.utils.LogUtil.LOGI;

/**
 * Created by tenqube on 2018. 6. 14..cw.jeon
 */

public class VisualCallAdapter {

    public static final String TAG = "VisualCallAdapter";
    /** A callback which offers granular callbacks for various conditions. */
    interface MyCallback<T> {
        void success(Response<T> response);
        /** Called for 401 responses. */
        void deny();
        /** Called for [400, 500) responses, except 401. */
        void fail(Response<?> response);
    }

    interface MyCall<T> {
        void cancel();
        void enqueue(MyCallback<T> callback);
        MyCall<T> clone();
    }

    public static class VisualCallAdapterFactory extends CallAdapter.Factory {
        @Override public CallAdapter<?, ?> get(@NonNull Type returnType, @NonNull Annotation[] annotations,
                                               @NonNull Retrofit retrofit) {
            if (getRawType(returnType) != MyCall.class) {
                return null;
            }
            if (!(returnType instanceof ParameterizedType)) {
                throw new IllegalStateException(
                        "MyCall must have generic type (e.g., MyCall<ResponseBody>)");
            }
            Type responseType = getParameterUpperBound(0, (ParameterizedType) returnType);
            Executor callbackExecutor = retrofit.callbackExecutor();
            return new ErrorHandlingAdapter<>(responseType, callbackExecutor);
        }

        private static final class ErrorHandlingAdapter<R> implements CallAdapter<R, MyCall<R>> {
            private final Type responseType;
            private final Executor callbackExecutor;

            ErrorHandlingAdapter(Type responseType, Executor callbackExecutor) {
                this.responseType = responseType;
                this.callbackExecutor = callbackExecutor;
            }

            @Override public Type responseType() {
                return responseType;
            }

            @Override public MyCall<R> adapt(@NonNull Call<R> call) {
                return new MyCallAdapter<>(call, callbackExecutor);
            }
        }
    }

    /** Adapts a {@link Call} to {@link MyCall}. */
    static class MyCallAdapter<T> implements MyCall<T> {
        private final Call<T> call;
        private final Executor callbackExecutor;

        MyCallAdapter(Call<T> call, Executor callbackExecutor) {
            this.call = call;
            this.callbackExecutor = callbackExecutor;
        }

        @Override public void cancel() {
            call.cancel();
        }

        @Override public void enqueue(final MyCallback<T> callback) {
            call.enqueue(new Callback<T>() {
                @Override public void onResponse(@NonNull Call<T> call, @NonNull Response<T> response) {
                    // TODO if 'callbackExecutor' is not null, the 'callback' methods should be executed
                    int code = response.code();
                    if (code >= 200 && code < 300) {
                        callback.success(response);
                    } else if (code == 403) {
                        callback.deny();
                    } else {
                        callback.fail(response);
                    }
                }

                @Override public void onFailure(@NonNull Call<T> call, @NonNull Throwable t) {

                    LOGI(TAG,"onFailure Msg:" + t.getMessage());
                    callback.fail(null);
                }
            });
        }

        @Override public MyCall<T> clone() {
            return new MyCallAdapter<>(call.clone(), callbackExecutor);
        }
    }
}
