package com.taboola.android.plus.core;


import android.accounts.NetworkErrorException;
import android.app.Application;
import android.os.Build;
import android.support.annotation.Keep;
import android.support.annotation.MainThread;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.RequiresApi;
import android.support.annotation.RestrictTo;
import android.util.Log;

import com.taboola.android.utils.Logger;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.Executor;

@Keep
@RequiresApi(api = Build.VERSION_CODES.LOLLIPOP)
public class TaboolaSdkPlus {
    private static final String TAG = TaboolaSdkPlus.class.getSimpleName();

    private static TaboolaSdkPlus singleton = null;

    private SdkPlusCore sdkPlusCore;

    private AbsScheduledManager scheduledNotificationsManager;
    private AbsPushManager pushNotificationsManager;

    // todo handle adding new callbacks after init has already failed
    private final List<SdkPlusInitCallback> externalCallbacks = new ArrayList<>(); // todo synchronize access

    private final List<InternalSdkPlusCoreInitCallback> coreInitCallbacks = new ArrayList<>(); // todo synchronize access
    private final List<InternalFeatureInitCallback> pushInitCallbacks = new ArrayList<>(); // todo synchronize access
    private final List<InternalFeatureInitCallback> scheduledInitCallbacks = new ArrayList<>(); // todo synchronize access

    private boolean isScheduledInitInProgress = false; // todo synchronize access (if needed)
    private boolean isPushInitInProgress = false; // todo synchronize access (if needed)

    private ICoreProvider coreProvider;

    private SdkPlusPublisherInfo sdkPlusPublisherInfo;

    private Executor mainThreadExecutor;

    private TaboolaSdkPlus() {
    }

    static void setSdkPlusCore(SdkPlusCore sdkPlusCore) {
        getInstanceInternal().sdkPlusCore = sdkPlusCore;
    }

    static void setCoreProvider(ICoreProvider coreProvider) {
        getInstanceInternal().coreProvider = coreProvider;
    }

    static ICoreProvider getCoreProvider() {
        return getInstanceInternal().coreProvider;
    }

    private static TaboolaSdkPlus getInstanceInternal() {
        if (singleton == null) {
            synchronized (TaboolaSdkPlus.class) {
                if (singleton == null) {
                    singleton = new TaboolaSdkPlus();
                }
            }
        }
        return singleton;
    }

    /**
     * method for TaboolaSdkPlus initialization, which allows to set properties for sdk+ features <p>
     * NOTE: must be called only in {@link Application#onCreate()} method (call must be synchronous).
     *
     * @param info     publisher info, which contains publisherName, configId, activeFeatures
     * @param callback that will be called for each feature independently after a feature has finished
     *                 initializing (either finished successfully or with an error).
     */
    @RequiresApi(api = Build.VERSION_CODES.LOLLIPOP)
    public synchronized static void init(@NonNull SdkPlusPublisherInfo info, @Nullable SdkPlusInitCallback callback) {
        getInstanceInternal().initInternal(info, callback);
    }

    private void initInternal(@NonNull final SdkPlusPublisherInfo info, @Nullable SdkPlusInitCallback callback) {
        sdkPlusPublisherInfo = info;
        if (callback != null) {
            externalCallbacks.add(callback);
        }

        sdkPlusCore.init(info, coreProvider.getMigrationManager(), new CoreInitCallback() {
            @Override
            public void onCoreInitSuccessful(@NonNull SdkPlusCore sdkPlusCore) {
                onCoreInitialized(sdkPlusCore);
                initFeatures();
            }

            @Override
            public void onCoreInitFailed(Throwable throwable) {
                Logger.e(TAG, "onCoreInitFailed: " + throwable.toString(), throwable);

                for (InternalSdkPlusCoreInitCallback coreInitCallback : coreInitCallbacks) {
                    coreInitCallback.onSdkPlusCoreInitFailed(throwable);
                }
                coreInitCallbacks.clear();

                for (PlusFeature plusFeature : info.getActiveFeatures()) {
                    for (SdkPlusInitCallback externalCallback : externalCallbacks) {
                        externalCallback.onFeatureInitFailed(plusFeature, throwable);
                    }
                }
                externalCallbacks.clear();
            }
        });

    }

    private void onCoreInitialized(@NonNull SdkPlusCore sdkPlusCore) {
        mainThreadExecutor = sdkPlusCore.getSdkPlusExecutors().getMainThreadExecutor();

        for (InternalSdkPlusCoreInitCallback coreInitCallback : coreInitCallbacks) {
            coreInitCallback.onSdkPlusCoreInitSuccessful(sdkPlusCore);
        }
        coreInitCallbacks.clear();
    }

    private void initFeatures() {
        for (final PlusFeature plusFeature : sdkPlusPublisherInfo.getActiveFeatures()) {
            final InternalFeatureInitCallback internalFeatureInitCallback = new InternalFeatureInitCallback() {
                @Override
                public void onFeatureInitSuccessful(final PlusFeature plusFeature) {
                    mainThreadExecutor.execute(new Runnable() {
                        @Override
                        public void run() {
                            TaboolaSdkPlus.this.onFeatureInitSuccessful(plusFeature);
                        }
                    });
                }

                @Override
                public void onFeatureInitFailed(final PlusFeature plusFeature, final Exception exception) {
                    mainThreadExecutor.execute(new Runnable() {
                        @Override
                        public void run() {
                            TaboolaSdkPlus.this.onFeatureInitFailed(plusFeature, exception);
                        }
                    });
                }
            };

            switch (plusFeature) {
                case SCHEDULED_NOTIFICATIONS:
                    initScheduledNotifications(internalFeatureInitCallback);
                    break;

                case PUSH_NOTIFICATIONS:
                    initPushNotifications(internalFeatureInitCallback);
                    break;
            }
        }
    }

    private void initScheduledNotifications(final InternalFeatureInitCallback internalFeatureInitCallback) {
        scheduledNotificationsManager = coreProvider.getScheduledNotificationsManager();
        isScheduledInitInProgress = true;
        scheduledNotificationsManager.init(sdkPlusCore, sdkPlusPublisherInfo.getScheduledNotificationExtraProperties(),
                internalFeatureInitCallback);
    }

    private void initPushNotifications(final InternalFeatureInitCallback internalFeatureInitCallback) {
        pushNotificationsManager = coreProvider.getPushNotificationsManager();

        if (sdkPlusCore.getNetworkState().isConnected()) {
            isPushInitInProgress = true;
            pushNotificationsManager.init(sdkPlusCore, sdkPlusPublisherInfo.getPublisherName(),
                    internalFeatureInitCallback);
        } else {
            internalFeatureInitCallback.onFeatureInitFailed(PlusFeature.PUSH_NOTIFICATIONS, new NetworkErrorException("No internet connection"));
        }
    }

    @MainThread
    private void onFeatureInitSuccessful(PlusFeature plusFeature) {
        List<InternalFeatureInitCallback> internalCallbacks = getFeatureInitCallbacks(plusFeature);
        if (internalCallbacks != null) {
            for (InternalFeatureInitCallback internalFeatureInitCallback : internalCallbacks) {
                //this will fire that this module was init to internal parts inside our code
                internalFeatureInitCallback.onFeatureInitSuccessful(plusFeature);
            }

            internalCallbacks.clear();
        }

        for (SdkPlusInitCallback externalCallback : externalCallbacks) {
            externalCallback.onFeatureInitSuccessful(this, plusFeature);
        }

        switch (plusFeature) {
            case SCHEDULED_NOTIFICATIONS:
                isScheduledInitInProgress = false;
                break;
            case PUSH_NOTIFICATIONS:
                isPushInitInProgress = false;
                break;
        }

        if (!isScheduledInitInProgress && !isPushInitInProgress) {
            externalCallbacks.clear();
        }

    }

    @MainThread
    private void onFeatureInitFailed(PlusFeature plusFeature, Exception exception) {
        List<InternalFeatureInitCallback> internalCallbacks = getFeatureInitCallbacks(plusFeature);
        if (internalCallbacks != null) {
            for (InternalFeatureInitCallback internalFeatureInitCallback : internalCallbacks) {
                //this will fire that this module was init to internal parts inside our code
                internalFeatureInitCallback.onFeatureInitFailed(plusFeature, exception);
            }

            internalCallbacks.clear();
        }

        for (SdkPlusInitCallback externalCallback : externalCallbacks) {
            externalCallback.onFeatureInitFailed(plusFeature, exception);
        }

        if (!isScheduledInitInProgress && !isPushInitInProgress) {
            externalCallbacks.clear();
        }
    }

    private void subscribeForSdkPlusCoreInit(final InternalSdkPlusCoreInitCallback coreInitCallback) {
        if (sdkPlusCore != null && sdkPlusCore.isCoreInitialized()) {
            mainThreadExecutor.execute(new Runnable() {
                @Override
                public void run() {
                    coreInitCallback.onSdkPlusCoreInitSuccessful(sdkPlusCore);
                }
            });
        } else {
            coreInitCallbacks.add(coreInitCallback);
        }
    }

    private void subscribeForFeatureInit(@NonNull PlusFeature plusFeature,
                                         @NonNull final InternalFeatureInitCallback callback) {
        switch (plusFeature) {
            case SCHEDULED_NOTIFICATIONS:
                if (scheduledNotificationsManager != null && scheduledNotificationsManager.isInitialized()) {
                    mainThreadExecutor.execute(new Runnable() {
                        @Override
                        public void run() {
                            callback.onFeatureInitSuccessful(PlusFeature.SCHEDULED_NOTIFICATIONS);
                        }
                    });
                } else {
                    scheduledInitCallbacks.add(callback);
                }
                break;

            case PUSH_NOTIFICATIONS:

                if (pushNotificationsManager != null && pushNotificationsManager.isInitialized()) {
                    mainThreadExecutor.execute(new Runnable() {
                        @Override
                        public void run() {
                            callback.onFeatureInitSuccessful(PlusFeature.PUSH_NOTIFICATIONS);
                        }
                    });
                } else {
                    pushInitCallbacks.add(callback);
                }
                break;
        }
    }

    @Nullable
    private List<InternalFeatureInitCallback> getFeatureInitCallbacks(PlusFeature plusFeature) {
        if (plusFeature != null) {
            switch (plusFeature) {
                case SCHEDULED_NOTIFICATIONS:
                    return scheduledInitCallbacks;

                case PUSH_NOTIFICATIONS:
                    return pushInitCallbacks;
            }
        }

        return null;
    }

    @RestrictTo(RestrictTo.Scope.LIBRARY_GROUP)
    static void getSdkPlusCoreAsync(final InternalSdkPlusCoreInitCallback callback) {
        getInstanceInternal().subscribeForSdkPlusCoreInit(callback);
    }

    @RestrictTo(RestrictTo.Scope.LIBRARY_GROUP)
    static void getScheduledNotificationManagerAsync(final ScheduledManagerCallback callback) {

        getInstanceInternal().subscribeForFeatureInit(PlusFeature.SCHEDULED_NOTIFICATIONS, new InternalFeatureInitCallback() {
            @Override
            public void onFeatureInitSuccessful(PlusFeature plusFeature) {
                AbsScheduledManager manager = getInstanceInternal().scheduledNotificationsManager;
                callback.onManagerRetrieved(manager);
            }

            @Override
            public void onFeatureInitFailed(PlusFeature plusFeature, Exception exception) {
                callback.onManagerRetrieveFailed(exception);
            }
        });
    }

    static void getPushNotificationManagerAsync(final PushManagerCallback pushManagerCallback) {

        getInstanceInternal().subscribeForFeatureInit(PlusFeature.PUSH_NOTIFICATIONS, new InternalFeatureInitCallback() {
            @Override
            public void onFeatureInitSuccessful(PlusFeature plusFeature) {
                final AbsPushManager manager = getInstanceInternal().pushNotificationsManager;

                if (manager != null) {
                    pushManagerCallback.onManagerRetrieved(manager);
                } else {
                    pushManagerCallback.onManagerRetrieveFailed(new Throwable("Fail to get Push manager"));
                }
            }

            @Override
            public void onFeatureInitFailed(PlusFeature plusFeature, Exception exception) {
                pushManagerCallback.onManagerRetrieveFailed(exception);
            }
        });
    }

    /**
     * Retrieves an instance of TBLScheduledManager<p>
     *
     * @return instance of TBLScheduledManager if it's initialized in {@link TaboolaSdkPlus},
     * in other case it returns {@code null}
     */
    @Nullable
    public static TBLScheduledManager getScheduledNotificationManager() {
        TBLScheduledManager manager = getInstanceInternal().scheduledNotificationsManager;

        if (manager != null && manager.isInitialized()) {
            return manager;
        } else {
            return null;
        }
    }

    /**
     * Retrieves an instance of TBLPushManager<p>
     *
     * @return instance of TBLPushManager if it's initialized in {@link TaboolaSdkPlus},
     * in other case it returns {@code null}
     */
    @Nullable
    public static TBLPushManager getPushNotificationManager() {
        TBLPushManager manager = getInstanceInternal().pushNotificationsManager;

        if (manager != null && manager.isInitialized()) {
            return manager;
        } else {
            return null;
        }
    }

    static boolean isFeatureActivated(PlusFeature plusFeature) {
        final SdkPlusPublisherInfo sdkPlusPublisherInfo = getInstanceInternal().sdkPlusPublisherInfo;

        PlusFeature[] activeFeatures = sdkPlusPublisherInfo != null ? sdkPlusPublisherInfo.getActiveFeatures() : null;

        if (activeFeatures == null) {
            Log.e(TAG, "onReceive: couldn't get active features");
            return false;
        }

        List<PlusFeature> activeFeaturesList = Arrays.asList(activeFeatures);
        boolean isFeatureActive = activeFeaturesList.contains(plusFeature);

        Log.d(TAG, "isFeatureActivated: " + plusFeature + " is activated during sdk+ init: " + isFeatureActive);
        return isFeatureActive;
    }


}
