package com.taboola.android.plus;

import android.content.Context;
import android.os.Build;
import android.support.annotation.Keep;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.RequiresApi;
import android.support.annotation.UiThread;
import android.support.annotation.WorkerThread;
import android.support.v4.util.Pair;
import android.text.TextUtils;
import android.util.Log;
import android.util.SparseArray;

import com.taboola.android.SdkCore;
import com.taboola.android.api.TaboolaApi;
import com.taboola.android.global_components.advertisingid.AdvertisingIdInfo;
import com.taboola.android.monitor.TBPlusFeaturesParams;
import com.taboola.android.monitor.TBSdkFeature;
import com.taboola.android.monitor.TBSuspendMonitorPlus;
import com.taboola.android.plus.content.ContentConfig;
import com.taboola.android.plus.content.LanguagesConfig;
import com.taboola.android.plus.notification.BridgeInternalNotification;
import com.taboola.android.plus.notification.NotificationConfig;
import com.taboola.android.plus.notification.NotificationRemoteRepository;
import com.taboola.android.plus.notification.TBDeviceInfoUtil;
import com.taboola.android.plus.notification.TBNotificationAnalyticsManager;
import com.taboola.android.plus.notification.TBNotificationLocalStore;
import com.taboola.android.plus.notification.TBNotificationManager;
import com.taboola.android.plus.notification.killSwitch.FrequentCrashBlockConfig;
import com.taboola.android.plus.notification.killSwitch.KillSwitchConfig;
import com.taboola.android.plus.notification.killSwitch.SdkPlusExceptionHandler;
import com.taboola.android.plus.push_notifications.BridgeInternalPushNotification;
import com.taboola.android.plus.push_notifications.OcambaInitCallback;
import com.taboola.android.plus.push_notifications.PushNotificationsAnalyticsManager;
import com.taboola.android.plus.push_notifications.PushNotificationsConfig;
import com.taboola.android.plus.push_notifications.PushNotificationsUtil;
import com.taboola.android.plus.push_notifications.TBPushNotificationsManager;
import com.taboola.android.plus.shared.ConditionCheckUtil;
import com.taboola.android.plus.shared.LocalizationStrings;
import com.taboola.android.plus.shared.MigrationManager;
import com.taboola.android.plus.shared.SharedAnalyticsManager;
import com.taboola.android.plus.shared.SharedLocalStorage;
import com.taboola.android.plus.shared.TBNotificationExecutors;
import com.taboola.android.plus.shared.network.SdkPlusNetworkManager;
import com.taboola.android.utils.Logger;
import com.taboola.android.utils.MonitorUtils;

import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;

import static com.taboola.android.plus.shared.ResourcesUtil.isResourceValid;

@Keep
@RequiresApi(api = Build.VERSION_CODES.LOLLIPOP)
public class TaboolaPlus implements PublicApi.PublicTaboolaPlus {
    private static final String TAG = TaboolaPlus.class.getSimpleName();
    private static TaboolaPlus singleton = null;
    private static Context applicationContext;
    private static SdkCore sdkCore;
    private static SdkPlusNetworkManager sdkPlusNetworkManager;
    private static SparseArray<TBSdkFeature> sSdkFeatures;

    private SdkPlusExceptionHandler sdkPlusExceptionHandler;

    private boolean isInitialized = false;

    private TBNotificationManager notificationManager;
    private TBPushNotificationsManager pushNotificationsManager;
    private TBNotificationLocalStore notificationLocalStorage;
    private SharedLocalStorage sharedLocalStorage;

    private final Executor mainThreadExecutor;
    private final TBNotificationAnalyticsManager tbNotificationAnalyticsManager;
    private final SharedAnalyticsManager sharedAnalyticsManager;
    private final PushNotificationsAnalyticsManager pushNotificationsAnalyticsManager;

    private TaboolaPlus() {
        mainThreadExecutor = new TBNotificationExecutors.MainThreadExecutor();

        notificationLocalStorage = new TBNotificationLocalStore(applicationContext);
        sharedLocalStorage = new SharedLocalStorage(applicationContext);
        verifySharedLocalStorage(applicationContext, sharedLocalStorage);

        sharedAnalyticsManager = new SharedAnalyticsManager(sdkPlusNetworkManager, applicationContext, sharedLocalStorage);

        tbNotificationAnalyticsManager = new TBNotificationAnalyticsManager(applicationContext,
                notificationLocalStorage, sharedAnalyticsManager);

        notificationManager = BridgeInternalNotification
                .newTBNotificationManager(applicationContext,
                        notificationLocalStorage, sharedLocalStorage, sharedAnalyticsManager,
                        new NotificationRemoteRepository(applicationContext),
                        tbNotificationAnalyticsManager);

        pushNotificationsAnalyticsManager = new PushNotificationsAnalyticsManager(applicationContext, sharedAnalyticsManager);

        pushNotificationsManager = BridgeInternalPushNotification
                .createPushNotificationsManager(applicationContext, pushNotificationsAnalyticsManager, sharedLocalStorage);

        Executors.newCachedThreadPool().execute(new Runnable() {
            @Override
            public void run() {
                try {
                    initSdkPlusUncaughtExceptionHandler(tbNotificationAnalyticsManager, sharedAnalyticsManager);
                } catch (Exception e) {
                    Logger.e(TAG, "error while initSdkPlusUncaughtExceptionHandler", e);
                }
            }
        });
    }

    private static TaboolaPlus getInstanceInternal() {
        if (singleton == null) {
            synchronized (TaboolaPlus.class) {
                if (singleton == null) {
                    singleton = new TaboolaPlus();
                }
            }
        }
        return singleton;
    }

    static void setApplicationContext(@NonNull Context applicationContext) {
        TaboolaPlus.applicationContext = applicationContext;
        if (applicationContext == null) {
            Log.e(TAG, "setApplicationContext: context is null");
        } else {
            Log.d(TAG, "setApplicationContext: context is not null");
        }
    }

    static void setSdkCore(SdkCore sdkCore) {
        TaboolaPlus.sdkCore = sdkCore;
    }

    static void setNetworkManager(SdkPlusNetworkManager networkManager) {
        sdkPlusNetworkManager = networkManager;
    }

    static SdkPlusNetworkManager getNetworkManager() {
        return sdkPlusNetworkManager;
    }

    static AdvertisingIdInfo getAdvertisingIdInfo() {
        return sdkCore.getAdvertisingIdInfo();
    }

    @Nullable
    static Context getApplicationContext() { // todo refactor
        return applicationContext;
    }


    static TBNotificationAnalyticsManager getTBNotificationAnalyticsManager() {
        return getInstanceInternal().tbNotificationAnalyticsManager;
    }

    static PushNotificationsAnalyticsManager getPushNotificationsAnalyticsManager() {
        return getInstanceInternal().pushNotificationsAnalyticsManager;
    }

    public static SharedLocalStorage getSharedLocalStorage() {
        return getInstanceInternal().sharedLocalStorage;
    }

    /**
     * Must be called in order to use TaboolaPlus object (Should be called only once). Requires
     * internet connection on first launch (after that cache can be used). After either successful or
     * failed init, corresponding callback will be called. If you also want to be notified when init has failed use
     * {@link TaboolaPlus#init(String, String, Map, TaboolaPlusRetrievedCallback, TaboolaPlusRetrieveFailedCallback)}
     *
     * @param publisherName        publisherName provided by your Taboola account manager
     * @param configId             configId provided by your Taboola account manager
     * @param onSuccessfulCallback callback that will provide initialized TaboolaPlus object
     */
    @UiThread
    public static void init(@NonNull String publisherName,
                            @NonNull String configId,
                            @NonNull final TaboolaPlusRetrievedCallback onSuccessfulCallback) {
        init(publisherName, configId, onSuccessfulCallback, null);
    }

    /**
     * Must be called in order to use TaboolaPlus object (Should be called only once). Requires
     * internet connection on first launch (after that cache can be used). After either successful or
     * failed init, corresponding callback will be called.
     *
     * @param publisherName        publisherName provided by your Taboola account manager
     * @param configId             configId provided by your Taboola account manager
     * @param onSuccessfulCallback callback that will provide initialized TaboolaPlus object
     * @param onFailedCallback     callback that will provide {@code Throwable} that indicates init error
     */
    @UiThread
    public static void init(@NonNull String publisherName,
                            @NonNull String configId,
                            @NonNull final TaboolaPlusRetrievedCallback onSuccessfulCallback,
                            @Nullable final TaboolaPlusRetrieveFailedCallback onFailedCallback) {
        init(publisherName, configId, null, onSuccessfulCallback, onFailedCallback);
    }


    /**
     * Must be called in order to use TaboolaPlus object (Should be called only once). Requires
     * internet connection on first launch (after that cache can be used). After either successful or
     * failed init, corresponding callback will be called.
     *
     * @param publisherName              publisherName provided by your Taboola account manager
     * @param configId                   configId provided by your Taboola account manager
     * @param taboolaPlusExtraProperties publisher specific properties. Only set them if you were explicitly told to do so
     * @param onSuccessfulCallback       callback that will provide initialized TaboolaPlus object
     * @param onFailedCallback           callback that will provide {@code Throwable} that indicates init error
     */
    @UiThread
    public static void init(@NonNull String publisherName,
                            @NonNull String configId,
                            @Nullable Map<String, String> taboolaPlusExtraProperties,
                            @NonNull final TaboolaPlusRetrievedCallback onSuccessfulCallback,
                            @Nullable final TaboolaPlusRetrieveFailedCallback onFailedCallback) {
        if (applicationContext != null) {
            final TaboolaPlus taboolaPlus = getInstanceInternal();
            SharedLocalStorage sharedLocalStorage = taboolaPlus.sharedLocalStorage;

            if (MigrationManager.isMigrationRequired(applicationContext)) {
                MigrationManager.migrate(applicationContext);
            }

            if (Build.VERSION.SDK_INT < Build.VERSION_CODES.LOLLIPOP) {
                Log.i(TAG, "isInitialized: Android version is less than LOLLIPOP. Ignoring call.");
                onSuccessfulCallback.onTaboolaPlusRetrieved(taboolaPlus);
                return;
            }

            if (shouldInitFromMonitor()) {
                Pair<String, String> pair = getInitParamsFromMonitor(publisherName, configId);
                publisherName = pair.first;
                configId = pair.second;

                Pair<String, String> urls = getConfigUrlsFromMonitor();
                if (urls == null || TextUtils.isEmpty(urls.first) || TextUtils.isEmpty(urls.second)) {
                    Logger.d(TAG, "no urls to override from monitor");
                } else {
                    getNetworkManager().getConfigHandler().enableMonitorMode();
                }
            }

            setDefaultSettings(sharedLocalStorage);

            sharedLocalStorage.setPublisher(publisherName);
            sharedLocalStorage.setConfigId(configId);
            sharedLocalStorage.setTaboolaPlusExtraProperties(taboolaPlusExtraProperties);

            applyNewConfigs(sharedLocalStorage, publisherName, configId, taboolaPlusExtraProperties, onSuccessfulCallback, onFailedCallback, taboolaPlus);

            ConfigManager.triggerAsyncConfigsUpdate(sharedLocalStorage, applicationContext, publisherName, configId);
        } else {
            Logger.e(TAG, "init TaboolaPlus failed, application context is null");
        }
    }


    private static void setDefaultSettings(@NonNull SharedLocalStorage sharedLocalStorage) {
        sharedLocalStorage.setWifiOnlyModeRuntimeFlag(false);
        sharedLocalStorage.setEnabled(false);
        sharedLocalStorage.setApplicationName("");
        sharedLocalStorage.setTaboolaPlusExtraProperties(null);
    }


    @WorkerThread
    private void initSdkPlusUncaughtExceptionHandler(TBNotificationAnalyticsManager tbNotificationAnalyticsManager, SharedAnalyticsManager sharedAnalyticsManager) {

        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.LOLLIPOP) {
            Log.v(TAG, "initSdkPlusUncaughtExceptionHandler: Android version is less than LOLLIPOP. Ignoring call.");
            return;
        }

        //fetch old config from cache
        ConfigManager.CachedSdkConfigContainer cachedSdkConfig =
                ConfigManager.getCurrentSdkConfig(applicationContext);

        NotificationConfig notificationConfig = ConfigManager.CachedSdkConfigContainer.getNotificationConfig(cachedSdkConfig);
        FrequentCrashBlockConfig frequentCrashBlockConfig;
        if (notificationConfig == null) {
            Log.v(TAG, "initSdkPlusUncaughtExceptionHandler: current sdkPlus config is empty," +
                    " open crash report by default");

            //using default values of FrequentCrashBlockConfig
            frequentCrashBlockConfig = new FrequentCrashBlockConfig();
        } else {
            KillSwitchConfig killSwitchConfig = notificationConfig.getKillSwitchConfig();
            frequentCrashBlockConfig = killSwitchConfig.getFrequentCrashBlockConfig();
        }

        if (frequentCrashBlockConfig.isFrequentCrashBlockEnabled() && sdkPlusExceptionHandler == null) {

            sdkPlusExceptionHandler =
                    new SdkPlusExceptionHandler(applicationContext,
                            frequentCrashBlockConfig, tbNotificationAnalyticsManager, sharedAnalyticsManager);

            mainThreadExecutor.execute(new Runnable() {
                @Override
                public void run() {
                    //register sdk plus exception handler with sdk core
                    sdkCore.registerTaboolaExceptionHandler(sdkPlusExceptionHandler);
                }
            });
        }
    }

    @Override
    public TBNotificationManager getNotificationManager() {
        return notificationManager;
    }

    public TBPushNotificationsManager getPushNotificationsManager() {
        return pushNotificationsManager;
    }

    /**
     * Restore TaboolaPlus with last used init parameters but with new(!) config, or
     * reInit existing instance with new config
     */
    @UiThread
    static void reInit(@NonNull final TaboolaPlusRetrievedCallback onSuccessfulCallback,
                       @Nullable final TaboolaPlusRetrieveFailedCallback onFailedCallback) {
        if (applicationContext != null) {
            final TaboolaPlus taboolaPlus = TaboolaPlus.getInstanceInternal();
            final SharedLocalStorage sharedLocalStorage = taboolaPlus.sharedLocalStorage;
            final String publisherName = sharedLocalStorage.getPublisher();
            final String configId = sharedLocalStorage.getConfigId();
            @Nullable final HashMap<String, String> extraProperties =
                    sharedLocalStorage.getTaboolaPlusExtraProperties();

            if (!TextUtils.isEmpty(publisherName) && !TextUtils.isEmpty(configId)) {
                new TBNotificationExecutors.MainThreadExecutor().execute(new Runnable() {
                    @Override
                    public void run() {
                        applyNewConfigs(sharedLocalStorage, publisherName, configId, extraProperties, onSuccessfulCallback,
                                onFailedCallback, taboolaPlus);
                    }
                });
            }

            ConfigManager.triggerAsyncConfigsUpdate(sharedLocalStorage,
                    applicationContext, publisherName, configId);
        } else {
            Logger.e(TAG, "reInit TaboolaPlus failed, application context is null");
        }
    }

    @UiThread
    private static void applyNewConfigs(SharedLocalStorage sharedLocalStorage, @NonNull final String publisherName, @NonNull String configId,
                                        @Nullable final Map<String, String> taboolaPlusExtraProperties,
                                        @NonNull final TaboolaPlusRetrievedCallback onSuccessfulCallback,
                                        @Nullable final TaboolaPlusRetrieveFailedCallback onFailedCallback,
                                        @NonNull final TaboolaPlus taboolaPlus) {
        ConfigManager.getNewConfigs(sharedLocalStorage, applicationContext, publisherName, configId, true,
                new ConfigManager.OnGetAllConfigsCallback() {
                    @Override
                    public void onConfigsFetched(final SdkPlusConfig config, final LanguagesConfig languagesConfig) {
                        taboolaPlus.sharedAnalyticsManager.sendNotificationConfigRefreshedEvent();

                        new TBNotificationExecutors.MainThreadExecutor().execute(new Runnable() {
                            @Override
                            public void run() {
                                taboolaPlus.applyConfigs(config, languagesConfig, taboolaPlusExtraProperties,
                                        publisherName, onSuccessfulCallback);
                            }
                        });
                    }

                    @Override
                    public void onConfigsFailed(Throwable throwable) {
                        Log.e(TAG, "applyNewConfigs: failed to get configs: " + throwable.getMessage());

                        if (onFailedCallback != null) {
                            onFailedCallback.onTaboolaPlusRetrieveFailed(throwable);
                        }
                    }
                });
    }

    /**
     * Restore TaboolaPlus with last used init parameters and config
     */
    @UiThread
    static void restore(@NonNull final TaboolaPlusRetrievedCallback successfulCallback,
                        @Nullable final TaboolaPlusRetrieveFailedCallback failedCallback) {
        if (applicationContext != null) {
            final TaboolaPlus taboolaPlus = TaboolaPlus.getInstanceInternal();

            if (taboolaPlus.isInitialized() && TaboolaApi.getInstance().isInitialized()) {
                Log.i(TAG, "restore: successful (already initialized)");
                successfulCallback.onTaboolaPlusRetrieved(taboolaPlus);
            } else {
                final String publisherName = taboolaPlus.sharedLocalStorage.getPublisher();
                String configId = taboolaPlus.sharedLocalStorage.getConfigId();
                @Nullable final HashMap<String, String> extraProperties =
                        taboolaPlus.sharedLocalStorage.getTaboolaPlusExtraProperties();

                if (TextUtils.isEmpty(publisherName) || TextUtils.isEmpty(configId)) {
                    Throwable throwable = new IllegalStateException("Missing publisher or configId to get config");

                    Log.e(TAG, "restore: failed to get configs: " + throwable.getMessage());
                    if (failedCallback != null) {
                        failedCallback.onTaboolaPlusRetrieveFailed(throwable);
                    }
                } else {
                    ConfigManager.getCurrentConfigs(taboolaPlus.sharedLocalStorage,
                            applicationContext, publisherName, configId,
                            new ConfigManager.OnGetAllConfigsCallback() {
                                @Override
                                public void onConfigsFetched(final SdkPlusConfig config, final LanguagesConfig languagesConfig) {
                                    new TBNotificationExecutors.MainThreadExecutor().execute(new Runnable() {
                                        @Override
                                        public void run() {
                                            taboolaPlus.applyConfigs(config, languagesConfig, extraProperties,
                                                    publisherName, successfulCallback);
                                        }
                                    });
                                }

                                @Override
                                public void onConfigsFailed(Throwable throwable) {
                                    Log.e(TAG, "restore: failed to get configs: " + throwable.getMessage());
                                    if (failedCallback != null) {
                                        failedCallback.onTaboolaPlusRetrieveFailed(throwable);
                                    }
                                }
                            });
                }
            }
        } else {
            Logger.e(TAG, "restore TaboolaPlus failed, application context is null");
        }
    }

    @UiThread
    private void applyConfigs(@NonNull SdkPlusConfig sdkPlusConfig,
                              @NonNull LanguagesConfig languagesConfig,
                              @Nullable Map<String, String> taboolaPlusExtraProperties,
                              @NonNull String nonLocalizedPublisherName,
                              @NonNull final TaboolaPlusRetrievedCallback successfulCallback) {
        NotificationConfig notificationConfig = sdkPlusConfig.getNotificationConfig();
        ContentConfig contentConfig = sdkPlusConfig.getContentConfig();
        PushNotificationsConfig pushNotificationsConfig = sdkPlusConfig.getPushNotificationsConfig();

        sharedLocalStorage.setNonClickableUrlMarker(notificationConfig.getNonClickableUrlMarker());
        String configVariant = notificationConfig.getConfigVariant();
        if (!TextUtils.isEmpty(configVariant)) {
            sharedLocalStorage.setConfigVariant(configVariant);
        }

        initTaboolaApi(sharedLocalStorage, applicationContext, contentConfig, languagesConfig, taboolaPlusExtraProperties);
        String isoLanguage = Locale.getDefault().getLanguage().toUpperCase();
        String isoCountry = TBDeviceInfoUtil.getSimCountry(sharedLocalStorage, applicationContext);

        LocalizationStrings localizationStrings =
                ConditionCheckUtil.getLocalizeStrings(sdkPlusConfig.getLocalizationStrings(), isoCountry, isoLanguage);

        BridgeInternalNotification
                .applyConfig(notificationManager, notificationConfig, contentConfig, localizationStrings);

        BridgeInternalPushNotification
                .applyConfig(pushNotificationsManager, pushNotificationsConfig, localizationStrings);

        if (PushNotificationsUtil.shouldUsePushNotificationsFeature(pushNotificationsConfig)) {
            String localizedPublisherName = ConfigManager
                    .getLocalizedPublisherName(sharedLocalStorage, applicationContext, languagesConfig);

            BridgeInternalPushNotification
                    .initPushNotifications(pushNotificationsManager, nonLocalizedPublisherName, localizedPublisherName, configVariant, new OcambaInitCallback() {
                        @Override
                        public void onFailure() {
                            Logger.e(TAG, "Fail to init push flow");
                            isInitialized = true;
                            successfulCallback.onTaboolaPlusRetrieved(TaboolaPlus.this);
                        }

                        @Override
                        public void onSuccess() {
                            isInitialized = true;
                            successfulCallback.onTaboolaPlusRetrieved(TaboolaPlus.this);
                        }
                    });
        } else {
            isInitialized = true;
            successfulCallback.onTaboolaPlusRetrieved(this);
        }
    }

    @UiThread
    static void initTaboolaApi(SharedLocalStorage sharedLocalStorage,
                               Context appContext, @NonNull ContentConfig contentConfig,
                               @NonNull LanguagesConfig languagesConfig,
                               @Nullable Map<String, String> taboolaPlusExtraProperties) {
        String localizedPublisherName = ConfigManager
                .getLocalizedPublisherName(sharedLocalStorage, appContext, languagesConfig);
        String apiKey = contentConfig.getTaboolaApiConfig().getApiKey();

        Map<String, String> mergedExtraProperties = new HashMap<>();

        if (taboolaPlusExtraProperties != null) {
            // do not pass extra properties that are only used by TaboolaPlus
            mergedExtraProperties.putAll(taboolaPlusExtraProperties);
        }

        HashMap<String, String> taboolaApiExtraProperties = contentConfig.getTaboolaApiConfig().getTaboolaApiExtraProperties();
        if (taboolaApiExtraProperties != null) {
            mergedExtraProperties.putAll(taboolaApiExtraProperties);
        }

        mergedExtraProperties.put("enableFullRawDataResponse", "true");

        TaboolaApi.getInstance().init(appContext, localizedPublisherName, apiKey, mergedExtraProperties);
    }

    // region SDK Monitor


    private static boolean shouldInitFromMonitor() {
        boolean isInitFromMonitor;
        try {
            isInitFromMonitor = applicationContext != null && MonitorUtils.isSdkMonitorInstalled(applicationContext)
                    && !isMissingSdkFeatures();
        } catch (Exception e) {
            Logger.d(TAG, "shouldInitFromMonitor fail " + e);
            isInitFromMonitor = false;
        }
        
        return isInitFromMonitor;
    }

    static void setSdkFeatures(SparseArray<TBSdkFeature> sdkFeatures) {
        sSdkFeatures = sdkFeatures;
    }

    static SparseArray<TBSdkFeature> getSdkFeatures() {
        return sSdkFeatures;
    }

    static private boolean isMissingSdkFeatures() {
        return sSdkFeatures == null || sSdkFeatures.size() == 0;
    }

    @Nullable
    @SuppressWarnings("unchecked")
    static private <T extends TBSdkFeature> T getFeature(Integer key) {
        if (isMissingSdkFeatures()) return null;
        final TBSdkFeature sdkFeature = sSdkFeatures.get(key);
        return (T) sdkFeature;
    }

    /**
     * return init params from monitor
     *
     * @param publisherName
     * @param configId
     * @return
     */
    private static Pair<String, String> getInitParamsFromMonitor(@NonNull String publisherName,
                                                                 @NonNull String configId) {

        TBPlusFeaturesParams paramsChange = getFeature(TBPlusFeaturesParams.KEY);
        if (paramsChange != null && paramsChange.getInitParams() != null && isSdkMonitorEnabled()) {
            TBPlusFeaturesParams.InitParams params = paramsChange.getInitParams();
            String newPubId = params.getPublisherName();
            String newConfigId = params.getConfigID();

            if (!TextUtils.isEmpty(newPubId)) {
                publisherName = newPubId;
            }

            if (!TextUtils.isEmpty(newConfigId)) {
                configId = newConfigId;
            }
        }

        return new Pair<>(publisherName, configId);
    }

    /**
     * return config and lang config url from monitor
     */
    @Nullable
    static Pair<String, String> getConfigUrlsFromMonitor() {
        TBPlusFeaturesParams paramsChange = getFeature(TBPlusFeaturesParams.KEY);
        if (paramsChange != null && isSdkMonitorEnabled()) {
            return new Pair<>(paramsChange.getConfigUrl(), paramsChange.getLangUrl());
        }

        return null;
    }

    static boolean isSdkMonitorSuspended() {
        final TBSuspendMonitorPlus suspendMonitor = getFeature(TBSuspendMonitorPlus.KEY);
        return suspendMonitor != null && suspendMonitor.isShouldSuspend();
    }

    static boolean isSdkMonitorEnabled() {
        return !isSdkMonitorSuspended();
    }

    // endregion


    private void verifySharedLocalStorage(@NonNull Context appContext, @NonNull SharedLocalStorage sharedLocalStorage) {

        // region verify NotificationIcon
        if (!isResourceValid(appContext, sharedLocalStorage.getNotificationIcon())) {
            //delete invalid icon id
            sharedLocalStorage.removeNotificationIcon();
        }
        // endregion
    }


    private boolean isInitialized() {
        return isInitialized;
    }

    public interface TaboolaPlusRetrievedCallback {

        void onTaboolaPlusRetrieved(TaboolaPlus taboolaPlus);
    }

    public interface TaboolaPlusRetrieveFailedCallback {

        void onTaboolaPlusRetrieveFailed(Throwable throwable);
    }
}