package com.taboola.android.infra.inappupdate.internals;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.Context;

import androidx.annotation.AnyThread;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RestrictTo;

import com.google.android.play.core.install.model.UpdateAvailability;
import com.taboola.android.infra.inappupdate.Configuration;
import com.taboola.android.infra.inappupdate.TriggerEventsCallback;
import com.taboola.android.infra.utilities.Executor2;

public class Sdk {
    private static volatile Sdk instance;

    private final Executor2 backgroundExecutor;
    private final Module module;
    private final Runnable countersReset;
    @Nullable
    private Configuration configuration;

    @NonNull
    private TriggerEventsCallback triggerEventsCallback = TriggerEventsCallback.NULL;


    private Sdk(Context context) {
        module = new Module(context);
        backgroundExecutor = module.getBackgroundExecutor();
        countersReset = module.getCountersReset();
    }

    @AnyThread
    public static void initialize(Context context) {
        if (instance == null) {
            synchronized (Sdk.class) {
                if (instance == null) {
                    instance = new Sdk(context);
                }
            }
        }
    }

    @NonNull
    @RestrictTo(RestrictTo.Scope.LIBRARY_GROUP)
    public static Sdk getInstance() {
        if (instance == null) {
            throw new IllegalStateException("Not initialized. Did you call InAppUpdate.initialize()?");
        }
        return instance;
    }

    public void setConfiguration(@Nullable Configuration configuration) {
        this.configuration = configuration;
    }

    public void setTriggerEventsCallback(@Nullable TriggerEventsCallback triggerEventsCallback) {
        this.triggerEventsCallback = triggerEventsCallback != null ? triggerEventsCallback : TriggerEventsCallback.NULL;
    }

    @SuppressLint("Assert")
    @AnyThread
    public void onTriggerOccurred(Activity activity) {
        PreconditionsChecker preconditionsChecker = module.getPreconditionsChecker(configuration, triggerEventsCallback,
                updateInfo -> {
                    assert updateInfo.updateAvailability() == UpdateAvailability.UPDATE_AVAILABLE; //called only if so
                    assert configuration != null; //checked in PreconditionsChecker
                    Runnable updateController = module.getUpdateController(updateInfo, configuration, triggerEventsCallback, activity);
                    backgroundExecutor.execute(updateController);
                });
        backgroundExecutor.execute(preconditionsChecker);
    }

    public void resetCounters() {
        countersReset.run();
    }
}
