package com.taboola.android.infra.inappupdate.internals;

import android.app.Activity;
import android.content.Context;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.core.util.Consumer;

import com.google.android.play.core.appupdate.AppUpdateInfo;
import com.google.android.play.core.appupdate.AppUpdateManager;
import com.google.android.play.core.appupdate.AppUpdateManagerFactory;
import com.google.android.play.core.tasks.Task;
import com.taboola.android.infra.BuildConfig;
import com.taboola.android.infra.R;
import com.taboola.android.infra.inappupdate.Configuration;
import com.taboola.android.infra.inappupdate.InAppUpdateException;
import com.taboola.android.infra.inappupdate.TriggerEventsCallback;
import com.taboola.android.infra.persistence.BuilderWithContext;
import com.taboola.android.infra.persistence.IntegerPersistentResource;
import com.taboola.android.infra.persistence.LongPersistentResource;
import com.taboola.android.infra.persistence.PersistentResource;
import com.taboola.android.infra.utilities.BackgroundExecutor;
import com.taboola.android.infra.utilities.Executor2;
import com.taboola.android.infra.utilities.GeneralizedSupplier;
import com.taboola.android.infra.utilities.UiExecutor;

public class Module {

    private static final String SHARED_PREFERENCES_NAME = BuildConfig.LIBRARY_PACKAGE_NAME + ".inappupdate";
    private final Context context;
    private final Executor2 uiExecutor;

    private final PersistentResource<Long> lastAppearanceTime;
    private final PersistentResource<Integer> appearancesCount;

    public Module(Context context) {
        this.context = context.getApplicationContext();
        uiExecutor = new UiExecutor();
        lastAppearanceTime = new LongPersistentResource(new BuilderWithContext<>(context, R.string.inappupdate__last_appearance_time_key, SHARED_PREFERENCES_NAME, ctx -> 0L));
        appearancesCount = new IntegerPersistentResource(new BuilderWithContext<>(context, R.string.inappupdate__appearences_count_key, SHARED_PREFERENCES_NAME, ctx -> 0));
    }

    @NonNull
    PreconditionsChecker getPreconditionsChecker(@Nullable Configuration configuration,
                                                 TriggerEventsCallback triggerEventsCallback,
                                                 Consumer<AppUpdateInfo> onAllChecksPassed) {
        return new PreconditionsChecker(configuration,
                triggerEventsCallback,
                uiExecutor,
                lastAppearanceTime, appearancesCount) {
            @NonNull
            @Override
            protected AppData getAppData() throws InAppUpdateException {
                return getAppDataRetriever().get();
            }

            @Override
            protected void onAllChecksPassed(AppUpdateInfo updateInfo) {
                onAllChecksPassed.accept(updateInfo);
            }

            @Override
            protected void startMonitoringUpdateInProgress() {
                Runnable updateInProgressMonitor = getUpdateInProgressMonitor(triggerEventsCallback);
                getBackgroundExecutor().execute(updateInProgressMonitor);
            }

            @NonNull
            @Override
            protected Task<AppUpdateInfo> getAppUpdateInfoTask() {
                return getAppUpdateManager().getAppUpdateInfo();
            }
        };
    }

    private AppUpdateManager getAppUpdateManager() {
        return AppUpdateManagerFactory.create(context);
    }

    private GeneralizedSupplier<AppData, InAppUpdateException> getAppDataRetriever() {
        return new AppInfoRetrieverImpl(context);
    }

    public Runnable getUpdateController(AppUpdateInfo updateInfo, Configuration configuration, TriggerEventsCallback callback, Activity activity) {
        return new UpdateRunner(uiExecutor, updateInfo, configuration, callback, getAppUpdateManager(), activity) {
            @Override
            protected void startMonitoringUpdateInProgress() {
                Runnable updateInProgressMonitor = getUpdateInProgressMonitor(callback);
                getBackgroundExecutor().execute(updateInProgressMonitor);
            }
        };
    }

    public Runnable getUpdateInProgressMonitor(TriggerEventsCallback callback) {
        return new UpdateInProgressMonitor(uiExecutor, callback, getAppUpdateManager());
    }

    public BackgroundExecutor getBackgroundExecutor() {
        return new BackgroundExecutor();
    }

    Runnable getCountersReset() {
        return () -> {
            lastAppearanceTime.set(0L);
            appearancesCount.set(0);
        };
    }

}
