package com.t1.cloud.http;

import android.net.Uri;
import android.os.AsyncTask;
import android.text.TextUtils;
import android.util.Log;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.FileNotFoundException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.Map;

/**
 * AsyncTask 异步处理后台任务
 *
 * @author PopApp_laptop
 */
public class HttpAgent extends AsyncTask<Void, Void, String> {
  private static final int DEFAULT_TIMEOUT = 1000 * 10;
  private static final String CONTENT_TYPE_APPLICATION_JSON = "application/json";

  private final String TAG = getClass().getName();

  private String mUrl;
  private HttpURLConnection mConnection;
  private String mVerb;
  private Map<String, String> mQueryParams = new HashMap<>();
  private int mTimeOut = DEFAULT_TIMEOUT;
  private String mBody;
  private boolean mHasBody = false;
  private String mContentType = CONTENT_TYPE_APPLICATION_JSON;
  private HttpAgentResults mCallback;
  private String mErrorMessage = "";

  private HttpAgent(String url, String httpVerb) {
    this.mUrl = url;
    this.mVerb = httpVerb;
  }

  public static HttpAgent get(String url) {
    return new HttpAgent(url, HTTP.GET);
  }

  public static HttpAgent post(String url) {
    return new HttpAgent(url, HTTP.POST);
  }

  public static HttpAgent put(String url) {
    return new HttpAgent(url, HTTP.PUT);
  }

  public static HttpAgent delete(String url) {
    return new HttpAgent(url, HTTP.DELETE);
  }

  private static String createQueryStringForParameters(Map<String, String> parameters) {
    if (parameters == null) {
      return "";
    }

    Uri.Builder builder = new Uri.Builder();
    for (String key : parameters.keySet()) {
      builder.appendQueryParameter(key, parameters.get(key));
    }

    String query = builder.build().getEncodedQuery();
    if (query == null) query = "";
    if (query.trim().length() > 0) { query = "?" + query; }
    return query;
  }

  public HttpAgent queryParams(Map<String, String> stringMap) {
    for (String key : stringMap.keySet()) {
      mQueryParams.put(key, stringMap.get(key));
    }
    return this;
  }

  public HttpAgent queryParams(String... queryParams) {
    if (queryParams.length % 2 != 0) {
      Log.e(TAG, "query params must be even number");
      return this;
    }
    for (int i = 0; i < queryParams.length; i += 2) {
      mQueryParams.put(queryParams[i], queryParams[i + 1]);
    }
    return this;
  }

  private void setTimeOut(int timeOutInMillis) {
    this.mTimeOut = timeOutInMillis;
  }

  public HttpAgent contentType(String contentType) {
    this.mContentType = contentType;
    return this;
  }

  public HttpAgent withBody(String body) {
    this.mBody = body;
    this.mHasBody = !TextUtils.isEmpty(mBody);
    return this;
  }

  public void go(SuccessCallback callback) {
    this.mCallback = callback;
    this.execute();
  }

  public void goJson(JsonCallback callback) {
    this.mCallback = callback;
    this.execute();
  }

  public void goJsonArray(JsonArrayCallback callback) {
    this.mCallback = callback;
    this.execute();
  }

  public void goString(StringCallback callback) {
    this.mCallback = callback;
    this.execute();
  }

  @Override
  protected void onPreExecute() {

  }

  @Override
  protected String doInBackground(Void... params) {
    try {
      String queryParams = createQueryStringForParameters(mQueryParams);
      String urlWithParams = mUrl + queryParams;

      mConnection = (HttpURLConnection) (new URL(urlWithParams)).openConnection();
      mConnection.setRequestMethod(mVerb);
      mConnection.setReadTimeout(mTimeOut);
      mConnection.setDoInput(true);
      mConnection.setDoOutput(mHasBody);
      mConnection.setRequestProperty("Content-Type", mContentType);
      mConnection.connect();

      Log.d(TAG, "sending request to:\n" +
          urlWithParams +
          (mHasBody ? ("\nwith body:\n" + mBody) : ""));

      if (mHasBody) {
        Writer writer = new BufferedWriter(new OutputStreamWriter(mConnection.getOutputStream(), Charset.forName("UTF-8")));
        writer.write(mBody);
        writer.close();
      }

      int mResponseCode = mConnection.getResponseCode();
      BufferedReader reader = new BufferedReader(new InputStreamReader(mResponseCode >= 400 ? mConnection.getErrorStream() : mConnection.getInputStream()));
      StringBuilder stringBuilder = new StringBuilder();

      String line;
      while ((line = reader.readLine()) != null) {
        stringBuilder.append(line).append("\n");
      }

      if (mResponseCode >= 400) {
        mErrorMessage = stringBuilder.toString();
        return null;
      }
      return stringBuilder.toString();
    } catch (FileNotFoundException e) {
      mErrorMessage = e.getMessage();

    } catch (Exception e) {
      if (e.getMessage() != null) {
        mErrorMessage = e.getMessage();
      }

    } finally {
      if (mConnection != null) { mConnection.disconnect(); }
    }
    return null;
  }

  @Override
  protected void onPostExecute(String result) {
    if (!TextUtils.isEmpty(mErrorMessage)) {
      if (mCallback != null) {
        HttpAgentResults.errorMessage = mErrorMessage;
        mCallback.notify(null);
      }
      return;
    }

    if (!TextUtils.isEmpty(result)) {
      Log.d(TAG, "Server Response: " + result);
      if (mCallback != null) {
        HttpAgentResults.stringResults = result;
        mCallback.notify(result.trim());
      }
    }
  }
}
