package com.surveymonkey.surveymonkeyandroidsdk.model;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;

public class SMRespondent {
    //Constants
    private static final String RESPONDENT_ID = "respondent_id";
    private static final String COMPLETION_STATUS = "completion_status";
    private static final String DATE_START = "date_start";
    private static final String DATE_MODIFIED = "date_modified";
    private static final String RESPONSES = "responses";
    private static final String[] SM_COMPLETION_STATUSES = new String[]{"partially", "completed"};
    private final static String PAPI_DATE_FORMAT = "yyyy-MM-dd'T'HH:mm:ss";

    //Local Variables
    private SMCompletionStatus mCompletionStatus;
    private Calendar mDateModified;
    private Calendar mDateStarted;
    private String mRespondentID;
    private ArrayList <SMQuestionResponse> mQuestionResponses;

    /**
     * The SMRespondent object represents a single survey respondent and can be created by calling the public constructor and passing the JSON returned when a user successfully completes a survey.
     * <p>The SMRespondent object includes relevant metadata, e.g. survey completion status, as well as an array of SMQuestionResponse objects representing that respondent’s response to the given survey.
     *
     * @param jsonObject The JSON representing a user's response to a survey - returned by the SMFeedbackActivity in onActivityResult
     * @see com.surveymonkey.surveymonkeyandroidsdk.SMFeedbackActivity
     */
    public SMRespondent(JSONObject jsonObject) {
        try {
            mRespondentID = jsonObject.getString(RESPONDENT_ID);
            mCompletionStatus = SMCompletionStatus.values()[(Arrays.asList(SM_COMPLETION_STATUSES).indexOf(jsonObject.getString(COMPLETION_STATUS)))];
            mDateStarted = gregorianFormattedDateCalendarFromString(jsonObject.getString(DATE_START));
            mDateModified = gregorianFormattedDateCalendarFromString(jsonObject.getString(DATE_MODIFIED));
            ArrayList <SMQuestionResponse> responses = new ArrayList<>();
            JSONArray responsesArray =  jsonObject.getJSONArray(RESPONSES);
            for (int i = 0; i< responsesArray.length(); i++){
                SMQuestionResponse currentResponse = new SMQuestionResponse(responsesArray.getJSONObject(i));
                responses.add(currentResponse);
            }
            mQuestionResponses = responses;

        } catch (JSONException e) {
            e.printStackTrace();
        }
    }

    private Calendar gregorianFormattedDateCalendarFromString(String dateString) {
        Calendar gregorianCalendarDateString = null;
        if (dateString != null && !dateString.equals("")) {
            try {
                Date simpleDateFormat = new SimpleDateFormat(PAPI_DATE_FORMAT).parse(dateString.substring(0, dateString.length() - 2));
                gregorianCalendarDateString = new GregorianCalendar();
                gregorianCalendarDateString.setTime(simpleDateFormat);
            } catch (ParseException e) {
                //SMLog.error("[gregorianFormattedDateFromString] Data parse error: ", e);
            }
        }
        return gregorianCalendarDateString;
    }

    /**
     * An enumeration for the two possible survey completion statuses
     * @see com.surveymonkey.surveymonkeyandroidsdk.SMFeedbackActivity
     */
    public enum SMCompletionStatus {
        SMCompletionStatusPartiallyComplete(0),
        SMCompletionStatusComplete(1);

        public final int mValue;

        SMCompletionStatus(int value) {
            mValue = value;
        }

        public int getValue() {
            return mValue;
        }
    }

    public ArrayList getQuestionResponses() {
        return mQuestionResponses;
    }

    public SMCompletionStatus getCompletionStatus() {
        return mCompletionStatus;
    }

    public Calendar getDateModified() {
        return mDateModified;
    }

    public Calendar getDateStarted() {
        return mDateStarted;
    }

    public String getRespondentID() {
        return mRespondentID;
    }

    public String getDescription() {
        int size = 0;
        if (mQuestionResponses != null) {
            size = mQuestionResponses.size();
        }
        return "<SMRespondent: " + this +"; respondentID='" + mRespondentID + "'; " + size + " questionResponses>";
    }


}
