package com.surveymonkey.surveymonkeyandroidsdk.utils;

import java.io.Serializable;

public class SMError extends Error implements Serializable {

    public enum ErrorType {
        ERROR_CODE_TOKEN(1),
        ERROR_CODE_BAD_CONNECTION(2),
        ERROR_CODE_RESPONSE_PARSE_FAILED(3),
        ERROR_CODE_COLLECTOR_CLOSED(4),
        ERROR_CODE_RETRIEVING_RESPONSE(5),
        ERROR_CODE_SURVEY_DELETED(6),
        ERROR_CODE_RESPONSE_LIMIT_HIT(7),
        ERROR_CODE_RESPONDENT_EXITED_SURVEY(8),
        ERROR_CODE_NONEXISTENT_LINK(9),
        ERROR_CODE_INTERNAL_SERVER_ERROR(10),
        ERROR_CODE_USER_CANCELED(1);

        public final int mValue;

        ErrorType(int value) {
            mValue = value;
        }

        public int getValue() {
            return mValue;
        }
    }

    public static class ErrorDescription {
        public static final String ERROR_CODE_TOKEN_DESCRIPTION = "Could not retrieve your respondent. Be sure you're using an SDK Collector.";
        public static final String ERROR_CODE_BAD_CONNECTION_DESCRIPTION = "There was an error connecting to the server.";
        public static final String ERROR_CODE_RESPONSE_PARSE_FAILED_DESCRIPTION = "There was an error parsing the response from the server.";
        public static final String ERROR_CODE_COLLECTOR_CLOSED_DESCRIPTION = "The collector for this survey has been closed.";
        public static final String ERROR_CODE_RETRIEVING_RESPONSE_DESCRIPTION = "There was a problem retrieving the user's response to this survey.";
        public static final String ERROR_CODE_SURVEY_DELETED_DESCRIPTION = "This survey has been deleted.";
        public static final String ERROR_CODE_RESPONSE_LIMIT_HIT_DESCRIPTION = "Response limit exceeded for your plan. Upgrade to access more responses through the SDK.";
        public static final String ERROR_CODE_RESPONDENT_EXITED_SURVEY_DESCRIPTION = "The user canceled out of the survey.";
        public static final String ERROR_CODE_NONEXISTENT_LINK_DESCRIPTION = "Custom link no longer exists.";
        public static final String ERROR_CODE_INTERNAL_SERVER_ERROR_DESCRIPTION = "Internal server error.";
    }

    public static class Domain {
        public static final String SDK_SERVER_DOMAIN = "SurveyMonkeySDK_ServerError";
        public static final String SDK_CLIENT_DOMAIN = "SurveyMonkeySDK_ClientError";
    }

    public int errorCode;
    public String description;
    public String domain;
    private Exception exception;


    /**
     * The error object returned by SMFeedbackActivity in onActivityResult (see error codes below) – will be nil if survey was completed successfully.
     * *  <h3>SurveyMonkeySDK_ServerError</h3>
     * <table summary="server errors">
     * <tr>
     * <th>Error</th>
     * <th>Code</th>
     * <th>Description</th>
     * </tr>
     * <tr>
     * <td>ERROR_CODE_TOKEN</td>
     * <td>1</td>
     * <td>Could not retrieve your respondent. Be sure you're using an SDK Collector.</td>
     * </tr>
     * <tr>
     * <td>ERROR_CODE_BAD_CONNECTION</td>
     * <td>2</td>
     * <td>There was an error connecting to the server.</td>
     * </tr>
     * <tr>
     * <td>ERROR_CODE_RESPONSE_PARSE_FAILED</td>
     * <td>3</td>
     * <td>There was an error parsing the response from the server.</td>
     * </tr>
     * <tr>
     * <td>ERROR_CODE_COLLECTOR_CLOSED</td>
     * <td>4</td>
     * <td>The collector for this survey has been closed.</td>
     * </tr>
     * <tr>
     * <td>ERROR_CODE_RETRIEVING_RESPONSE</td>
     * <td>5</td>
     * <td>There was a problem retrieving the user's response to this survey.</td>
     * </tr>
     * <tr>
     * <td>ERROR_CODE_SURVEY_DELETED</td>
     * <td>6</td>
     * <td>This survey has been deleted.</td>
     * </tr>
     * <tr>
     * <td>ERROR_CODE_RESPONSE_LIMIT_HIT</td>
     * <td>7</td>
     * <td>Response limit exceeded for your plan. Upgrade to access more responses through the SDK.</td>
     * </tr>
     * <tr>
     * <td>ERROR_CODE_RESPONDENT_EXITED_SURVEY</td>
     * <td>8</td>
     * <td>The user canceled out of the survey.</td>
     * </tr>
     * <tr>
     * <td>ERROR_CODE_NONEXISTENT_LINK</td>
     * <td>9</td>
     * <td>Custom link no longer exists.</td>
     * </tr>
     * <tr>
     * <td>ERROR_CODE_INTERNAL_SERVER_ERROR</td>
     * <td>10</td>
     * <td>Internal server error.</td>
     * </tr>
     * </table>
     * <h3>SurveyMonkeySDK_ClientErrorC</h3>
     * <table>
     * <tr>
     * <th>Error</th>
     * <th>Code</th>
     * <th>Description</th>
     * </tr>
     * <tr>
     * <td>ERROR_CODE_USER_CANCELED</td>
     * <td>1</td>
     * <td>The user canceled out of the survey.</td>
     * </tr>
     * </table>
     */
    public SMError(String domain, ErrorType errorType, Exception exception, String description) {
        this.domain = domain;
        this.errorCode = errorType.getValue();
        this.description = description;
        this.exception = exception;
    }

    public static SMError sdkClientErrorFromCode(ErrorType errorType, Exception exception) {
        String description = "";
        switch (errorType) {
            case ERROR_CODE_USER_CANCELED:
                description = ErrorDescription.ERROR_CODE_RESPONDENT_EXITED_SURVEY_DESCRIPTION;
                break;
            default:
                break;
        }
        return new SMError(Domain.SDK_CLIENT_DOMAIN, errorType, exception, description);
    }

    public static SMError sdkServerErrorFromCode(ErrorType errorType, Exception exception) {
        String description = "";
        switch (errorType) {
            case ERROR_CODE_TOKEN:
                description = ErrorDescription.ERROR_CODE_TOKEN_DESCRIPTION;
                break;
            case ERROR_CODE_BAD_CONNECTION:
                description = ErrorDescription.ERROR_CODE_BAD_CONNECTION_DESCRIPTION;
                break;
            case ERROR_CODE_RESPONSE_PARSE_FAILED:
                description = ErrorDescription.ERROR_CODE_RESPONSE_PARSE_FAILED_DESCRIPTION;
                break;
            case ERROR_CODE_COLLECTOR_CLOSED:
                description = ErrorDescription.ERROR_CODE_COLLECTOR_CLOSED_DESCRIPTION;
                break;
            case ERROR_CODE_RETRIEVING_RESPONSE:
                description = ErrorDescription.ERROR_CODE_RETRIEVING_RESPONSE_DESCRIPTION;
                break;
            case ERROR_CODE_SURVEY_DELETED:
                description = ErrorDescription.ERROR_CODE_SURVEY_DELETED_DESCRIPTION;
                break;
            case ERROR_CODE_RESPONSE_LIMIT_HIT:
                description = ErrorDescription.ERROR_CODE_RESPONSE_LIMIT_HIT_DESCRIPTION;
                break;
            case ERROR_CODE_RESPONDENT_EXITED_SURVEY:
                description = ErrorDescription.ERROR_CODE_RESPONDENT_EXITED_SURVEY_DESCRIPTION;
                break;
            case ERROR_CODE_NONEXISTENT_LINK:
                description = ErrorDescription.ERROR_CODE_NONEXISTENT_LINK_DESCRIPTION;
                break;
            case ERROR_CODE_INTERNAL_SERVER_ERROR:
                description = ErrorDescription.ERROR_CODE_INTERNAL_SERVER_ERROR_DESCRIPTION;
                break;
            default:
                break;
        }
        return new SMError(Domain.SDK_SERVER_DOMAIN, errorType, exception, description);
    }

    public String getDomain() {
        return domain;
    }

    public String getDescription() {
        if (exception == null) {
            return "Domain: " + domain + " Code: " + errorCode + " SMDescription: " + description;
        }

        return "Domain: " + domain + " Code: " + errorCode + " Description: " + exception.getLocalizedMessage() + " SMDescription: " + description;
    }

    public int getErrorCode() {
        return errorCode;
    }

    public Exception getException() {
        return exception;
    }
}
