package com.segway.robot.algo.tf;

import android.os.Parcel;
import android.os.Parcelable;

/**
 * Created by Yusen.QIN on 2017/10/16.
 */

public class Quaternion implements Parcelable {
    public float x;
    public float y;
    public float z;
    public float w;

    protected Quaternion(Parcel in) {
        x = in.readFloat();
        y = in.readFloat();
        z = in.readFloat();
        w = in.readFloat();
    }

    public static final Creator<Quaternion> CREATOR = new Creator<Quaternion>() {
        @Override
        public Quaternion createFromParcel(Parcel in) {
            return new Quaternion(in);
        }

        @Override
        public Quaternion[] newArray(int size) {
            return new Quaternion[size];
        }
    };

    public Quaternion set (float x, float y, float z, float w) {
        this.x = x;
        this.y = y;
        this.z = z;
        this.w = w;
        return this;
    }

    public Quaternion (float x, float y, float z, float w) {
        this.set(x, y, z, w);
    }

    public Quaternion set (Quaternion quaternion) {
        return this.set(quaternion.x, quaternion.y, quaternion.z, quaternion.w);
    }

    /** Get the if there has any pole of the gimbal lock.
     * reference : https://en.wikipedia.org/wiki/Gimbal_lock
     * @return 1: north pole, -1:outh pole, 0:no gimbal lock */
    public int getGimbalPole () {
        final float t = y * x + z * w;
        if (t > 0.49999f){
            return 1;
        }
        if(t < -0.49999f){
            return -1;
        }
        return 0;
    }

    /** Get the roll euler angle in radians, which is the rotation around the z axis. Requires that this quaternion is normalized.
     * Get
     * @return the rotation around the z axis in radians (between -PI and +PI) */
    public float getRollRad () {
        final int pole = getGimbalPole();
        if (pole == 0){
            return MathUtils.atan2(2f * (w * z + y * x), 1f - 2f * (x * x + z * z));
        } else {
            return (float)pole * 2f* MathUtils.atan2(y, w);
        }
    }

    /** Get the pitch euler angle in radians, which is the rotation around the x axis. Requires that this quaternion is normalized.
     * @return the rotation around the x axis in radians (between -(PI/2) and +(PI/2)) */
    public float getPitchRad () {
        final int pole = getGimbalPole();
        return pole == 0 ? (float)Math.asin(MathUtils.clamp(2f * (w * x - z * y), -1f, 1f)) : (float)pole * MathUtils.PI * 0.5f;
    }

    /** Get the yaw euler angle in radians, which is the rotation around the y axis. Requires that this quaternion is normalized.
     * @return the rotation around the y axis in radians (between -PI and +PI) */
    public float getYawRad () {
        return getGimbalPole() == 0 ? MathUtils.atan2(2f * (y * w + x * z), 1f - 2f * (y * y + x * x)) : 0f;
    }

    /**
     * get sample with roll = getDegreeFromRadians(q.getRollRad())
     * @param degree
     * @return degree
     */
    public float getDegreeFromRadians(float degree){
        return degree * MathUtils.radiansToDegrees;
    }

    public Quaternion setEulerRad (float yaw, float pitch, float roll) { //XYZ axis
        final float halfRoll = roll * 0.5f;
        final float sHalfRoll = (float)Math.sin(halfRoll);
        final float cHalfRoll = (float)Math.cos(halfRoll);
        final float halfPitch = pitch * 0.5f;
        final float sHalfPitch = (float)Math.sin(halfPitch);
        final float cHalfPitch = (float)Math.cos(halfPitch);
        final float halfYaw = yaw * 0.5f;
        final float sHalfYaw = (float)Math.sin(halfYaw);
        final float cHalfYaw = (float)Math.cos(halfYaw);

        x = (sHalfRoll * cHalfPitch * cHalfYaw) - (cHalfRoll * sHalfPitch * sHalfYaw);
        y = (cHalfRoll * sHalfPitch * cHalfYaw) + (sHalfRoll * cHalfPitch * sHalfYaw);
        z = (cHalfRoll * cHalfPitch * sHalfYaw) - (sHalfRoll * sHalfPitch * cHalfYaw);
        w = (cHalfRoll * cHalfPitch *cHalfYaw) + (sHalfRoll * sHalfPitch * sHalfYaw);
        /* reference C++ CODE is from tf/include/linear_math/Quaternion.h line 75
        which is the same to the function setRPY(...)
		setValue(sinRoll * cosPitch * cosYaw - cosRoll * sinPitch * sinYaw, //x
                 cosRoll * sinPitch * cosYaw + sinRoll * cosPitch * sinYaw, //y
                 cosRoll * cosPitch * sinYaw - sinRoll * sinPitch * cosYaw, //z
                 cosRoll * cosPitch * cosYaw + sinRoll * sinPitch * sinYaw);
         */
        return this;
    }

    public Quaternion setEulerRadZYX(float yaw, float pitch, float roll){
        this.setEulerRad(roll,pitch,yaw);
        return this;
    }

    public Quaternion inv(){
        x = -x;
        y = -y;
        z = -z;
        return this;
    }

    public Quaternion mul(Quaternion q){
        final float resX = this.w * q.x + this.x * q.w + this.y * q.z - this.z * q.y;
        final float resY = this.w * q.y + this.y * q.w + this.z * q.x - this.x * q.z;
        final float resZ = this.w * q.z + this.z * q.w + this.x * q.y - this.y * q.x;
        final float resW = this.w * q.w - this.w * q.x - this.y * q.y - this.z * q.z;
        this.x = resX;
        this.y = resY;
        this.z = resZ;
        this.w = resW;
        return this;
    }

    public Quaternion mul(final float x, final float y, final float z, final float w){
        Quaternion q = new Quaternion(x, y, z, w);
        this.mul(q);
        return this;
    }

    public Quaternion mulVector3(Translation t){
        Quaternion q = new Quaternion(0,0,0,0);
        q.x = this.w * t.x + this.y * t.z - this.z * t.y;
        q.y = this.w * t.y + this.z * t.x - this.x * t.z;
        q.z = this.w * t.z + this.x * t.y - this.y * t.x;
        q.w = -this.x * t.x - this.y * t.y - this.z * t.z;
        return q;
    }

    public Quaternion leftMulVector3(Translation t){
        Quaternion q = new Quaternion(0,0,0,0);
        q.x = t.x * this.w + t.y * this.z - t.z * this.y;
        q.y = t.y * this.w + t.z * this.x - t.x * this.z;
        q.z = t.z * this.w + t.x * this.y - t.y * this.x;
        q.w = -t.x * this.x - t.y * this.y - t.z * this.z;
        return q;
    }

    public Translation quaternionRotate(Translation t){
        Quaternion qInv = this.inv();
        Quaternion q = new Quaternion(0,0,0,0);
        q = this.mulVector3(t);
        q = q.mul(qInv);
        Translation v = new Translation(q.x,q.y,q.z);
        return v;
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeFloat(x);
        dest.writeFloat(y);
        dest.writeFloat(z);
        dest.writeFloat(w);
    }

    @Override
    public String toString() {
        return "Quaternion{" +
                "x=" + x +
                ", y=" + y +
                ", z=" + z +
                ", w=" + w +
                '}';
    }

}
