package com.segway.robot.algo;

import android.os.Parcel;
import android.os.Parcelable;

import com.segway.robot.sdk.base.time.StampedData;

/**
 * Point2DF describe a point on the ground at a certain time
 */
public class Point2DF implements StampedData, Parcelable {
    protected long mTimeStamp;
    protected float mX;
    protected float mY;

    public Point2DF(float x, float y, long timeStamp) {
        mX = x;
        mY = y;
        mTimeStamp = timeStamp;
    }

    protected Point2DF(Parcel in) {
        mTimeStamp = in.readLong();
        mX = in.readFloat();
        mY = in.readFloat();
    }

    /**
     * Get the X coordinate of the point
     * @return X coordinate of the point
     */
    public float getX() {
        return mX;
    }

    /**
     * Get the Y coordinate of the point
     * @return Y coordinate of the point
     */
    public float getY() {
        return mY;
    }

    /**
     * Get the timestamp of the point
     * @return timestamp of the point
     */
    @Override
    public long getTimestamp() {
        return mTimeStamp;
    }

    /**
     * Get the distance from this point to target point
     * @param targetPoint target point
     * @return the distance from this point to target point
     */
    public float distanceTo(Point2DF targetPoint) {
        return (float) Math.sqrt(Math.pow(getX() - targetPoint.getX(), 2) + Math.pow(getY() - targetPoint.getY(), 2));
    }

    @Override
    public long diff(StampedData source) {
        if (source == null) {
            throw new IllegalArgumentException("diff source is null");
        }
        return source.getTimestamp() - getTimestamp();
    }


    public static final Creator<Point2DF> CREATOR = new Creator<Point2DF>() {
        @Override
        public Point2DF createFromParcel(Parcel in) {
            return new Point2DF(in);
        }

        @Override
        public Point2DF[] newArray(int size) {
            return new Point2DF[size];
        }
    };

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeLong(mTimeStamp);
        dest.writeFloat(mX);
        dest.writeFloat(mY);
    }
}
