package com.samebug.notifier;

import java.lang.Thread.UncaughtExceptionHandler;
import java.util.Date;
import java.util.UUID;

import com.samebug.notifier.exceptions.BadAppKey;
import com.samebug.notifier.exceptions.BadServerAddress;
import com.samebug.notifier.exceptions.MultipleConfigFileException;
import com.samebug.notifier.exceptions.NoConfigFileException;
import com.samebug.notifier.exceptions.NotifierException;
import com.samebug.notifier.proxy.ThrowableProxy;

/**
 * Static facilities of Samebug.
 * 
 * @author poroszd
 * 
 */
public class Samebug {
    private static boolean initialized = false;
    private static INotifier notifier;

    /**
     * Install a default exception handler on the current thread.
     * <p>
     * In case of any error (properties file not found, multiple file found,
     * wrong format of properties, missing obligatory entries) the constructed
     * notifier will not function as intended.
     */
    public static void handleUncaughtExceptions() throws NoConfigFileException, MultipleConfigFileException, BadAppKey, BadServerAddress {
        SamebugUncaughtExceptionHandler handler = new SamebugUncaughtExceptionHandler();
        installHandler(handler);
    }

    /**
     * Install a default exception handler on the current thread, with the given
     * application key.
     */
    public static void handleUncaughtExceptions(final String applicationKey) throws BadAppKey {
        SamebugUncaughtExceptionHandler handler = new SamebugUncaughtExceptionHandler(applicationKey);
        installHandler(handler);
    }

    /**
     * Install a default exception handler on the current thread, with the given
     * config
     */
    public static void handleUncaughtExceptions(final Configuration config) throws BadAppKey, BadServerAddress {
        SamebugUncaughtExceptionHandler handler = new SamebugUncaughtExceptionHandler(config);
        installHandler(handler);
    }

    private static void installHandler(final UncaughtExceptionHandler handler) {
        Thread.setDefaultUncaughtExceptionHandler(handler);
    }

    /**
     * Initialize a singleton notifier instance.
     * <p>
     * Constructs a singleton Samebug Notifier instance using the default
     * configuration.
     * <p>
     * You only need to use this initialization method if you want to use
     * {Samebug{@link #notify(String, Throwable)} explicitly. Only the first
     * call has effect, every consecutive call is practically ignored.
     * <p>
     * Using explicit notifications is discouraged, one should rather use
     * uncaught exception handler or logger appenders.
     */
    public static synchronized void init() throws NoConfigFileException, MultipleConfigFileException, BadAppKey, BadServerAddress {
        if (!initialized) {
            notifier = new SamebugNotifier();
            initialized = true;
        }
    }

    /**
     * Initialize a singleton notifier instance.
     * <p>
     * Constructs a singleton Samebug Notifier instance with the given
     * application key.
     * <p>
     * You only need to use this initialization method if you want to use
     * {Samebug{@link #notify(String, Throwable)} explicitly. Only the first
     * call has effect, every consecutive call is practically ignored.
     * <p>
     * Using explicit notifications is discouraged, one should rather use
     * uncaught exception handler or logger appenders.
     */
    public static synchronized void init(final String applicationKey) throws BadAppKey {
        if (!initialized) {
            notifier = new SamebugNotifier(applicationKey);
            initialized = true;
        }
    }

    /**
     * Initialize a singleton notifier instance.
     * <p>
     * Constructs a singleton Samebug Notifier instance with the given config.
     * <p>
     * You only need to use this initialization method if you want to use
     * {Samebug{@link #notify(String, Throwable)} explicitly. Only the first
     * call has effect, every consecutive call is practically ignored.
     * <p>
     * Using explicit notifications is discouraged, one should rather use
     * uncaught exception handler or logger appenders.
     */
    public static synchronized void init(final Configuration config) throws BadAppKey, BadServerAddress {
        if (!initialized) {
            notifier = new SamebugNotifier(config);
            initialized = true;
        }
    }

    /**
     * Notify Samebug about an exception
     * 
     * @param message
     *            custom message describing the event
     * @param throwable
     *            the caught exception
     * @return A UUID of the bug where this report was categorized
     * @throws NotifierException
     */
    public static UUID notify(final String message, final Throwable throwable) throws NotifierException {
        if (!initialized) {
            throw new IllegalStateException("Samebug not initialized");
        } else {
            return notifier.notify(message, throwable);
        }
    }

    /**
     * Notify Samebug about an exception
     * 
     * @param message
     *            custom message describing the event
     * @param throwable
     *            the caught exception
     * @param timestamp
     *            the time of occurrence
     * @return A UUID of the bug where this report was categorized
     * @throws NotifierException
     */
    public static UUID notify(final String message, final Throwable throwable, final Date timestamp) throws NotifierException {
        if (!initialized) {
            throw new IllegalStateException("Samebug not initialized");
        } else {
            return notifier.notify(message, throwable, timestamp);
        }
    }

    /**
     * Notify Samebug about an exception
     * 
     * @param message
     *            custom message describing the event
     * @param throwableProxy
     *            proxy for the caught exception
     * @return A UUID of the bug where this report was categorized
     * @throws NotifierException
     */
    public static UUID notify(final String message, final ThrowableProxy throwableProxy) throws NotifierException {
        if (!initialized) {
            throw new IllegalStateException("Samebug not initialized");
        } else {
            return notifier.notify(message, throwableProxy);
        }
    }

    /**
     * Notify Samebug about an exception
     * 
     * @param message
     *            custom message describing the event
     * @param throwableProxy
     *            proxy for the caught exception
     * @param timestamp
     *            the time of occurrence
     * @return A UUID of the bug where this report was categorized
     * @throws NotifierException
     */
    public static UUID notify(final String message, final ThrowableProxy throwableProxy, final Date timestamp) throws NotifierException {
        if (!initialized) {
            throw new IllegalStateException("Samebug not initialized");
        } else {
            return notifier.notify(message, throwableProxy, timestamp);
        }
    }
}
