package com.samebug.notifier;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.net.SocketTimeoutException;
import java.net.URLConnection;
import java.util.UUID;

import com.samebug.notifier.exceptions.AccountExpired;
import com.samebug.notifier.exceptions.AccountLimitExceeded;
import com.samebug.notifier.exceptions.BadRequest;
import com.samebug.notifier.exceptions.ConnectionError;
import com.samebug.notifier.exceptions.ConnectionTimeout;
import com.samebug.notifier.exceptions.NoResponse;
import com.samebug.notifier.exceptions.NoURI;
import com.samebug.notifier.exceptions.ReadError;
import com.samebug.notifier.exceptions.RecorderError;
import com.samebug.notifier.exceptions.RequestTooLarge;
import com.samebug.notifier.exceptions.UnknownProtocol;

/**
 * This class is responsible for managing http connection between the samebug
 * client and the server.
 * <p>
 * The connection settings are specified in the configuration.
 */
class Connection {
    private static final String USER_AGENT = "Samebug-Notifier-Java/1.0.0";

    private final Configuration config;

    public Connection(final Configuration config) {
        this.config = config;
    }

    /**
     * Sets up and opens an http connection to the samebug server.
     * <p>
     * The client is allowed only to get the output stream of the connection,
     * and write the serialized crash. Afterwards he must call
     * {@link processResponse} on the returned connection.
     * 
     * @return an http connection that conforms the settings specified in the
     *         configuration.
     */
    public HttpURLConnection createConnection() {
        try {
            URLConnection urlConnection = config.getServerURL().openConnection();
            if (urlConnection == null || !(urlConnection instanceof HttpURLConnection)) { throw new UnknownProtocol("Cannot connect to " + config.getServerURL()); }
            HttpURLConnection connection = (HttpURLConnection) urlConnection;
            connection.setReadTimeout(10000);
            connection.setConnectTimeout(5000);
            connection.setRequestProperty("Content-Type", "application/json;charset=utf-8");
            connection.setRequestProperty("charset", "utf-8");
            connection.setRequestMethod("POST");
            connection.setRequestProperty("Accept", "text/plain");
            connection.setRequestProperty("User-Agent", USER_AGENT);
            connection.setRequestProperty("X-Samebug-Key", config.getAppKey().toString());
            connection.setDoOutput(true);
            connection.connect();
            return connection;
        } catch (final SocketTimeoutException e) {
            throw new ConnectionTimeout("Cannot connect to " + config.getServerURL(), e);
        } catch (final IOException e) {
            throw new ConnectionError("IO error while opening connection to " + config.getServerURL(), e);
        }
    }

    /**
     * Processes the response on the http connection, and close it.
     * 
     * @param conn
     *            an http connection created via {@link createConnection}.
     * @return the UUID of the bug where the posted crash is categorized.
     */
    public UUID processResponse(final HttpURLConnection conn) throws RecorderError, NoResponse, NoURI, ReadError {
        // Get response code
        try {
            final int rc = conn.getResponseCode();
            final String rm = conn.getResponseMessage();
            switch (rc) {
            case 200:
                BufferedReader rd = null;
                try {
                    rd = new BufferedReader(new InputStreamReader(conn.getInputStream()));
                    String line;
                    if ((line = rd.readLine()) != null) { return UUID.fromString(line); }
                    throw new NoURI("No report URI.");
                } catch (final SocketTimeoutException e) {
                    throw new ReadError("Timeout read from " + config.getServerURL(), e);
                } catch (final IOException e) {
                    throw new ReadError("Unable to read response from " + config.getServerURL(), e);
                } finally {
                    if (rd != null) {
                        try {
                            rd.close();
                        } catch (final IOException e) {
                            if (config.getDebug()) {
                                System.err.println("Samebug: Error while closing connection stream:");
                                e.printStackTrace(System.err);
                            }
                        }
                    }
                }
            case 400:
                throw new BadRequest("Samebug rejected your request. Are you sure your application key is " + config.getAppKey() + " ?");
            case 403:
                throw new AccountExpired("You are not authorized to send notifications here. Probably your account is expired!");
            case 413:
                throw new RequestTooLarge("Samebug rejected your request. The sent crash log is too large.");
            case 429:
                throw new AccountLimitExceeded("Samebug rejected your request. You have sent too many requests from this account. You should enhance your account.");
            case 500:
                throw new RecorderError(rc, rm);
            default:
                throw new RecorderError(rc, "Unexpected response:\n" + rm);
            }
        } catch (final SocketTimeoutException e) {
            throw new ConnectionTimeout("Timeout while waiting response from " + config.getServerURL(), e);
        } catch (final IOException e) {
            throw new NoResponse("Unable to process response from " + config.getServerURL(), e);
        }
    }
}
