package com.pushpole.sdk.task.tasks;

import android.app.job.JobParameters;
import android.app.job.JobService;
import android.content.Context;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.net.TrafficStats;
import android.net.wifi.WifiInfo;
import android.os.Build;

import androidx.annotation.RequiresApi;

import com.pushpole.sdk.Constants;
import com.pushpole.sdk.PlainConstants;
import com.pushpole.sdk.internal.db.KeyStore;
import com.pushpole.sdk.internal.log.LogData;
import com.pushpole.sdk.internal.log.Logger;
import com.pushpole.sdk.network.SendManager;
import com.pushpole.sdk.util.NetworkHelper;
import com.pushpole.sdk.util.Pack;

import static com.pushpole.sdk.receiver.ConnectivityReceiver.getPackageUid;

/**
 * Created on 2017-12-26, 9:49 AM.
 *
 * @author Akram Shokri
 */

@RequiresApi(api = Build.VERSION_CODES.LOLLIPOP)
public class NetworkConnect extends JobService {

    @Override
    public boolean onStartJob(JobParameters jobParameters) {
        Logger.debug("Network Connect job called ...");
        try {
            usageProcess();
        } catch (Exception e) {
            Logger.error("Problem interacting with connectivity service", new LogData(
                    "Message", e.getMessage()
            ));
        }
        return true;
    }

    @Override
    public boolean onStopJob(JobParameters jobParameters) {
        return true;
    }

    private void usageProcess() {
        Context context = getApplicationContext();
        final ConnectivityManager connectivityManager = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
        NetworkInfo ni;
        try {
            ni = connectivityManager.getActiveNetworkInfo();
        } catch (NullPointerException e) {
            ni = null;
        }

        String wasConnectedTo = null;
        String connectedTo = null;

        //save prev-connection info
        long deviceRx = TrafficStats.getTotalRxBytes();
        long deviceTx = TrafficStats.getTotalTxBytes();
        long mobileRx = TrafficStats.getMobileRxBytes();
        long mobileTx = TrafficStats.getMobileTxBytes();
        long appRx = TrafficStats.getUidRxBytes(getPackageUid(context, context.getApplicationContext().getPackageName()));
        long appTx = TrafficStats.getUidTxBytes(getPackageUid(context, context.getApplicationContext().getPackageName()));
        long discTimestamp = System.currentTimeMillis();

        if (ni != null && ni.isConnectedOrConnecting()) {
            wasConnectedTo = KeyStore.getInstance(context).getString(Constants.getVal(Constants.KEY_STORE_CONNECTED_NETWORK_TYPE), null);
            if (!Constants.getVal(Constants.WIFI).equals(wasConnectedTo))
                wasConnectedTo = KeyStore.getInstance(context).getString(Constants.getVal(Constants.KEY_STORE_CONNECTED_NETWORK_SUBTYPE), null);

            String typeName = ni.getTypeName(), subtypeName = ni.getSubtypeName();
            if (typeName != null) {
                connectedTo = ni.getTypeName().toLowerCase();
            }
            if (ni.getType() == 0 && subtypeName != null) {
                connectedTo = ni.getSubtypeName().toLowerCase();
            }
            if (connectedTo == null /*&& connectedTo.equals(wasConnectedTo)*/)
                return;

            //connected to a new network
            handleLastConnectionInfo(context, deviceRx, deviceTx, appRx, appTx, mobileRx, mobileTx, discTimestamp, ni);

            //below call must be after 'handleLastConnectionInfo', because it overrides values needed in above code
            saveCurrentlyConnectedNetInfoToKeystore(context, ni, deviceRx, appRx,
                    appTx, deviceTx, mobileRx, mobileTx, discTimestamp); //save currently connected net info
        } else /*if (intent.getBooleanExtra(ConnectivityManager.EXTRA_NO_CONNECTIVITY, Boolean.FALSE))*/ {
            handleLastConnectionInfo(context, deviceRx, deviceTx, appRx, appTx, mobileRx, mobileTx, discTimestamp, null);
            deleteConnectedNetInfoToKeystore(context);
        }
    }

    private void handleLastConnectionInfo(Context context,
                                          long deviceRx, long deviceTx, long appRx, long appTx,
                                          long mobileRx, long mobileTx, long discTimestamp,
                                          NetworkInfo ni) {
        long wasConnectedTimestamp = KeyStore.getInstance(context).getLong(Constants.getVal(Constants.KEY_STORE_CONNECTED_NETWORK_TIMESTAMP), 0);

        long appRxStart = KeyStore.getInstance(context).getLong(Constants.getVal(Constants.KEY_STORE_CONNECTED_NETWORK_APP_RX_BYTE), 0);
        long appTxStart = KeyStore.getInstance(context).getLong(Constants.getVal(Constants.KEY_STORE_CONNECTED_NETWORK_APP_TX_BYTE), 0);
        long deviceTxStart = KeyStore.getInstance(context).getLong(Constants.getVal(Constants.KEY_STORE_CONNECTED_NETWORK_TOTAL_TX_BYTE), 0);
        long deviceRxStart = KeyStore.getInstance(context).getLong(Constants.getVal(Constants.KEY_STORE_CONNECTED_NETWORK_TOTAL_RX_BYTE), 0);
        long mobileRxStart = KeyStore.getInstance(context).getLong(Constants.getVal(Constants.KEY_STORE_CONNECTED_NETWORK_MOBILE_RX), 0);
        long mobileTxStart = KeyStore.getInstance(context).getLong(Constants.getVal(Constants.KEY_STORE_CONNECTED_NETWORK_MOBILE_TX), 0);

        String networkType;
        String networkSubtype = "";
        String networkExtra = null;
        boolean roaming = false;

        if (ni == null || ni.getTypeName() == null) {
            networkType = KeyStore.getInstance(context).getString(Constants.getVal(Constants.KEY_STORE_CONNECTED_NETWORK_TYPE), null);
            if (networkType == null)
                return; // we have no data of previously connected network

            networkSubtype = KeyStore.getInstance(context).getString(Constants.getVal(Constants.KEY_STORE_CONNECTED_NETWORK_SUBTYPE), null);
            networkExtra = KeyStore.getInstance(context).getString(Constants.getVal(Constants.KEY_STORE_CONNECTED_NETWORK_EXTRA), null);
            roaming = KeyStore.getInstance(context).getBoolean(Constants.getVal(Constants.KEY_STORE_CONNECTED_NETWORK_ROAMING), false);
        } else {
            networkType = ni.getTypeName().toLowerCase();
            if (ni.getTypeName() != null && !Constants.getVal(Constants.WIFI).equals(ni.getTypeName().toLowerCase())) { //its mobile network
                networkSubtype = ni.getSubtypeName().toLowerCase();
                networkExtra = ni.getExtraInfo().toLowerCase();
                roaming = ni.isRoaming();
            } else if (ni.getTypeName() != null && Constants.getVal(Constants.WIFI).equals(ni.getTypeName().toLowerCase())) { //its wifi network
                WifiInfo wifiInfo = NetworkHelper.WifiNetwork.connectedNetworkStat(context);
                if (wifiInfo != null) {
                    networkExtra = wifiInfo.getBSSID();//macAddress of connected network
                }
            }
        }

        Pack pack = new Pack();
        pack.putString(Constants.getVal(Constants.CONNECTION_INFO_NET_TYPE), networkType);
        if (networkExtra != null)
            pack.putString(Constants.getVal(Constants.CONNECTION_INFO_EXTRA), networkExtra);
        if (!Constants.getVal(Constants.WIFI).equals(networkType.toLowerCase())) {
            pack.putBool(Constants.getVal(Constants.CONNECTION_INFO_ROAMING), roaming);
            pack.putString(Constants.getVal(Constants.CONNECTION_INFO_NETWORK), networkSubtype);
        }
        if (mobileRx - mobileRxStart > 0)
            pack.putLong(Constants.getVal(Constants.CONNECTION_INFO_DATA_MOBILE_RX), (mobileRx - mobileRxStart) / 1000);
        if (mobileTx - mobileTxStart > 0)
            pack.putLong(Constants.getVal(Constants.CONNECTION_INFO_DATA_MOBILE_TX), (mobileTx - mobileTxStart) / 1000);
        if ((deviceRx - deviceRxStart) > 0) //total usage minus mobile usage is equal to wifi usage
            pack.putLong(Constants.getVal(Constants.CONNECTION_INFO_DATA_WIFI_RX), ((deviceRx - deviceRxStart) - (mobileRx - mobileRxStart)) / 1000);
        if ((deviceTx - deviceTxStart) > 0)
            pack.putLong(Constants.getVal(Constants.CONNECTION_INFO_DATA_WIFI_TX), ((deviceTx - deviceTxStart) - (mobileTx - mobileTxStart)) / 1000);

        pack.putLong(Constants.getVal(Constants.CONNECTION_INFO_CONNECT_AT), wasConnectedTimestamp);
        pack.putLong(Constants.getVal(Constants.CONNECTION_INFO_DISCONNECT_AT), discTimestamp);
        //threshold is an int, specifying minimum net usage (in KB) which we consider for reporting
        int threshold = KeyStore.getInstance(context).getInt(Constants.getVal(Constants.KEY_STORE_CONNECTED_NETWORK_THRESHOLD),
                PlainConstants.CONNECTION_INFO_DEFAULT_THRESHOLD);

        if (((deviceRx - deviceRxStart) + (deviceTx - deviceTxStart) / 1000) < threshold) {
            Logger.debug("Network connectivity usage was less than %dKB and dropped", threshold);
            return;
        }
        if ((appRx - appRxStart) > 0)
            pack.putLong(Constants.getVal(Constants.CONNECTION_INFO_APP_DATA_RX), (appRx - appRxStart) / 1000);
        if ((appTx - appTxStart) > 0)
            pack.putLong(Constants.getVal(Constants.CONNECTION_INFO_APP_DATA_TX), (appTx - appTxStart) / 1000);

        SendManager.getInstance(context).send(Constants.getVal(Constants.CONNECTIVITY_INFO_T), pack);
    }

    private void saveCurrentlyConnectedNetInfoToKeystore(Context context, NetworkInfo ni, long deviceRx, long appRx, long appTx, long deviceTx, long mobileRx, long mobileTx, long timestamp) {
        String networkTypeName = ni.getTypeName(), subtypeName = ni.getSubtypeName(), extraInfo = ni.getExtraInfo();
        KeyStore.getInstance(context).putLong(Constants.getVal(Constants.KEY_STORE_CONNECTED_NETWORK_APP_RX_BYTE), appRx);
        KeyStore.getInstance(context).putLong(Constants.getVal(Constants.KEY_STORE_CONNECTED_NETWORK_APP_TX_BYTE), appTx);
        KeyStore.getInstance(context).putLong(Constants.getVal(Constants.KEY_STORE_CONNECTED_NETWORK_TOTAL_TX_BYTE), deviceTx);
        KeyStore.getInstance(context).putLong(Constants.getVal(Constants.KEY_STORE_CONNECTED_NETWORK_TOTAL_RX_BYTE), deviceRx);
        KeyStore.getInstance(context).putString(Constants.getVal(Constants.KEY_STORE_CONNECTED_NETWORK_TYPE), networkTypeName == null ? null : networkTypeName.toLowerCase());
        if (networkTypeName != null && !Constants.getVal(Constants.WIFI).equals(networkTypeName.toLowerCase())) { //its mobile network
            KeyStore.getInstance(context).putString(Constants.getVal(Constants.KEY_STORE_CONNECTED_NETWORK_SUBTYPE), subtypeName == null ? null : subtypeName.toLowerCase());
            KeyStore.getInstance(context).putString(Constants.getVal(Constants.KEY_STORE_CONNECTED_NETWORK_EXTRA), extraInfo == null ? null : extraInfo.toLowerCase());
            KeyStore.getInstance(context).putBoolean(Constants.getVal(Constants.KEY_STORE_CONNECTED_NETWORK_ROAMING), ni.isRoaming());
        } else if (networkTypeName != null && Constants.getVal(Constants.WIFI).equals(networkTypeName.toLowerCase())) { //its wifi network
            WifiInfo wifiInfo = NetworkHelper.WifiNetwork.connectedNetworkStat(context);
            if (wifiInfo != null) {
                KeyStore.getInstance(context).putString(Constants.getVal(Constants.KEY_STORE_CONNECTED_NETWORK_EXTRA), wifiInfo.getBSSID());//macAddress of connected network
            }
        }
        KeyStore.getInstance(context).putLong(Constants.getVal(Constants.KEY_STORE_CONNECTED_NETWORK_MOBILE_TX), mobileTx);
        KeyStore.getInstance(context).putLong(Constants.getVal(Constants.KEY_STORE_CONNECTED_NETWORK_MOBILE_RX), mobileRx);

        KeyStore.getInstance(context).putLong(Constants.getVal(Constants.KEY_STORE_CONNECTED_NETWORK_TIMESTAMP), timestamp);
    }

    private void deleteConnectedNetInfoToKeystore(Context context) {
        KeyStore.getInstance(context).delete(Constants.getVal(Constants.KEY_STORE_CONNECTED_NETWORK_TYPE));
        KeyStore.getInstance(context).delete(Constants.getVal(Constants.KEY_STORE_CONNECTED_NETWORK_SUBTYPE));
        KeyStore.getInstance(context).delete(Constants.getVal(Constants.KEY_STORE_CONNECTED_NETWORK_ROAMING));
        KeyStore.getInstance(context).delete(Constants.getVal(Constants.KEY_STORE_CONNECTED_NETWORK_EXTRA));
        KeyStore.getInstance(context).delete(Constants.getVal(Constants.KEY_STORE_CONNECTED_NETWORK_TIMESTAMP));
    }
}
