package com.pushpole.sdk.controller.controllers;

import android.app.Notification;
import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.content.Context;
import android.content.Intent;
import android.media.AudioManager;
import android.media.MediaPlayer;
import android.os.AsyncTask;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;

import android.util.DisplayMetrics;
import android.util.Log;
import android.util.Patterns;

import androidx.annotation.RequiresApi;
import androidx.core.app.NotificationManagerCompat;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

import com.pushpole.sdk.Constants;
import com.pushpole.sdk.NotificationButtonData;
import com.pushpole.sdk.PlainConstants;
import com.pushpole.sdk.PushPole;
import com.pushpole.sdk.NotificationData;
import com.pushpole.sdk.controller.DownstreamApiController;
import com.pushpole.sdk.internal.db.KeyStore;
import com.pushpole.sdk.internal.db.NotifAndUpstreamMsgsDbOperation;
import com.pushpole.sdk.internal.exceptions.NotificationBuildFailed;
import com.pushpole.sdk.internal.log.LogData;
import com.pushpole.sdk.internal.log.Logger;
import com.pushpole.sdk.message.MessageStore;
import com.pushpole.sdk.message.downstream.DownstreamMessage;
import com.pushpole.sdk.message.downstream.NotificationDownstreamMessage;
import com.pushpole.sdk.task.TaskManager;
import com.pushpole.sdk.task.NotificationBuildTask;
import com.pushpole.sdk.util.IdGenerator;
import com.pushpole.sdk.util.NotificationBuilder;
import com.pushpole.sdk.util.Pack;
import com.pushpole.sdk.util.PackBundler;
import com.pushpole.sdk.util.ScreenWaker;

import static com.pushpole.sdk.PlainConstants.*;
import static com.pushpole.sdk.util.Utility.isValidWebUrl;


/***
 * a class for handling {@link NotificationDownstreamMessage}
 * show notification if possible
 * deliver message content and customized json to client through {@link com.pushpole.sdk.PushPoleListenerService}
 */
public class NotificationController implements DownstreamApiController {
    private Context mContext;
    private MediaPlayer mMediaPlayer;

    public NotificationController(Context context) {
        mContext = context;
    }


    /***
     * handler for notification downstream message
     * try to show notification, if failed then reschedule it to show in future
     *
     * @param downstreamMessage
     */
    @Override
    public void handleDownstreamMessage(DownstreamMessage downstreamMessage) {
        if (!DownstreamMessage.Type.NOTIFICATION.equals(downstreamMessage.getMessageType()) &&
                !Constants.getVal(Constants.NOTIF_NEW_CODE_T).equals(downstreamMessage.getMessageType())) {
            return;
        }

        NotificationDownstreamMessage message = (NotificationDownstreamMessage) downstreamMessage;

        //handle welcome message notification
        if (message.isWelcomeMsg() && !message.showWelcomeNotificationMsg(mContext))
            return;

        //handle update app notification message
        if(message.isUpdateAppNotification() && ! message.showUpdateAppNotification(mContext))
            return;
        if(message.getIconUrl() != null)
            message.setIconUrl(changeIcon(message.getIconUrl(), mContext));
        boolean isNotificationOff = KeyStore.getInstance(mContext).getBoolean(Constants.getVal(Constants.NOTIFICATION_OFF), false);//user of lib can set notification off

        if (message.getShowNotification() && message.getTitle() != null && (!isNotificationOff || message.isForcePublish())) {
            try {
                showNotificationMessage(message);

            } catch (NotificationBuildFailed e) {
                // If failed serialize message into KeyStore Pack and Start NotificationBuild Task
                MessageStore.getInstance().storeMessage(mContext, message);
                Pack taskData = new Pack();
                taskData.putString(Constants.getVal(Constants.MESSAGE_ID), message.getMessageId());
                taskData.putString(Constants.getVal(Constants.MESSAGE_IMAGE_RETRY), "true");
                Logger.warning("First attempt at loading notification failed, scheduling task");
                TaskManager.getInstance(mContext).scheduleTask(NotificationBuildTask.class, taskData);
            }
        }else{
            if(isNotificationOff && message.getShowNotification() && message.getTitle() != null)
                NotificationBuildTask.sendNotifPublishStatus(mContext, message.getMessageId(), NotificationBuildTask.NOTIF_DISABLED);
        }
        deliverMessage(message);
    }

    /***
     * show notification to client
     *
     * @param message the downstream message
     * @throws NotificationBuildFailed
     */
    public void showNotificationMessage(NotificationDownstreamMessage message) throws NotificationBuildFailed {
        //check if msgId does not exist in notifTable
        NotifAndUpstreamMsgsDbOperation notifAndUpstreamMsgsDbOperation = NotifAndUpstreamMsgsDbOperation.getInstance(mContext);
        notifAndUpstreamMsgsDbOperation.open();
        if (!notifAndUpstreamMsgsDbOperation.isMsgExists(message.getMessageId())) {
            notifAndUpstreamMsgsDbOperation.insertNotifMsg(message.getMessageId(), message.getMessageType().getTypeCode());
        } else {
            if(!message.isUpdateAppNotification() && !message.isRetryImgFetch()) { //update message are shown more than once
                Logger.debug("Ignoring duplicate notification with MessageID = " + message.getMessageId());
                return;//message is received before, no need to show it again.
            }
        }

        int notificationId = IdGenerator.generateIntegerId();
        NotificationBuilder builder = NotificationBuilder.getBuilder(mContext, message, notificationId);
        Notification notification = builder.build();

        if (isValidWebUrl(message.getSoundUrl())) {//it play the sound and shows the notification
            new PlaySoundNotifAsyncTask().execute(message.getSoundUrl(), notificationId, notification);
        } else {
            NotificationManager notificationManager =
                    (NotificationManager) mContext.getSystemService(Context.NOTIFICATION_SERVICE);
            notificationManager.notify(notificationId, notification);
        }
        if (message.getWakeScreen()) {
            ScreenWaker.wakeScreen(mContext);
        }
         //server upstream to tell if notification is published or is disabled
        if(NotificationManagerCompat.from(mContext).areNotificationsEnabled()) {//works for API 19+ on below 19 returns true
            NotificationBuildTask.sendNotifPublishStatus(mContext, message.getMessageId(), NotificationBuildTask.NOTIF_PUBLISHED);
        }else{
            NotificationBuildTask.sendNotifPublishStatus(mContext, message.getMessageId(), NotificationBuildTask.NOTIF_SYSTEM_DISABLE);
        }

    }

    @RequiresApi(Build.VERSION_CODES.O)
    private void createSilentNotificationChannel() {
        NotificationManager notificationManager = (NotificationManager) mContext.getSystemService(Context.NOTIFICATION_SERVICE);

        if (notificationManager.getNotificationChannel(PlainConstants.DEFAULT_SILENT_CHANNEL_ID) == null) {
//            Plog.debug(T_NOTIF, "Creating default silent notification channel")
            NotificationChannel channel = new NotificationChannel(PlainConstants.DEFAULT_SILENT_CHANNEL_ID,
                    PlainConstants.DEFAULT_SILENT_CHANNEL_NAME, NotificationManager.IMPORTANCE_HIGH);

            channel.setSound(null, null);
            channel.enableLights(true);
            notificationManager.createNotificationChannel(channel);
        }
    }

    //----------------------------------------------------------------------------------------------

    /***
     * Deliver Message content and custom json to client through {@link com.pushpole.sdk.PushPoleListenerService}
     *
     * @param message the downstream message
     */

    public void deliverMessage(NotificationDownstreamMessage message) {
        Intent intent = new Intent(Constants.getVal(Constants.ACTION_PUSHPOLE_RECEIVE));
        intent.setPackage(mContext.getPackageName());

        if (message.getCustomContent() != null) {

            Logger.info("Invoking custom message handler", new LogData(
                    "Custom Message", message.getCustomContent().toJson()
            ));
            Bundle jsonBundle = PackBundler.packToBundle(message.getCustomContent());
            intent.putExtra("json", jsonBundle);
        }

        Pack messageContent = new Pack();

        messageContent.put(USER_TITLE, message.getTitle());
        messageContent.put(USER_CONTENT, message.getContent());
        messageContent.put(USER_BIG_TITLE, message.getBigTitle());
        messageContent.put(USER_BIG_CONTENT, message.getBigContent());
        messageContent.put(USER_SUMMARY, message.getSummary());
        messageContent.put(USER_IMG_URL, message.getImageUrl());
        messageContent.put(USER_ICON_URL, message.getIconUrl());

        messageContent.put(USER_TICKER, message.getTicker());

        Bundle messageBundle = PackBundler.packToBundle(messageContent);

        intent.putExtra("messageContent", messageBundle);

        mContext.startService(intent);

        // Notify user from the notification
        if (PushPole.pushpoleNotificationListener != null) {
            List<NotificationButtonData> buttons = new ArrayList<>();
            if (message.getButtons() != null) {
                for (NotificationDownstreamMessage.Button b : message.getButtons()) {
                    buttons.add(new NotificationButtonData(b.getText(), b.getId()));
                }
            }
            String customContent = null;
            if (message.getCustomContent() != null) {
                customContent = message.getCustomContent().toJson();
            }
            NotificationData notification = new NotificationData(
                    message.getTitle(),
                    message.getContent(),
                    message.getBigTitle(),
                    message.getBigContent(),
                    message.getSummary(),
                    message.getImageUrl(),
                    message.getIconUrl(),
                    customContent,
                    buttons
            );
            PushPole.pushpoleNotificationListener.onNotificationReceived(notification);

            // Notify customContent if exists.
            if (message.getCustomContent() != null) {
                try {
                    PushPole.pushpoleNotificationListener.onCustomContentReceived(new JSONObject(customContent));
                } catch (JSONException e) {
                    Log.e(PlainConstants.PUSHPOLE, "Failed to convert custom content to JSONObject.\n" + customContent, e);
                    Logger.error("Custom content exists, but failed to get it's Json object.", new LogData("Cause", e.getMessage()));
                }
            }
        }
    }


    //----------------------------------------------------------------------------------------------

    public class PlaySoundNotifAsyncTask extends AsyncTask implements MediaPlayer.OnPreparedListener {
        private String soundUrl;
        private int notifId;
        private Notification notif;

        void prepareAndPlaySound(String soundUrl) {
            killMediaPlayer();
            mMediaPlayer = new MediaPlayer();
            try {
                mMediaPlayer.setAudioStreamType(AudioManager.STREAM_NOTIFICATION);
                //mMediaPlayer.setAudioAttributes(Notification.DEFAULT_VIBRATE);
                mMediaPlayer.setDataSource(soundUrl);
                mMediaPlayer.prepareAsync();
                mMediaPlayer.setOnPreparedListener(this);
            } catch (Exception e) {
                Logger.warning("Error in setting notification's sound url", e);
            }
        }

        private void killMediaPlayer() {
            if (mMediaPlayer != null) {
                mMediaPlayer.release();
            }
        }

        @Override
        protected Object doInBackground(Object[] params) {
            soundUrl = (String) params[0];
            notifId = (int) params[1];
            notif = (Notification) params[2];
            prepareAndPlaySound(soundUrl);
            return null;
        }

        /**
         * called when custom notification's sound is ready to play
         *
         * @param mp
         */
        @Override
        public void onPrepared(MediaPlayer mp) {
            NotificationManager notificationManager =
                    (NotificationManager) mContext.getSystemService(Context.NOTIFICATION_SERVICE);

            //check whether sound should played or not according to phone current ring mode
            AudioManager am = (AudioManager) mContext.getSystemService(Context.AUDIO_SERVICE);
            int ringerMode = am.getRingerMode();
            if (ringerMode != AudioManager.RINGER_MODE_NORMAL && ringerMode == AudioManager.RINGER_MODE_VIBRATE) {//vibrate phone
                notif.defaults |= Notification.DEFAULT_VIBRATE;
            } else if (ringerMode == AudioManager.RINGER_MODE_NORMAL && ringerMode != AudioManager.RINGER_MODE_VIBRATE
                    && ringerMode == AudioManager.RINGER_MODE_SILENT) {//silent mode
                notificationManager.notify(notifId, notif); //just show notification and return
                return;
            }

            //in other cases, play sound, show notif and set postDelayed handler to stop sound from playing after a few seconds
            mp.start();
            notificationManager.notify(notifId, notif);

            Handler mHandler = new Handler(Looper.getMainLooper());
            mHandler.postDelayed(new Runnable() {
                public void run() {
                    try {
                        if (mMediaPlayer != null && mMediaPlayer.isPlaying())
                            mMediaPlayer.stop();
                    } catch (Exception e) {
                        Logger.error("Error in stopping media player of notification's sound", e);
                    }
                }
            }, Constants.F_SOUND_DURATION * 1000);
        }
    }

    private static String changeIcon(String iconUrl, Context context){
        DisplayMetrics metrics = context.getResources().getDisplayMetrics();
        String suffix = "";
        //below checks MUST be in this increasing order or it may faile
        if(metrics.densityDpi <= DisplayMetrics.DENSITY_MEDIUM)
            suffix = "-m";
        else if (metrics.densityDpi <= DisplayMetrics.DENSITY_HIGH)
            suffix=  "-h";
        else if(metrics.densityDpi <= DisplayMetrics.DENSITY_XHIGH)
            suffix = "-xh";
        else if(metrics.densityDpi <= DisplayMetrics.DENSITY_XXHIGH || metrics.densityDpi > DisplayMetrics.DENSITY_XXHIGH)
            suffix= "-xxh";

        String pasvand =  iconUrl.substring(iconUrl.lastIndexOf("."));
        String str = iconUrl.substring(0, iconUrl.lastIndexOf(".")).concat(suffix).concat(pasvand);
        Logger.debug("Notification Icon url for this device ", new LogData("density", String.valueOf(metrics.densityDpi), "Icon url", str));
        return str;
    }

}
