package com.pushpole.sdk.action.actions;

import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.net.Uri;
import android.os.Build;

import androidx.core.app.NotificationCompat;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.net.URL;
import java.net.URLConnection;

import com.pushpole.sdk.Constants;
import com.pushpole.sdk.R;
import com.pushpole.sdk.action.Action;
import com.pushpole.sdk.action.ActionFactory;
import com.pushpole.sdk.action.ActionType;
import com.pushpole.sdk.internal.log.Logger;
import com.pushpole.sdk.provider.PushPoleProvider;
import com.pushpole.sdk.util.IdGenerator;
import com.pushpole.sdk.util.Pack;

/**
 * Created on 16-08-24, 9:58 AM.
 *
 * @author Akram Shokri
 */
public class DownloadFileAction extends Action {

    private String downloadUrl;

    @Override
    public void execute(Context context) {
        Logger.info("Executing download a file action. Url: " + downloadUrl);
        new DownloadFile().downloadFile(context, downloadUrl);
    }

    @Override
    public ActionType getActionType() {
        return ActionType.DOWNLOAD;
    }


    @Override
    public Pack toPack() {
        Pack pack = super.toPack();
        pack.putString(Constants.getVal(Constants.DOWNLOAD_ACTION_URL), downloadUrl);
        return pack;
    }

    public static class Factory extends ActionFactory {
        @Override
        public Action buildAction(Pack data) {
            DownloadFileAction action = new DownloadFileAction();
            action.downloadUrl = data.getString(Constants.getVal(Constants.DOWNLOAD_ACTION_URL), "");
            return action;
        }
    }

    public static class DownloadFile {
        private int downloadNotifId;
        private NotificationManager notificationManager;

        public void downloadFile(Context context, String dlUrl) {
            String fileName = "temp.apk";
            File fileLocation = new File(context.getFilesDir().getAbsolutePath() + "/"+ fileName);
            try {
                if (fileLocation.exists()) {
                    fileLocation.delete();
                } else {
                    fileLocation.createNewFile();
                }

                try {
                    NotificationCompat.Builder builder = new NotificationCompat.Builder(context);
                    builder.setSmallIcon(R.drawable.ic_file_download);
                    builder.setContentTitle(context.getString(R.string.pushpole_starting_apk_download_title));
                    builder.setContentText(context.getString(R.string.pushpole_starting_apk_download));
                    PendingIntent pIntent = PendingIntent.getActivity(context, 0, new Intent(), 0);
                    builder.setContentIntent(pIntent);
                    builder.setAutoCancel(false);
                    builder.setProgress(100, 10, true);
                    builder.setOngoing(true);
                    //builder.addAction(R.drawable.ic_cancel, context.getString(R.string.cancel), pIntent);
                    notificationManager = (NotificationManager) context.getSystemService(context.NOTIFICATION_SERVICE);
                    downloadNotifId = IdGenerator.generateIntegerId();
                    notificationManager.notify(downloadNotifId, builder.build());
                } catch (Exception e) {
                    Logger.warning("Can't show download started notification. " + e.getLocalizedMessage());
                }

                URLConnection conn = new URL(dlUrl).openConnection();
                conn.setUseCaches(false);
                conn.connect();

                InputStream inputStream = new BufferedInputStream(conn.getInputStream());

                byte[] buffer = new byte[1024];
                int bufferLength = 0;
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                    FileOutputStream fos = context.openFileOutput(fileName, Context.MODE_PRIVATE);
                    while ((bufferLength = inputStream.read(buffer)) > 0) {
                        fos.write(buffer, 0, bufferLength);
                    }
                    fos.close();
                    notificationManager.cancel(downloadNotifId);
                    File fPath = new File(context.getFilesDir().getAbsolutePath());
                    File dlFile = new File(fPath, fileName);
                    Uri contentUri = PushPoleProvider.getUriForFile(context, context.getPackageName() + ".com.pushpole.sdk.pushpoleprovider", dlFile);
                    Intent intent = new Intent(Intent.ACTION_INSTALL_PACKAGE);
                    intent.setData(contentUri)
                            .addFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION);
                    context.startActivity(intent);
                } else {
                    FileOutputStream fos = context.openFileOutput(fileName, Context.MODE_WORLD_READABLE | Context.MODE_WORLD_WRITEABLE);
                    while ((bufferLength = inputStream.read(buffer)) > 0) {
                        fos.write(buffer, 0, bufferLength);
                    }
                    fos.close();
                    notificationManager.cancel(downloadNotifId);
                    Intent intent = new Intent(Intent.ACTION_VIEW);
                    intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                    intent.setDataAndType(Uri.fromFile(fileLocation),
                            "application/vnd.android.package-archive");
                    context.startActivity(intent);
                }
            } catch (Exception e) {
                Logger.warning("Error in executing download action", e);
                notificationManager.cancel(downloadNotifId);
                return;
            }
        }
    }
}
