package com.pushpole.sdk.controller.controllers;

import android.content.Context;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.support.annotation.NonNull;
import android.util.Log;

import com.google.android.gms.tasks.OnCompleteListener;
import com.google.android.gms.tasks.Task;
import com.google.android.gms.tasks.Tasks;
import com.google.firebase.iid.FirebaseInstanceId;
import com.google.firebase.iid.InstanceIdResult;

import java.util.concurrent.ExecutionException;

import com.pushpole.sdk.SenderInfo;
import com.pushpole.sdk.collection.CollectionManager;
import com.pushpole.sdk.collection.CollectionType;
import com.pushpole.sdk.controller.ResponseApiController;
import com.pushpole.sdk.internal.db.KeyStore;
import com.pushpole.sdk.internal.log.LogData;
import com.pushpole.sdk.internal.log.Logger;
import com.pushpole.sdk.message.Message;
import com.pushpole.sdk.message.ResponseMessage;
import com.pushpole.sdk.message.upstream.UpstreamMessage;
import com.pushpole.sdk.task.PushPoleAsyncTask;
import com.pushpole.sdk.task.TaskManager;
import com.pushpole.sdk.task.tasks.FcmRegisterTask;
import com.pushpole.sdk.task.tasks.FlushDBTask;
import com.pushpole.sdk.task.tasks.ServerRegisterTask;
import com.pushpole.sdk.topic.TopicSubscriber;

/**
 * Handles client app registration
 */
public class RegisterController implements ResponseApiController {
    private final static String APP_VERSION_KEY = "$latest_registered_version";

    private Context mContext;

    public RegisterController(Context context) {
        mContext = context;
    }

    public void register() {
        Logger.info("Scheduling register task");
        TaskManager.getInstance(mContext).scheduleTask(FcmRegisterTask.class);
    }

    public void invalidateRegistration() {
        Logger.info("Invalidating registration");
        SenderInfo.getInstance(mContext).setToken(null);
        SenderInfo.getInstance(mContext).setTokenState(SenderInfo.NO_TOKEN);
    }

    /**
     * Checks whether app needs to register or re-register.
     * <p/>
     * If this method detects package update, it invalidates registration token.
     *
     * @return true if no registration token exists, registration has not been synced with server or
     * if package has been updated since last registration sync.
     */
    public boolean needsRegistration() {
        // Register if not registered
        if (!SenderInfo.getInstance(mContext).isRegistrationComplete()) {
            return true;
        }
        return false;
    }

    /***
     * Update app version code
     */
    private void updateLatestRegisteredVersion() {
        PackageInfo packageInfo = null;
        try {
            PackageManager manager = mContext.getPackageManager();
            packageInfo = manager.getPackageInfo(mContext.getPackageName(), 0);
        } catch (PackageManager.NameNotFoundException e) {
            Logger.error("Retrieving application package info failed");
        }

        if (packageInfo != null) {
            KeyStore.getInstance(mContext).putInt(APP_VERSION_KEY, packageInfo.versionCode);
        }
    }

    private int getLatestRegisteredVersion() {
        return KeyStore.getInstance(mContext).getInt(APP_VERSION_KEY, 0);
    }


    /**
     * Is called when GCM Registration is complete and a token is obtained
     * If token is a fresh token, replace the old token and register again with the PushPole server
     */
    public void onGCMRegisterComplete(String token) {
        //subscribing to "broadcast" topic
        String oldToken = SenderInfo.getInstance(mContext).getToken();
        if(!token.equals(oldToken)) {
            subscribeToBroadcastTopic();
        }
        int tokenState = SenderInfo.getInstance(mContext).getTokenState();
        if (tokenState != SenderInfo.TOKEN_SYNCED || !token.equals(oldToken)) {//TODO: this if changed, check if it works fine
            SenderInfo.getInstance(mContext).setToken(token);
            SenderInfo.getInstance(mContext).setTokenState(SenderInfo.TOKEN_RECEIVED);
            TaskManager.getInstance(mContext).scheduleTask(ServerRegisterTask.class);
        }

        updateLatestRegisteredVersion();
    }

    /**
     * After successful registration to PushPole, subscribe this device to broadcast topic of this app
     * It is useful in case registration to PushPole failed, because registration of this device to
     * broadcast topic make it capable to receive push notifications that app-developer sends to all
     * its users.
     */
    public void subscribeToBroadcastTopic() {
        String broadcastTopic = "broadcast";
        TopicSubscriber topicSubscriber = new TopicSubscriber(mContext);
        topicSubscriber.subscribeToChannel(broadcastTopic);
    }

    /**
     * Called when register response is received
     * @param responseMessage the message
     */
    @Override
    public void handleUpstreamMessageResponse(ResponseMessage responseMessage) {
        if (!UpstreamMessage.Type.REGISTER.equals(responseMessage.getMessageType())) {
            //Logger.warning("processing register response: handleUpstreamMessageResponse()-type=" + responseMessage.getMessageType());
            return;
        }

        // Check is already registered to pushpole-server
        if (SenderInfo.getInstance(mContext).getTokenState() == SenderInfo.TOKEN_SYNCED) {
            //Logger.warning("processing register response: handleUpstreamMessageResponse() tokensatet="+SenderInfo.getInstance(mContext).getTokenState());
            return;
        }

        if (responseMessage.getStatus() == Message.STATUS_SUCCESS) {
            SenderInfo.getInstance(mContext).setTokenState(mContext, SenderInfo.TOKEN_SYNCED);
            Log.i("PushPole", "Successfully registered to pushpole");
            RunAllTasks();
            ScheduleAllTasks();
        }
        else if(responseMessage.getStatus() == Message.STATUS_FAILED){
            SenderInfo.getInstance(mContext).setTokenState(mContext, SenderInfo.TOKEN_RECEIVED);
            TaskManager.getInstance(mContext).scheduleTask(ServerRegisterTask.class);
        }
    }

    private void RunAllTasks() {
        // Run all tasks Once after successfully registered to pushpole server
        TaskManager.getInstance(mContext).asyncTask(new PushPoleAsyncTask() {
            @Override
            public void run(Context context) {

                for (CollectionType cType : CollectionType.values()) {
                    CollectionManager.getInstance().collectNow(context, cType);
                }
                //starting periodic task for sending collected data every 24h
                new FlushDBTask().runTask(mContext, null);

                //starting periodic task for broadcasting if this app is responsible for sending common device data
                /*PushPoleTask btask = (PushPoleTask) new BroadcastTask(); //TODO: app-exist broadcast disabled temporary
                btask.runTask(context, null);*/
            }
        });
    }

    private void ScheduleAllTasks() {
        //Schedule tasks after successfully registered to pushpole server to run periodically
        TaskManager.getInstance(mContext).asyncTask(new PushPoleAsyncTask() {
            @Override
            public void run(Context context) {

                for (CollectionType cType : CollectionType.values()) {
                    CollectionManager.getInstance().startCollection(context, cType);
                }
                //starting periodic task for sending collected data every 24h
                TaskManager.getInstance(context).scheduleTask(FlushDBTask.class);

                //starting periodic task for broadcasting if this app is responsible for sending common device data
                /*TaskManager.getInstance(context).scheduleTask(BroadcastTask.class);*/ //TODO: app-exist broadcast disabled temporary
            }
        });
    }


}
