package com.pushpole.sdk.action;

import android.content.Context;

import com.pushpole.sdk.Constants;
import com.pushpole.sdk.util.Pack;

/**
 * A Notification Action to be executed once a notification or a notification button is clicked.
 * <p>
 * In order to add a new Action you must extend this class along with all it's methods.
 * You must also create a {@link ActionFactory} which creates an instance of your action and
 * add a new Action Type in {@link ActionType} which links to your Factory class.
 * The general convention is for the Factory class to be a static inner class of your Action class.
 *
 * @author Hadi Zolfaghari
 */
public abstract class Action {
    /**
     * Executes the action.
     * <p>
     * Called once a notification or notification button with this action has been clicked.
     * <p>
     * Override this method to define the behaviour of your action.
     *
     * @param context the application context.
     */
    public abstract void execute(Context context);

    /**
     * Gets the {@link ActionType} of this action.
     * <p>
     * Every {@code Action} must have a unique {@code ActionType}
     *
     * @return this action's {@code ActionType}
     */
    public abstract ActionType getActionType();

    /**
     * Serializes this {@code Action} and all it's related data into a {@link Pack}
     * <p>
     * It is important to override this method if your {@code Action} is constructed from any
     * specific data. The {@code Pack} returned would later be given to your {@link ActionFactory}
     * to reconstruct this {@code Action} so the data must be in the exact format your {@code ActionFactory}
     * accepts.
     * <p>
     * Calling the {@code super} method is mandatory. The general structure of the overridden method is like below:
     * <pre>
     * <code>@Override
     * public Pack toPack() {
     *     Pack pack = super.toPack();
     *     pack.putString("my_field", "my action data");
     *     // Put other Action related data
     *     return pack;
     * }
     * </code>
     * </pre>
     *
     * @return the serialization of this {@code Action} as a {@code Pack}
     */
    public Pack toPack() {
        Pack pack = new Pack();
        pack.putString(Constants.getVal(Constants.F_ACTION_TYPE), getActionType().getCode());
        return pack;
    }

}
