package com.pushpole.sdk.util;

import android.annotation.TargetApi;
import android.app.Notification;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.graphics.Bitmap;
import android.os.Build;
import android.provider.Settings;
import android.support.v4.app.NotificationCompat;
import android.widget.RemoteViews;

import com.pushpole.sdk.Constants;
import com.pushpole.sdk.R;
import com.pushpole.sdk.action.Action;
import com.pushpole.sdk.controller.controllers.NotificationController;
import com.pushpole.sdk.internal.exceptions.NotificationBuildFailed;
import com.pushpole.sdk.message.downstream.NotificationDownstreamMessage;
import com.pushpole.sdk.network.ImageDownloader;
import com.pushpole.sdk.task.TaskManager;
import com.pushpole.sdk.task.tasks.NotificationHandleTask;

import static com.pushpole.sdk.PlainConstants.NOTIFICATION_CONTENT;
import static com.pushpole.sdk.PlainConstants.USER_BIG_CONTENT;
import static com.pushpole.sdk.PlainConstants.USER_BIG_TITLE;
import static com.pushpole.sdk.PlainConstants.USER_BUTTONS;
import static com.pushpole.sdk.PlainConstants.USER_CONTENT;
import static com.pushpole.sdk.PlainConstants.USER_ICON_URL;
import static com.pushpole.sdk.PlainConstants.USER_IMG_URL;
import static com.pushpole.sdk.PlainConstants.USER_JSON;
import static com.pushpole.sdk.PlainConstants.USER_SUMMARY;
import static com.pushpole.sdk.PlainConstants.USER_TITLE;
import static com.pushpole.sdk.util.Utility.isValidWebUrl;

//import com.pushpole.sdk.R;

/***
 * A class to build pushpole_custom_notification message for android device with api>=9 and <=15
 */
public class NotificationBuilderApiBelow16 extends NotificationBuilder {
    private Context mContext;
    private NotificationDownstreamMessage mMessage;
    private int mNotificationId;

    /***
     * constructor
     *
     * @param context        app context
     * @param message        downstream message
     * @param notificationId unique ID
     */
    public NotificationBuilderApiBelow16(Context context, NotificationDownstreamMessage message, int notificationId) {
        mContext = context;
        mMessage = message;
        mNotificationId = notificationId;
    }

    /***
     * Build pushpole_custom_notification object
     */
    @SuppressWarnings("deprecation")
    @TargetApi(15)
    public Notification build() throws NotificationBuildFailed {
        NotificationCompat.Builder builder = new NotificationCompat.Builder(mContext);
        setDefaultAttributes(builder);
        setNotificationContent(builder);
        setNotificationIcon(builder);
        setNotificationLed(builder);
        setNotificationButtons(builder);
        setNotificationSound(builder);
        if(Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            if(mMessage.getNotifChannelID() != null && !mMessage.getNotifChannelID().isEmpty()){
                builder.setChannelId(mMessage.getNotifChannelID());
            }else
                builder.setChannelId(Constants.getVal(Constants.NOTIF_DEFAULT_CHANNEL_ID));
        }
        builder.setOngoing(mMessage.isPermanentPush()); //make push permanent if asked by user
        builder.setPriority(mMessage.getPriority());
        /*if (Build.VERSION.SDK_INT <= Build.VERSION_CODES.GINGERBREAD_MR1 ){
            final Intent emptyIntent = new Intent();
            PendingIntent pendingIntent = PendingIntent.getActivity(mContext, 20107634 *//*a dummy code*//*, emptyIntent, PendingIntent.FLAG_UPDATE_CURRENT);
            builder.setContentIntent(pendingIntent); //Required on Gingerbread and below
        }*/
        return builder.getNotification();
    }


    /***
     * set default attribute of pushpole_custom_notification
     */
    @TargetApi(15)
    private void setDefaultAttributes(NotificationCompat.Builder builder) {
        PendingIntent actionPendingIntent =
                PendingIntent.getService(mContext, IdGenerator.generateIntegerId(),
                        createActionIntent(mMessage.getAction(), null),
                        PendingIntent.FLAG_UPDATE_CURRENT);

        PendingIntent dismissPendingIntent =
                PendingIntent.getService(mContext, IdGenerator.generateIntegerId(),
                        createDismissIntent(), PendingIntent.FLAG_UPDATE_CURRENT);

        builder
                .setContentIntent(actionPendingIntent)
                .setDeleteIntent(dismissPendingIntent);
    }

    /***
     * set pushpole_custom_notification content
     */
    @TargetApi(15)
    private void setNotificationContent(NotificationCompat.Builder builder) throws NotificationBuildFailed {
        boolean isJustImgBkground = (mMessage.getJustImageUrl() != null && !mMessage.getJustImageUrl().isEmpty());
        if(isJustImgBkground) { //only show an image in background of notification
            setCustomLayout(builder);
        }
        else {
            builder.setContentTitle(mMessage.getTitle());
            builder.setContentText(mMessage.getContent());
        }

        if (mMessage.getTicker() != null && !mMessage.getTicker().isEmpty()) {
            builder.setTicker(mMessage.getTicker());
        }

        NotificationCompat.Style style = null;
        if (mMessage.getImageUrl() != null && !mMessage.getImageUrl().isEmpty()) {
            style = buildPictureImageStyle();
        } else if ((mMessage.getBigTitle() != null && !mMessage.getBigTitle().isEmpty()) ||
                (mMessage.getBigContent() != null && !mMessage.getBigContent().isEmpty())) {
            style = buildBigTextStyle();
        }

        if (style != null) {
            builder.setStyle(style);
        }
    }

    /***
     * build pushpole_custom_notification bigText, bigContentTitle and setSummaryText
     */
    @TargetApi(15)
    private NotificationCompat.Style buildBigTextStyle() {
        NotificationCompat.BigTextStyle style = new NotificationCompat.BigTextStyle();
        if (mMessage.getBigTitle() != null && !mMessage.getBigTitle().isEmpty()) {
            style.setBigContentTitle(mMessage.getBigTitle());
        } else {
            style.setBigContentTitle(mMessage.getTitle());
        }

        if (mMessage.getBigContent() != null && !mMessage.getBigContent().isEmpty()) {
            style.bigText(mMessage.getBigContent());
        } else {
            style.bigText(mMessage.getContent());
        }

        if (mMessage.getSummary() != null && !mMessage.getSummary().isEmpty()) {
            style.setSummaryText(mMessage.getSummary());
        }

        return style;
    }

    /***
     * build pushpole_custom_notification bigPicture and bigIcon and and bigContentTitle and setSummaryText
     */
    @TargetApi(15)
    private NotificationCompat.Style buildPictureImageStyle() throws NotificationBuildFailed {
        NotificationCompat.BigPictureStyle style = new NotificationCompat.BigPictureStyle();
        if (mMessage.getBigTitle() != null && !mMessage.getBigTitle().isEmpty()) {
            style.setBigContentTitle(mMessage.getBigTitle());
        }

        if (mMessage.getSummary() != null && !mMessage.getSummary().isEmpty()) {
            style.setSummaryText(mMessage.getSummary());
        }

        ImageDownloader downloader = new ImageDownloader(/*mContext*/);

        if (mMessage.getBigIconUrl() != null && !mMessage.getBigIconUrl().isEmpty()) {
            Bitmap bigIcon = downloader.downloadImage(mMessage.getBigIconUrl());
            if (bigIcon == null) {
                throw new NotificationBuildFailed("Downloading pushpole_custom_notification big icon failed");
            }
            style.bigLargeIcon(bigIcon);
        }

        if (mMessage.getImageUrl() != null && !mMessage.getImageUrl().isEmpty()) {
            Bitmap image = downloader.downloadImage(mMessage.getImageUrl());
            if (image == null) {
                throw new NotificationBuildFailed("Downloading pushpole_custom_notification image failed");
            }
            style.bigPicture(image);
        }
        return style;
    }


    /***
     * set notification largeIcon
     */
    @TargetApi(15)
    private void setNotificationIcon(NotificationCompat.Builder builder) throws NotificationBuildFailed {
        boolean hasLargeIcon = (mMessage.getIconUrl() != null && !mMessage.getIconUrl().isEmpty());
        boolean usePushPoleIcon = mMessage.getUsePushPoleIcon();

        if (usePushPoleIcon) {
            builder.setSmallIcon(R.drawable.ic_pushpole);
        } else {
            if(mMessage.getNotifIcon() != null && !mMessage.getNotifIcon().isEmpty()){//if user specified an icon from drawable as its notifIcon
                String mName = "ic_" + mMessage.getNotifIcon().replaceAll("[- ]", "_");
                int resId0 = mContext.getResources().getIdentifier(mName, "drawable", mContext.getPackageName());
                if(resId0 > 0){
                    builder.setSmallIcon(resId0);
                }
            }else {
                boolean useWhiteIcon = (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.LOLLIPOP);
                int resId = mContext.getResources().getIdentifier("ic_silhouette", "drawable", mContext.getPackageName());
                if (useWhiteIcon && resId > 0) {
                    builder.setSmallIcon(resId);
                } else
                    builder.setSmallIcon(mContext.getApplicationInfo().icon);
            }
        }

        if (hasLargeIcon) {
            ImageDownloader downloader = new ImageDownloader(/*mContext*/);
            Bitmap image = downloader.downloadImage(mMessage.getIconUrl());

            if (image == null) {
                throw new NotificationBuildFailed("Downloading pushpole_custom_notification large image icon failed");
            }
            builder.setLargeIcon(image);
        }
    }

    /**
     * set notification LED color
     *
     * @param builder notification builder
     */
    private void setNotificationLed(NotificationCompat.Builder builder) {
        if (mMessage.getLedColor() != 0) {
            builder.setLights(mMessage.getLedColor(), mMessage.getLedOnTime(), mMessage.getLedOffTime());
        }
    }

    /**
     * set notification buttons
     *
     * @param builder notification builder
     */
    @TargetApi(15)
    private void setNotificationButtons(NotificationCompat.Builder builder) {
        if (mMessage.getButtons() == null) {
            return;
        }

        for (NotificationDownstreamMessage.Button button : mMessage.getButtons()) {
            int iconId = MaterialIconHelper.getIconResourceByMaterialName(mContext, button.getIcon());
            if (iconId == 0) {//TODO: PU-8
                //iconId = R.drawable.ic_empty; //This approach don't works when lib is pro-guarded
                iconId= mContext.getResources().getIdentifier("ic_empty", "drawable", mContext.getPackageName());
            }
            PendingIntent pendingIntent = PendingIntent.getService(mContext, IdGenerator.generateIntegerId(),
                    createActionIntent(button.getAction(), button.toPack()), 0);
            builder.addAction(iconId, button.getText(), pendingIntent);
        }
    }

    /***
     * set pushpole_custom_notification sound,
     * NO_SOUND
     * DEFAULT_SOUND
     * download sound from server when soundUrl available. In the case of failure default sound will be set
     *
     * @param builder pushpole_custom_notification builder
     */
    @TargetApi(15)
    private void setNotificationSound(NotificationCompat.Builder builder) {
        if(!isValidWebUrl(mMessage.getSoundUrl())) {
            builder.setSound(Settings.System.DEFAULT_NOTIFICATION_URI);
        }
    }

    @TargetApi(15)
    private void setCustomLayout(NotificationCompat.Builder builder) throws NotificationBuildFailed {
        RemoteViews contentView = new RemoteViews(mContext.getPackageName(), R.layout.pushpole_custom_notification);
        ImageDownloader downloader = new ImageDownloader(/*mContext*/);
        Bitmap bckgroundImg = downloader.downloadImage(mMessage.getJustImageUrl());
        if (bckgroundImg == null) {
            throw new NotificationBuildFailed("Downloading pushpole_custom_notification big icon failed");
        }
        contentView.setImageViewBitmap(R.id.pushpole_notif_bkgrnd_image, bckgroundImg);
        builder.setContent(contentView);
    }

    /***
     * create an intent for showing pushpole_custom_notification
     *
     * @param action intent
     * @param button  button ID
     * @return intent
     */
    protected Intent createActionIntent(Action action, Pack button) {
        Pack data = new Pack();
        if (mMessage.getAction() != null) {
            data.putPack(Constants.getVal(Constants.F_ACTION), action.toPack());
        }
        data.putString(Constants.getVal(Constants.NOTIFICATION_ID), String.valueOf(mNotificationId));
        data.putString(Constants.getVal(Constants.F_ORIGINAL_MESSAGE_ID), mMessage.getMessageId());
        data.putString(Constants.getVal(Constants.F_RESPONSE_ACTION), Constants.getVal(Constants.RESPONSE_ACTION_CLICK));
        data.putPack(Constants.getVal(Constants.F_RESPONSE_BUTTON_ID), button);
        data.putPack(NOTIFICATION_CONTENT, notificationContent());
        return TaskManager.getInstance(mContext).createTaskIntent(
                NotificationHandleTask.class, data);
    }

    /***
     * Create an intent to dismiss pushpole_custom_notification
     *
     * @return intent
     */
    protected Intent createDismissIntent() {
        Pack data = new Pack();
        data.putString(Constants.getVal(Constants.NOTIFICATION_ID), String.valueOf(mNotificationId));
        data.putString(Constants.getVal(Constants.F_ORIGINAL_MESSAGE_ID), mMessage.getMessageId());
        data.putString(Constants.getVal(Constants.F_RESPONSE_ACTION),
                Constants.getVal(Constants.RESPONSE_ACTION_DISMISS));
        data.putPack("data", notificationContent());
        return TaskManager.getInstance(mContext).createTaskIntent(
                NotificationHandleTask.class, data);
    }

    /**
     * Extract notification content that needs to be returned to the user.
     * @return a pack of the notification data that holds {title, content, bigTitle, bigContent, summary, customContent and buttons}.
     */
    private Pack notificationContent() {
        Pack data = new Pack();
        // Include message content
        data.putString(USER_TITLE, mMessage.getTitle());
        data.putString(USER_CONTENT, mMessage.getContent());
        if (mMessage.getBigTitle() != null)
            data.putString(USER_BIG_TITLE, mMessage.getBigTitle());
        if (mMessage.getBigContent() != null)
            data.putString(USER_BIG_CONTENT, mMessage.getBigContent());
        if (mMessage.getSummary() != null)
            data.putString(USER_SUMMARY, mMessage.getSummary());
        if (mMessage.getCustomContent() != null)
            data.putString(USER_JSON, mMessage.getCustomContent().toJson());
        if (mMessage.getIconUrl() != null)
            data.putString(USER_ICON_URL, mMessage.getIconUrl());
        if (mMessage.getImageUrl() != null)
            data.putString(USER_IMG_URL, mMessage.getImageUrl());
        if (mMessage.getButtons() != null) {
            ListPack buttonData = new ListPack();
            for (NotificationDownstreamMessage.Button b : mMessage.getButtons()) {
                buttonData.add(b.toPack());
            }
            data.putListPack(USER_BUTTONS, buttonData);
        }
        return data;
    }

}
