package com.pushpole.sdk.topic;

import android.content.Context;

import java.util.Set;

import com.pushpole.sdk.internal.db.KeyStore;
import com.pushpole.sdk.util.Pack;

/***
 * A singleton class to store topics
 */
public class TopicStore {
    private final static String KEYSTORE_KEY = "$#topics";
    private final static int ADD = 0;
    private final static int REMOVE = 1;
    private volatile static TopicStore mInstance;

    /***
     * empty constructor
     */
    private TopicStore() {

    }

    /***
     * get single instance of {@link TopicStore}, you must always use this method for creating instance of {@link TopicStore}
     *
     * @return single instance of {@link TopicStore}
     */
    public static TopicStore getInstance() {
        if (mInstance == null) {
            synchronized (TopicStore.class) {
                if (mInstance == null) {
                    mInstance = new TopicStore();
                }
            }
        }
        return mInstance;
    }

    /***
     * add new topic
     *
     * @param context   the app context
     * @param topicCode the topic code
     */
    public void addTopic(Context context, String topicCode) {
        modify(context, topicCode, ADD);
    }

    /***
     * add new topic
     *
     * @param context the app context
     * @param topic   the {@link Topic} object
     */
    public void addTopic(Context context, Topic topic) {
        addTopic(context, topic.getCode());
    }

    /***
     * remove a topic from {@link TopicStore}
     *
     * @param context   app context
     * @param topicCode the topic code
     */
    public void removeTopic(Context context, String topicCode) {
        modify(context, topicCode, REMOVE);
    }

    /***
     * remove a topic from {@link TopicStore}
     *
     * @param context app context
     */
    public void removeTopic(Context context, Topic topic) {
        removeTopic(context, topic.getCode());
    }

    /***
     * Get set of topics
     *
     * @param context app context
     * @return {@link Set} of topics
     */
    public Set<String> getTopics(Context context) {
        Pack topicList = KeyStore.getInstance(context).getPack(KEYSTORE_KEY, null);
        if (topicList == null) {
            topicList = new Pack();
        }

        return topicList.keySet();
    }

    /***
     * modify {@link TopicStore} topics
     *
     * @param context
     * @param topicCode
     * @param action
     */
    private void modify(Context context, String topicCode, int action) {
        Pack topicList = KeyStore.getInstance(context).getPack(KEYSTORE_KEY, null);

        if (topicList == null) {
            topicList = new Pack();
        }

        if (action == ADD && topicList.getString(topicCode, null) == null) {
            topicList.putString(topicCode, "1");
        } else if (action == REMOVE) {
            topicList.remove(topicCode);
        }

        KeyStore.getInstance(context).putPack(KEYSTORE_KEY, topicList);
    }

}
