package com.pushpole.sdk.task.tasks;

import android.app.NotificationManager;
import android.content.Context;
import android.content.Intent;

import com.pushpole.sdk.Constants;
import com.pushpole.sdk.NotificationButtonData;
import com.pushpole.sdk.NotificationData;
import com.pushpole.sdk.PlainConstants;
import com.pushpole.sdk.PushPole;
import com.pushpole.sdk.action.Action;
import com.pushpole.sdk.action.ActionFactory;
import com.pushpole.sdk.action.ActionType;
import com.pushpole.sdk.internal.log.LogData;
import com.pushpole.sdk.internal.log.Logger;
import com.pushpole.sdk.message.upstream.NotificationUpstreamMessage;
import com.pushpole.sdk.message.upstream.UpstreamMessage;
import com.pushpole.sdk.network.SendManager;
import com.pushpole.sdk.task.PushPoleTask;
import com.pushpole.sdk.task.Result;
import com.pushpole.sdk.util.InsufficientPermissionsException;
import com.pushpole.sdk.util.NetworkHelper;
import com.pushpole.sdk.util.Pack;

/***
 * A class that represent task for handling notification
 */
public class NotificationHandleTask implements PushPoleTask {

    /***
     * The runTask contains operations for handling notification
     *
     * @param context  the app context
     * @param taskPack the task pack
     * @return @return * @return {@code RESULT_SUCCESS}
     */
    @Override
    public Result runTask(Context context, Pack taskPack) {
        cancelNotification(context, taskPack);
        executeAction(context, taskPack);
        sendUpstreamMessage(context, taskPack);
        return Result.SUCCESS;
    }

    /***
     * cancel notification by unique ID
     * retrieve ID from taskPack
     *
     * @param context  the app context
     * @param taskPack the task pack
     */
    private void cancelNotification(Context context, Pack taskPack) {
        int notificationId;
        try {
            notificationId = Integer.parseInt(taskPack.getString(Constants.getVal(Constants.NOTIFICATION_ID)));
        } catch (NumberFormatException e) {
            //Logger.error("Invalid notification id %s", taskPack.getString(Constants.getVal(Constants.NOTIFICATION_ID)));
            return;
        }

        // Cancel notification
        NotificationManager notificationManager = (NotificationManager) context.getSystemService(Context.NOTIFICATION_SERVICE);
        if (notificationManager != null) {
            notificationManager.cancel(notificationId);
        }

        // Close notification bar if not dismiss
        String responseAction = taskPack.getString(Constants.getVal(Constants.F_RESPONSE_ACTION), "");
        if (!responseAction.equals(Constants.getVal(Constants.RESPONSE_ACTION_DISMISS))) {
            Intent it = new Intent(Intent.ACTION_CLOSE_SYSTEM_DIALOGS);
            it.setPackage(context.getPackageName());
            context.sendBroadcast(it);
        }
    }

    /***
     * retrieve action from taskPack and execute appropriate action
     *
     * @param context  the app context
     * @param taskPack the task pack
     */
    private void executeAction(Context context, Pack taskPack) {
        Pack actionData = taskPack.getPack(Constants.getVal(Constants.F_ACTION));
        if (actionData == null) {
            return;
        }

        String actionCode = actionData.getString(Constants.getVal(Constants.F_ACTION_TYPE), null);
        if (actionCode == null) {
            return;
        }

        ActionType actionType = ActionType.fromCode(actionCode);

        if (actionType == null) {
            Logger.error("Attempted to handle invalid Action Type: %s", actionCode);
            return;
        }

        ActionFactory factory = actionType.getFactory();
        Action action = factory.buildAction(actionData);

        action.execute(context);
    }

    /***
     * send appropriate message to pushpole-server when notification response action occur
     * response actions are : RESPONSE_ACTION_CLICK, RESPONSE_ACTION_DISMISS
     */
    private void sendUpstreamMessage(Context context, Pack taskPack) {

        Pack notificationContent = taskPack.getPack(PlainConstants.NOTIFICATION_CONTENT, new Pack());

        String originalMessageId = taskPack.getString(Constants.getVal(Constants.F_ORIGINAL_MESSAGE_ID), "");
        String responseAction = taskPack.getString(Constants.getVal(Constants.F_RESPONSE_ACTION), "");
        Pack clickedButtonPack = taskPack.getPack(Constants.getVal(Constants.F_RESPONSE_BUTTON_ID), new Pack());
        NotificationButtonData clickedButton = NotificationButtonData.fromPack(clickedButtonPack);
        Integer buttonId = null;

        String notificationId = "";

        try {
            String[] idParts = originalMessageId.split("#");
            int idPartLength = idParts.length;
            if (idPartLength > 0) {
                notificationId = idParts[idPartLength - 1];
            }
        } catch (Exception e) {
            Logger.warning("Could not set notification id", new LogData(
                    "Message ID", originalMessageId,
                    "Cause", e.getMessage()
            ));
        }

        try {
            buttonId = clickedButton.getId();
        } catch (NumberFormatException ignored) {}


        // Notification click or button click
        if (responseAction.equals(Constants.getVal(Constants.RESPONSE_ACTION_CLICK))) {
            Logger.debug("Notification Clicked", new LogData(
                    "Message ID", originalMessageId,
                    "Button Clicked", clickedButton.toString(),
                    "NotificationContent", notificationContent.toJson()
            ));

            Intent intent = new Intent();
            intent.setPackage(context.getPackageName());
            if (!notificationId.isEmpty()) {
                intent.putExtra(Constants.NOTIFICATION_ID, notificationId);
            }
            if(clickedButton.getId() != -1 && !clickedButtonPack.isEmpty()) {

                if (PushPole.pushpoleNotificationListener != null) {
                    PushPole.pushpoleNotificationListener.onNotificationButtonClicked(NotificationData.fromPack(notificationContent), clickedButton);
                }

                intent.setAction(context.getPackageName() + Constants.getVal(Constants.NOTIF_BTN_CLICKED_BROADCAST));
                intent.putExtra(Constants.getVal(Constants.NOTIF_BTN_ID_EXTRA), clickedButton.getId());
            }
            else {
                if (PushPole.pushpoleNotificationListener != null) {
                    PushPole.pushpoleNotificationListener.onNotificationClicked(NotificationData.fromPack(notificationContent));
                }
                intent.setAction(context.getPackageName() + Constants.getVal(Constants.NOTIF_CLICKED_BROADCAST));
            }
            context.sendBroadcast(intent);
        // Notification dismiss
        } else if (responseAction.equals(Constants.getVal(Constants.RESPONSE_ACTION_DISMISS))) {
            Logger.debug("Notification Dismissed", new LogData(
                    "Message ID", originalMessageId,
                    "NotificationContent", notificationContent.toJson()
            ));
            if (PushPole.pushpoleNotificationListener != null) {
                PushPole.pushpoleNotificationListener.onNotificationDismissed(NotificationData.fromPack(notificationContent));
            }

            Intent intent = new Intent();
            intent.setPackage(context.getPackageName());
            if (!notificationId.isEmpty()) {
                intent.putExtra(Constants.NOTIFICATION_ID, notificationId);
            }
            intent.setAction(context.getPackageName() + Constants.getVal(Constants.NOTIF_DISMISSED_BROADCAST));
            intent.putExtra(PlainConstants.NOTIFICATION_CONTENT, notificationContent);
            context.sendBroadcast(intent);
        } else {
            Logger.warning("Invalid RESPONSE_ACTION on notification", new LogData(
                    "Message ID", originalMessageId,
                    responseAction, responseAction
            ));
        }

        NotificationUpstreamMessage.Factory factory = new NotificationUpstreamMessage.Factory();
        UpstreamMessage message = factory.buildMessage(originalMessageId, responseAction, buttonId);

        Pack msgActionPack = message.toPack();

        //adding network connectivity status to click action
        String networkTypeName = Constants.getVal(Constants.NONE);
        try {
            networkTypeName = NetworkHelper.getNetworkTypeName(context);
        } catch (InsufficientPermissionsException ignored) {}
        if (Constants.getVal(Constants.WIFI).equals(networkTypeName)) {
            msgActionPack.putString(Constants.getVal(Constants.F_INTERNET_STATUS),
                    Constants.getVal(Constants.WIFI));
        } else if (!Constants.getVal(Constants.NONE).equals(networkTypeName)) {//here networkType name contains name of mobile network
            msgActionPack.putString(Constants.getVal(Constants.F_INTERNET_STATUS),
                    Constants.getVal(Constants.MOBILE));
            msgActionPack.putString(Constants.getVal(Constants.F_MOBILE_NETWORK_NAME), networkTypeName);
        } else{
            msgActionPack.putString(Constants.getVal(Constants.F_INTERNET_STATUS), networkTypeName);
        }

        //new UpstreamSender(context).sendMessage(message);
        SendManager.getInstance(context).send(Constants.getVal(Constants.NOTIFICATION_ACTION_T), msgActionPack);
    }
}
