package com.pushpole.sdk.task;

import android.content.Context;

import com.pushpole.sdk.Constants;
import com.pushpole.sdk.controller.controllers.NotificationController;
import com.pushpole.sdk.internal.exceptions.NotificationBuildFailed;
import com.pushpole.sdk.internal.log.LogData;
import com.pushpole.sdk.internal.log.Logger;
import com.pushpole.sdk.message.MessageStore;
import com.pushpole.sdk.message.downstream.NotificationDownstreamMessage;
import com.pushpole.sdk.network.SendManager;
import com.pushpole.sdk.task.options.DefaultOptions;
import com.pushpole.sdk.util.InsufficientPermissionsException;
import com.pushpole.sdk.util.NetworkHelper;
import com.pushpole.sdk.util.Pack;

/***
 * A class that represent task for building notification
 */
@DefaultOptions(window = 30, retryCount = 5, persisted = true, network = true)
public class NotificationBuildTask implements PushPoleTask {
    public static final int NOTIF_PUBLISHED = 1;
    public static final int NOTIF_FAILED = 2;
    public static final int NOTIF_DISABLED = 3;
    public static final int NOTIF_SYSTEM_DISABLE = 4;

    /***
     * The runTask contains operations for building notification
     *
     * @param context  the app context
     * @param taskPack the task pack
     * @return * @return {@code RESULT_SUCCESS} or {@code RESULT_RESCHEDULE} or {@code RESULT_FAILURE}
     */
    @Override
    public Result runTask(Context context, Pack taskPack) {
        if (taskPack == null) {
            Logger.warning("No data given to Notification Build Task");
            return Result.FAIL;
        }

        String messageId = taskPack.getString(Constants.getVal(Constants.MESSAGE_ID), null);
        if (messageId == null) {
            Logger.error("No message id given in Notification Build Task");
            return Result.FAIL;
        }

        NotificationDownstreamMessage message = null;
        try {
            message = (NotificationDownstreamMessage) MessageStore.getInstance().getMessage(context, messageId);
            if(taskPack.getString(Constants.getVal(Constants.MESSAGE_IMAGE_RETRY)) != null){
                message.setRetryImgFetch(true);
            }
        } catch (MessageStore.NoSuchMessageException e) {
            Logger.error("Message not found in Notification Build Task", new LogData(
                    "Message ID", messageId
            ));
            sendNotifPublishStatus(context, messageId, NOTIF_FAILED);
            return Result.FAIL;
        } catch (ClassCastException e) {
            Logger.error("Invalid message given to Notification Build Task", new LogData(
                    "Message ID", messageId
            ));
            sendNotifPublishStatus(context, messageId, NOTIF_FAILED);
            return Result.FAIL;
        }

        try {
            new NotificationController(context).showNotificationMessage(message);
        } catch (NotificationBuildFailed notificationBuildFailed) {
            return Result.RESCHEDULE;
        }

        MessageStore.getInstance().deleteMessage(context, message);
        return Result.SUCCESS;
    }

    public static void sendNotifPublishStatus(Context context, String msgId, int status){
        try{
//            DetectUserActivityTask detectActivity = new DetectUserActivityTask();
//            detectActivity.connectToActivityRecogApi(context, true, 1000 /*4.5 second*/, 5000); //this call saves most probable activity in KeyStore

            Pack notifStatusPack = new Pack();
            notifStatusPack.putString(Constants.getVal(Constants.F_ORIGINAL_MESSAGE_ID), msgId);
            switch (status){
                case NOTIF_PUBLISHED:
                    notifStatusPack.putString(Constants.getVal(Constants.NOTIF_PUBLISH_STAT), Constants.getVal(Constants.NOTIF_PUBLICATION_STATUS_PUBLISHED));
                    break;
                case NOTIF_FAILED:
                    notifStatusPack.putString(Constants.getVal(Constants.NOTIF_PUBLISH_STAT), Constants.getVal(Constants.NOTIF_PUBLICATION_STATUS_FAILED));
                    break;
                case NOTIF_DISABLED:
                    notifStatusPack.putString(Constants.getVal(Constants.NOTIF_PUBLISH_STAT), Constants.getVal(Constants.NOTIF_PUBLICATION_STATUS_IS_OFF));
                    break;
                case NOTIF_SYSTEM_DISABLE:
                    notifStatusPack.putString(Constants.getVal(Constants.NOTIF_PUBLISH_STAT), Constants.getVal(Constants.NOTIF_PUBLICATION_STATUS_DISABLED));
                    break;
            }

//            String userActivity= KeyStore.getInstance(context).getString(Constants.getVal(Constants.MOST_PROBABLE_USER_ACTIVITY), "");
//            KeyStore.getInstance(context).delete(Constants.getVal(Constants.MOST_PROBABLE_USER_ACTIVITY));
//            String [] tmp = userActivity.split("_");
//            if(tmp.length == 2){
//                notifStatusPack.putString(Constants.getVal(Constants.ACTIVITY_TYPE), tmp[0]);
//                try {
//                    notifStatusPack.putInt(Constants.getVal(Constants.ACTIVITY_CONFIDENCE), Integer.valueOf(tmp[1]));
//                }catch (Exception e){}
//            }

            //adding network connectivity status to click action
            String networkTypeName = "none";
            try {
                networkTypeName = NetworkHelper.getNetworkTypeName(context);

            } catch (InsufficientPermissionsException e) {
            }
            if (Constants.getVal(Constants.WIFI).equals(networkTypeName)) {
                notifStatusPack.putString(Constants.getVal(Constants.F_INTERNET_STATUS), Constants.getVal(Constants.WIFI));
            } else if (!Constants.getVal(Constants.NONE).equals(networkTypeName)) {//here networkType name contains name of mobile network
                notifStatusPack.putString(Constants.getVal(Constants.F_INTERNET_STATUS), Constants.getVal(Constants.MOBILE));
                notifStatusPack.putString(Constants.getVal(Constants.F_MOBILE_NETWORK_NAME), networkTypeName);
            }
            else{
                notifStatusPack.putString(Constants.getVal(Constants.F_INTERNET_STATUS), networkTypeName);
            }
            notifStatusPack.putString(Constants.getVal(Constants.TIMESTAMP), String.valueOf(System.currentTimeMillis()));
            SendManager.getInstance(context).send(Constants.getVal(Constants.NOTIFICATION_SHOWN_STAT_T), notifStatusPack);
        }catch (Exception e){}
    }
}
