package com.pushpole.sdk.internal.log;

import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.os.Bundle;

import com.pushpole.sdk.PlainConstants;
import com.pushpole.sdk.internal.db.KeyStore;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import com.pushpole.sdk.util.Utility;


public class Logger {
    private static Logger mInstance;
    private Map<LogHandler, LogLevel> mLogHandlers;
    private boolean mEnabled;
    private LogLevel mMinLogLevel;
    private boolean initialized = false;

    private Logger() {
        mLogHandlers = new HashMap<>();
        mMinLogLevel = LogLevel.ERROR;
    }

    public static Logger getInstance() {
        if (mInstance == null) {
            initialize(null);
        }
        return mInstance;
    }

    public synchronized static void initialize(Context context) {
        if (mInstance == null) {
            mInstance = new Logger();
        }
        if (!mInstance.initialized && context != null) {
            mInstance.initializeSentry(context);

            mInstance.initializeHandlers(context);
            mInstance.initialized = true;
        }

    }

    private void initializeSentry(Context context) {
        try {
            ExceptionCatcher.getInstance(context).setDefaultUncaughtExcepHandler(Thread.getDefaultUncaughtExceptionHandler());
            //TODO: reade sentry dsn from manifest

            String userSentryDsn = KeyStore.getInstance(context).getString(PlainConstants.USER_DSN_URL_KEYSTORE, null);//dsn received from command #25
            if (userSentryDsn == null || userSentryDsn.isEmpty()) {
                userSentryDsn = KeyStore.getInstance(context).getString(PlainConstants.USER_MANIFEST_DSN_URL_KEYSTORE, null);//user default dsn
                if (userSentryDsn == null || userSentryDsn.isEmpty()) {
                    userSentryDsn = readDsnFromManifestToKeyStore(context);
                }
            }
            if (userSentryDsn != null && !userSentryDsn.startsWith("http")) {
                userSentryDsn = null;
            }
            if (userSentryDsn != null) {
                Sentry.init(context, userSentryDsn);
            }
            ExceptionCatcher.makePushPoleDefaultExceptionCatcher(context);//set our exceptionCatcher as default catcher
        } catch (Exception ex) {
            android.util.Log.e("PushPole", "Error occurred while initializing PushPole", ex);
        }
    }

    private  String readDsnFromManifestToKeyStore(Context context){
        String sentryDsn = null;
        try {
            ApplicationInfo ai = context.getPackageManager().getApplicationInfo(context.getPackageName(), PackageManager.GET_META_DATA);
            Bundle bundle = ai.metaData;
            sentryDsn = bundle.getString(PlainConstants.CONFIG_SENTRY_DSN);
        } catch (Exception e) {
            android.util.Log.e("PushPole", "Initializing Crash-Reporter failed.", e);
        }

        if (sentryDsn == null || sentryDsn.isEmpty()){
//            String defaultDsn = Utility.decodeBase64(Constants.SENTRY_ANDROID_DSN);
//            KeyStore.getInstance(context).putString(Constants.USER_MANIFEST_DSN_URL_KEYSTORE, defaultDsn);
//            return defaultDsn;
            return null;
        }

        String dsnValue = Utility.decodeBase64(sentryDsn);
        KeyStore.getInstance(context).putString(PlainConstants.USER_MANIFEST_DSN_URL_KEYSTORE, dsnValue);
        return dsnValue;
    }

    public static void log(Log log) {
        getInstance().logGeneral(log);
    }

    public static void log(LogLevel level, String message, LogData data) {
        getInstance().logGeneral(level, message, data);
    }

    public static void log(LogLevel level, String message, Object... params) {
        getInstance().logGeneral(level, message, params);
    }

    public static void debug(Log log) {
        getInstance().logDebug(log);
    }

    public static void debug(String message, LogData data) {
        getInstance().logDebug(message, data);
    }

    public static void debug(String message, Object... params) {
        getInstance().logDebug(message, params);
    }

    public static void info(Log log) {
        getInstance().logInfo(log);
    }

    public static void info(String message, LogData data) {
        getInstance().logInfo(message, data);
    }

    public static void info(String message, Object... params) {
        getInstance().logInfo(message, params);
    }


    //-----------------------------------------------------------------------

    public static void warning(Log log) {
        getInstance().logWarning(log);
    }

    public static void warning(String message, LogData data) {
        getInstance().logWarning(message, data);
    }

    // TODO handle null messages

    public static void warning(String message, Object... params) {
        getInstance().logWarning(message, params);
    }

    public static void error(Log log) {
        getInstance().logError(log);
    }

    public static void error(String message, LogData data) {
        getInstance().logError(message, data);
    }

    public static void error(String message, Object... params) {
        getInstance().logError(message, params);
    }

    public static void fatal(Log log) {
        getInstance().logFatal(log);
    }

    public static void fatal(String message, LogData data) {
        getInstance().logFatal(message, data);
    }

    public static void fatal(String message, Object... params) {
        getInstance().logFatal(message, params);
    }

    public void setEnabled(boolean enabled) {
        mEnabled = enabled;
    }

    public synchronized void registerHandler(LogHandler handler, LogLevel level) {
        LogLevel mLevel = level;
        if (mLevel == null) {
            mLevel = LogLevel.ERROR;
        }
        if (mLevel.ordinal() < mMinLogLevel.ordinal()) {
            mMinLogLevel = mLevel;
        }
        mLogHandlers.put(handler, mLevel);
    }

    public boolean isRegistered(LogHandler handler) {
        return mLogHandlers.containsKey(handler);
    }

    public synchronized void registerHandler(LogHandler handler, String levelString) {
        LogLevel level;
        switch (levelString.toLowerCase().trim()) {
            case "debug":
                level = LogLevel.DEBUG;
                break;

            case "error":
                level = LogLevel.ERROR;
                break;

            case "fatal":
                level = LogLevel.FATAL;
                break;

            case "info":
                level = LogLevel.INFO;
                break;

            case "warning":
                level = LogLevel.WARN;
                break;

            default:
                level = LogLevel.DEBUG;
                break;

        }
        registerHandler(handler, level);
    }

    public synchronized void unregisterHandler(LogHandler handler) {
        mLogHandlers.remove(handler);
    }

    public synchronized void unregisterAllHandlers() {
        mLogHandlers.clear();
        mMinLogLevel = LogLevel.ERROR;
    }

    public Set<LogHandler> getHandlers() {
        return mLogHandlers.keySet();
    }

    /**
     * initialize log handlers
     * by default initialize SentryHandler
     * if {@code Constants.DEBUG} true then initialize DbLogHandler and  LogcatLogHandler
     *
     * @param context
     */
    private synchronized void initializeHandlers(Context context) {

        setEnabled(true);

        /*if(Constants.DEBUG) {
            registerHandler(new DbLogHandler(context), LogLevel.DEBUG);
            registerHandler(new LogcatLogHandler(context), LogLevel.DEBUG);
        }

        registerHandler(new SentryHandler(context), LogLevel.WARNING);*/
        readAndInitHandlers(context); //using reflection to register desired handlers
    }

    //-----------------------------------------------------------------------
    private void readAndInitHandlers(Context context) {

        BufferedReader reader = null;
        StringBuilder sb = new StringBuilder();
        try {
            //android.util.Log.i("PushPole handlers ", "before open json file ");
            int jsonFileId = context.getResources().getIdentifier("log_handlers", "raw", context.getPackageName() );
            InputStream inputStream = context.getResources().openRawResource(jsonFileId);
            //android.util.Log.i("PushPole handlers ", "after opening json with name:" + R.raw.log_handlers);
            reader = new BufferedReader(
                    new InputStreamReader(inputStream, "UTF-8"), 8);
            //android.util.Log.i("PushPole handlers ", "init bufferReader done");
            String line = "";
            while ((line = reader.readLine()) != null) {
                //android.util.Log.i("PushPole handlers","read line: " + line);
                sb.append(line);
                sb.append('\n');
            }
            String json = sb.toString();
            //android.util.Log.i("PushPole handlers", "Json file read:" + json);
            if (!json.isEmpty()) {
                JSONObject jsonObject = new JSONObject(json);
                JSONArray jsonArray = jsonObject.getJSONArray("logHandlers");
                if (jsonArray != null && jsonArray.length() > 0) {
                    for (int i = 0; i < jsonArray.length(); i++) {
                        //android.util.Log.i("PushPole handlers","get object : " + jsonArray.get(i));
                        JSONObject entry = jsonArray.getJSONObject(i);
                        String className = entry.getString("logHandlerClass");
                        String logLevelStr = entry.getString("logLevel");

                        Constructor c = Class.forName(className).getConstructor(Context.class);
                        Object aClassObj =  c.newInstance(context);

                        //Object aClassObj = (LogHandler) Class.forName(className).newInstance(); //calls default constructor of loaded class
                        if (aClassObj instanceof LogHandler) {
                            LogHandler handler = (LogHandler) aClassObj;
                            registerHandler(handler, logLevelStr);
                        }
                        else{
                            android.util.Log.e("PushPole", className + ", used in log_handlers.json as a handler, is not a subClass of LogHandler");
                        }
                    }
                }
            }

        } catch (FileNotFoundException | UnsupportedEncodingException | JSONException | ClassNotFoundException
                | InstantiationException | IllegalAccessException | ClassCastException
                | NoSuchMethodException | InvocationTargetException e) { //todo: handle below error log with logger class
            android.util.Log.e("PushPole" , "Exception in readAndInitHandlers() ", e);
        }  catch (IOException e) {
            android.util.Log.e("PushPole", "IOException in readAndInitHandlers() ", e);
        }
    }

    private synchronized void notifyLog(Log log) {
        if (!mEnabled) {
            return;
        }

        for (LogHandler handler : mLogHandlers.keySet()) {
            LogLevel level = mLogHandlers.get(handler);
            if (level != null && level.ordinal() <= log.getLogLevel().ordinal()) {
                handler.onLog(log);
            }
        }
    }

    private void notifyLog(LogLevel level, LogData data, String message, Object... params) {
//        android.util.Log.d("PushPole", "[L] " + message + " [#R] " + initialized + " " +  mLogHandlers.size()  + " [#I] " + this.toString());

        if (!mEnabled || (level != null && level.ordinal() < mMinLogLevel.ordinal())) {
            return;
        }

        Log log = new Log()
                .setLogLevel(level)
                .setMessage(message)
                .setParams(params)
                .setLogData(data)
                .setTimestamp(new Date().getTime());

        for (Object param : params) {
            if (param instanceof Throwable) {
                log.setException((Throwable) param);
            }
        }

        // Add stacktrace to error and warning messages by default
        if (level == LogLevel.ERROR || level == LogLevel.WARN) {
//            log.setStackTrace();
        }

        notifyLog(log);
    }

    public void logGeneral(Log log) {
        notifyLog(log);
    }

    public void logGeneral(LogLevel level, String message, LogData data) {
        notifyLog(level, data, message);
    }

    public void logGeneral(LogLevel level, String message, Throwable exception, LogData data) {
        notifyLog(level, data, message, exception);
    }

    public void logGeneral(LogLevel level, String message, Object... params) {
        notifyLog(level, null, message, params);
    }

    public void logDebug(Log log) {
        notifyLog(log.setLogLevel(LogLevel.DEBUG));
    }

    public void logDebug(String message, LogData data) {
        notifyLog(LogLevel.DEBUG, data, message);
    }

    public void logDebug(String message, Object... params) {
        notifyLog(LogLevel.DEBUG, null, message, params);
    }

    public void logInfo(Log log) {
        notifyLog(log.setLogLevel(LogLevel.INFO));
    }

    public void logInfo(String message, LogData data) {
        notifyLog(LogLevel.INFO, data, message);
    }

    public void logInfo(String message, Object... params) {
        notifyLog(LogLevel.INFO, null, message, params);
    }

    public void logWarning(Log log) {
        notifyLog(log.setLogLevel(LogLevel.WARN));
    }

    public void logWarning(String message, LogData data) {
        notifyLog(LogLevel.WARN, data, message);
    }

    public void logWarning(String message, Object... params) {
        notifyLog(LogLevel.WARN, null, message, params);
    }

    public void logError(Log log) {
        notifyLog(log.setLogLevel(LogLevel.ERROR));
    }

    public void logError(String message, LogData data) {
        notifyLog(LogLevel.ERROR, data, message);
    }

    public void logError(String message, Object... params) {
        notifyLog(LogLevel.ERROR, null, message, params);
    }

    public void logFatal(Log log) {
        notifyLog(log.setLogLevel(LogLevel.FATAL));
    }

    public void logFatal(String message, LogData data) {
        notifyLog(LogLevel.FATAL, data, message);
    }

    public void logFatal(String message, Object... params) {
        notifyLog(LogLevel.FATAL, null, message, params);
    }
}
