package com.pushpole.sdk.internal.db;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.SQLException;
import android.database.sqlite.SQLiteDatabase;

import java.util.Date;

import com.pushpole.sdk.Constants;
import com.pushpole.sdk.internal.log.LogData;
import com.pushpole.sdk.internal.log.Logger;
import com.pushpole.sdk.message.upstream.UpstreamMessage;
import com.pushpole.sdk.network.GcmHandler;
import com.pushpole.sdk.util.InvalidJsonException;
import com.pushpole.sdk.util.Pack;
import com.pushpole.sdk.util.Utility;

/**
 * Created on 16-04-04, 2:12 PM.
 *
 * @author Akram Shokri
 */
public class NotifAndUpstreamMsgsDbOperation {
    private static volatile NotifAndUpstreamMsgsDbOperation mInstance;
    private SQLiteDatabase database;
    private DatabaseHelper dbHelper;
    private Context mContext;

    private NotifAndUpstreamMsgsDbOperation(Context context) {
        dbHelper = DatabaseHelper.getInstance(context);
        open(); //to prevent error when someone use this class and forgot open()
    }

    public static NotifAndUpstreamMsgsDbOperation getInstance(Context context) {
        if (mInstance == null) {
            synchronized (NotifAndUpstreamMsgsDbOperation.class) {
                if (mInstance == null) {
                    mInstance = new NotifAndUpstreamMsgsDbOperation(context);
                    mInstance.mContext = context.getApplicationContext();
                }
            }
        }
        return mInstance;
    }

    public void open() throws SQLException {
        if (database == null || (database != null && !database.isOpen()))
            database = dbHelper.getWritableDatabase();
    }

    public void close() {
        if (dbHelper != null)
            dbHelper.close();
        if (database != null && database.isOpen())
            database.close();
    }

    public boolean isMsgExists(String msgId) {
        String selectQuery = "SELECT * FROM " + DatabaseHelper.TABLE_UPSTREAM_MSG_DOWNSTREAM_NOTIF + " t WHERE t."
                + DatabaseHelper.COLUMN_GCM_MSG_ID + " = '" + msgId + "'";
        Cursor cursor = database.rawQuery(selectQuery, null);

        boolean res = cursor.moveToFirst();
        cursor.close();
        return res;
    }

    public void updateNotifMsg(String msgId, String key, String value) {
    try {
        String where = "" + DatabaseHelper.COLUMN_GCM_MSG_ID + " = '" + msgId;
        ContentValues cv = new ContentValues();
        cv.put(key, value);
        database.update(DatabaseHelper.TABLE_UPSTREAM_MSG_DOWNSTREAM_NOTIF, cv, where, null);
    }catch (Exception e){}

    }

    public Pack findMsg(String msgId) {
        String selectQuery = "SELECT * FROM " + DatabaseHelper.TABLE_UPSTREAM_MSG_DOWNSTREAM_NOTIF + " t WHERE t."
                + DatabaseHelper.COLUMN_GCM_MSG_ID + " = '" + msgId + "'";

        Cursor cursor = database.rawQuery(selectQuery, null);
        Pack pack = null;
        if (cursor.moveToFirst()) {
            String msgData = cursor.getString(cursor.getColumnIndex(DatabaseHelper.COLUMN_GCM_MSG_DATA));
            try {
                pack = Pack.fromJson(msgData);
            } catch (Exception e) {
                //TODO: log error
            }
        }
        // make sure to close the cursor
        cursor.close();
        return pack;
    }

    public long insertMsg(UpstreamMessage msg) {
        if (isMsgExists(msg.getMessageId()))
            return 0; //messageId already exists in db

        ContentValues values = new ContentValues();
        values.put(DatabaseHelper.COLUMN_GCM_MSG_ID, msg.getMessageId());
        if(msg.getMessageType() != null)
            values.put(DatabaseHelper.COLUMN_GCM_MSG_TYPE, msg.getMessageType().getTypeCode());

        values.put(DatabaseHelper.COLUMN_GCM_MSG_CREATED_DATE, getDate());
        values.put(DatabaseHelper.COLUMN_GCM_MSG_CREATED_TIME, new Date().getTime());//current dateTime in long format
        values.put(DatabaseHelper.COLUMN_GCM_MSG_DATA, msg.toPack().toJson());//converts msg to json and saves it

        try {
            long insertId = database.insertOrThrow(DatabaseHelper.TABLE_UPSTREAM_MSG_DOWNSTREAM_NOTIF, null, values);
            if (insertId == -1) {
                Logger.debug("Inserting message to android sqlite DB failed.");
            }
            return insertId;
        } catch (SQLException e) {
            Logger.warning("Inserting message errored: " + e.getMessage(), e);
            return -1;
        }
    }

    /**
     * Send upstream message in compliance with <a href: https://docs.google.com/document/d/12IZSE_uhQUttmF2k5Blu9XQgOZaMcZoZ62MLwpFPy1I>Message Passing Stream document </>
     * @param msgPack UpstreamMessage as a Pack object
     * @param msgId   Id of message
     * @return        rowId of sent message in upstream message table
     */

    public long insertMsg(Pack msgPack, String msgId) { //for refactored upstream message
        if (isMsgExists(msgId))
            return 0; //messageId already exists in db

        ContentValues values = new ContentValues();
        values.put(DatabaseHelper.COLUMN_GCM_MSG_ID, msgId);
        values.put(DatabaseHelper.COLUMN_GCM_MSG_TYPE, UpstreamMessage.Type.REFACTORED_UPSTREAM.getTypeCode());

        values.put(DatabaseHelper.COLUMN_GCM_MSG_CREATED_DATE, getDate());
        values.put(DatabaseHelper.COLUMN_GCM_MSG_CREATED_TIME, new Date().getTime());//current dateTime in long format
        values.put(DatabaseHelper.COLUMN_GCM_MSG_DATA, msgPack.toJson());//converts msg to json and saves it

        try {
            long insertId = database.insertOrThrow(DatabaseHelper.TABLE_UPSTREAM_MSG_DOWNSTREAM_NOTIF, null, values);
            if (insertId == -1) {
                Logger.debug("Inserting message to android sqlite DB failed.");
            }
            return insertId;
        } catch (SQLException e) {
            Logger.warning("Inserting message errored: " + e.getMessage(), e);
            return -1;
        }
    }

    public long insertNotifMsg(String messageId, int msgType) {//TODO: get MsgType and insert it along with other fields
        return insertNotifMsg(messageId, msgType, getDate(), new Date().getTime());
    }

    public long insertNotifMsg(String messageId, int msgType, String date, long dateInMilliSec) {
        if (isMsgExists(messageId))
            return 0; //messageId already exists in db

        ContentValues values = new ContentValues();
        values.put(DatabaseHelper.COLUMN_GCM_MSG_ID, messageId);
        values.put(DatabaseHelper.COLUMN_GCM_MSG_TYPE, msgType);

        values.put(DatabaseHelper.COLUMN_GCM_MSG_CREATED_DATE, date);
        values.put(DatabaseHelper.COLUMN_GCM_MSG_CREATED_TIME, dateInMilliSec);//current dateTime in long format
        try {
            long insertId = database.insertOrThrow(DatabaseHelper.TABLE_UPSTREAM_MSG_DOWNSTREAM_NOTIF, null, values);
            if (insertId == -1) {
                Logger.debug("Inserting notification messageId to android sqlite DB failed.");
            }
            return insertId;
        } catch (SQLException e) {
            Logger.warning("Error when inserting notification messageId to android sqlite DB: " + e.getMessage(), e);
            return -1;
        }
    }

    public int removeMsgByMsgId(String messageId) {
        String whereClause = DatabaseHelper.COLUMN_GCM_MSG_ID + " = ? ";

        String[] whereArgs = new String[]{
                messageId
        };

        int affectedRows = database.delete(DatabaseHelper.TABLE_UPSTREAM_MSG_DOWNSTREAM_NOTIF, whereClause, whereArgs);

        if (affectedRows != 1) {
            Logger.debug("Removing MessageId=" + messageId + " from android sqlite DB affected " + affectedRows + " rows");
        }
        return affectedRows;
    }

    public void retrySendingOutdatedMsgs(long minValidTime) {
        String selectQuery = "SELECT * FROM " + DatabaseHelper.TABLE_UPSTREAM_MSG_DOWNSTREAM_NOTIF + " t WHERE t."
                + DatabaseHelper.COLUMN_GCM_MSG_CREATED_TIME + " < " + minValidTime;

        Cursor cursor = database.rawQuery(selectQuery, null);
        LogData logData = new LogData();
        int i = 0;
        while (cursor.moveToNext()) {
            String msgData = cursor.getString(cursor.getColumnIndex(DatabaseHelper.COLUMN_GCM_MSG_DATA));
            if(msgData != null && msgData.contains(Constants.getVal(Constants.F_MESSAGE_ID))){
                try {//This is an upstream message which its ack is not received, so it is scheduled for being sent again
                    Pack taskPack = Pack.fromJson(msgData);
                    if(taskPack != null && !taskPack.isEmpty()) {
                        taskPack.putInt(
                                Constants.getVal(Constants.MESSAGE_RETRY_COUNT),
                                taskPack.getInt(Constants.getVal(Constants.MESSAGE_RETRY_COUNT),
                                        0)+1
                        );
                        GcmHandler.retrySendingMessage(taskPack, mContext);
                        Logger.debug("Retry sending upstream message", new LogData("message", msgData));
                    }
                } catch (InvalidJsonException e) {

                }
            }
            logData.put("msg #" + i, msgData);
            i++;
        }
        if (i > 0)
            Logger.warning("List and Content of outdated upstream messages: ", logData);
        // make sure to close the cursor
        cursor.close();

    }

    public int removeOutDatedMsg() {
        long minValidTime = new Date().getTime() - (24 * 3600000) /*24 hours in millisecond*/;
        retrySendingOutdatedMsgs(minValidTime);
        String whereClause = DatabaseHelper.COLUMN_GCM_MSG_CREATED_TIME + " < ? ";

        String[] whereArgs = new String[]{
                String.valueOf(minValidTime),
        };

        int affectedRows = database.delete(DatabaseHelper.TABLE_UPSTREAM_MSG_DOWNSTREAM_NOTIF, whereClause, whereArgs);
        return affectedRows;
    }


    private String getDate() {
        return Utility.getFormattedDate(new Date(), "yyyy-MM-dd");
    }

}
