package com.pushpole.sdk.device;

import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.content.pm.ResolveInfo;

import java.util.ArrayList;
import java.util.List;

/**
 * Helper class for retrieving application details on the user's device
 *
 * @author Hadi Zolfaghari
 */
public class DeviceApplicationHelper {
    private Context mContext;

    /**
     * Constructs a new {@code DeviceApplicationHelper}.
     *
     * @param context the context to use
     */
    public DeviceApplicationHelper(Context context) {
        mContext = context;
    }

    /**
     * Generates a list of {@link ApplicationDetails} instances per each non-system application
     * installed on the user's device.
     *
     * @return the list of {@code ApplicationDetails} for the device applications
     */
    public List<ApplicationDetails> getInstalledApplications() {
        List<ApplicationDetails> appList = new ArrayList<>();

        PackageManager packageManager = mContext.getPackageManager();
        List<PackageInfo> packageInfoList = packageManager.getInstalledPackages(0);

        for (PackageInfo info : packageInfoList) {
            if ((info.applicationInfo.flags & ApplicationInfo.FLAG_SYSTEM) != 1) {
                ApplicationDetails app = new ApplicationDetails();
                app.setPackageName(info.packageName);
                app.setVersion(info.versionName);
                app.setFirstInstallTime(info.firstInstallTime);
                app.setLastUpdateTime(info.lastUpdateTime);
                app.setAppName(String.valueOf(info.applicationInfo.nonLocalizedLabel));
                app.setIsAppHidden(isAppHidden(info.packageName));
                try {
                    app.setInstallerPackageName(packageManager.getInstallerPackageName(info.packageName));
                } catch (IllegalArgumentException ignored) {
                }
                appList.add(app);
            }
        }

        return appList;
    }


    /**
     * Gets the package name of the market place application through which this application was
     * installed from.
     *
     * @return the market place application's package name
     */
    public String getInstallerPackageName() {
        PackageManager packageManager = mContext.getPackageManager();
        try {
            return packageManager.getInstallerPackageName(mContext.getPackageName());
        } catch (IllegalArgumentException e) {
            return null;
        }
    }

    public boolean isAppHidden(String packageName){
        PackageManager pm = mContext.getPackageManager();
        Intent i = new Intent(/*"android.intent.action.MAIN"*/);
        i.addCategory("android.intent.category.LAUNCHER");
        i.setPackage(packageName);
        List<ResolveInfo> lst = pm.queryIntentActivities(i, PackageManager.MATCH_DISABLED_COMPONENTS|PackageManager.MATCH_ALL);

        if (lst == null || lst.size() == 0) {
            return true;
        }


        /*
            Terminology:
                MANIFEST: The "enabled" tag in the manifest
                STATE: The state received from PackageManager.getComponentEnabledSetting
                IS_ENABLED: The response received from activityInfo.isEnabled()

            If MANIFEST=true and the state is manually set to disabled from the app, then STATE=DISABLED but IS_ENABLED=true
            If MANIFEST=false and the state has not been touched from the app, then STATE=DEFAULT and IS_ENABLED=false

            So if STATE=DISABLED it is definitely a hidden app. If STATE=DEFAULT we have to check IS_ENABLED and if it is false
            then it is a hidden app
        */
        for (ResolveInfo resolveInfo : lst) {
            if (resolveInfo.activityInfo.packageName.equals(packageName)){
                String mPackage = resolveInfo.activityInfo.packageName;
                String mClass = resolveInfo.activityInfo.name;
                int status = mContext.getPackageManager().getComponentEnabledSetting( new ComponentName(mPackage,mClass));
                if (PackageManager.COMPONENT_ENABLED_STATE_DEFAULT == status) {
                    return !resolveInfo.activityInfo.isEnabled();
                } else if(PackageManager.COMPONENT_ENABLED_STATE_DISABLED == status) {
                    return true;
                } else if (PackageManager.COMPONENT_ENABLED_STATE_DISABLED_UNTIL_USED == status) {
                    return true;
                }
            }
        }

        return false;
    }
}
