package com.pushpole.sdk;

import android.app.AlertDialog;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.net.Uri;

import com.google.android.gms.common.ConnectionResult;
import com.google.android.gms.common.GoogleApiAvailability;

import java.util.Date;

import com.pushpole.sdk.internal.db.KeyStore;
import com.pushpole.sdk.internal.log.Log;
import com.pushpole.sdk.internal.log.LogData;
import com.pushpole.sdk.internal.log.Logger;

/***
 * A helper classs to get details about googlePlayServices
 */
public class GooglePlayServicesHelper {
    private static int mGooglePlayServicesVersionCode;
    private static String mGooglePlayServicesVersionName;

    /***
     * get google play services version code
     *
     * @param context app context
     * @return version code
     */
    public static int getGooglePlayServicesVersionCode(Context context) {
        if (mGooglePlayServicesVersionCode != -1) {
            try {
                mGooglePlayServicesVersionCode =
                        context.getPackageManager().getPackageInfo("com.google.android.gms", 0).versionCode;
            } catch (PackageManager.NameNotFoundException e) {
                Logger.warning(new Log().setMessage(e.getMessage()).setException(e).setTimestamp(new Date().getTime()));
            }
        }
        return mGooglePlayServicesVersionCode;
    }

    /***
     * get google play services version name
     *
     * @param context app context
     * @return version name
     */
    public static String getGooglePlayServicesVersionName(Context context) {
        if (mGooglePlayServicesVersionName == null) {
            try {
                mGooglePlayServicesVersionName =
                        context.getPackageManager().getPackageInfo("com.google.android.gms", 0).versionName;
            } catch (PackageManager.NameNotFoundException e) {
                Logger.warning(new Log().setMessage(e.getMessage()).setException(e).setTimestamp(new Date().getTime()));
            }
        }
        return mGooglePlayServicesVersionName;
    }

    /***
     * Check is client's device support networkManager
     *
     * @param context app context
     * @return boolean
     */
    public static boolean isGcmNetworkManagerSupported(Context context) {
        return getGooglePlayServicesVersionCode(context) >= 7500000;
    }

    /***
     * Check google play services is installed or updated
     *
     * @param context    app context
     * @param showDialog
     * @return a flag that indicate show googlePlayServices installation dialog or not when googlePlayServices is not installed or updated
     */

    public static boolean checkGooglePlayServiceAvailability(final Context context, final boolean showDialog) {
        try {
            int status = GoogleApiAvailability.getInstance().isGooglePlayServicesAvailable(context);
            ApplicationInfo ai = null;

            ai = context.getPackageManager().getApplicationInfo("com.google.android.gms", 0);
            if (ai != null && ai.enabled) {
                if ((status == ConnectionResult.SUCCESS) || (status == ConnectionResult.SERVICE_VERSION_UPDATE_REQUIRED)
                        || (status == ConnectionResult.SERVICE_UPDATING)) {
                    return true;
                }
            }

            Logger.error("Checking google play service availability returned false or encountered an exception", new LogData("Connection Result Status Code" , String.valueOf(status) ));
            KeyStore.getInstance(context).putString(Constants.getVal(Constants.GOOGLE_PLAY_STATUS_CODE), String.valueOf(status));

            if (showDialog) {
                if (GoogleApiAvailability.getInstance().isUserResolvableError(status)) {
                    new AlertDialog.Builder(context)
                            .setTitle(R.string.pushpole_missing_google_play_services_title)
                            .setMessage(R.string.pushpole_missing_google_play_services_text)
                            .setCancelable(true)
                            .setNegativeButton(R.string.pushpole_missing_google_play_services_dismiss, new DialogInterface.OnClickListener() {
                                @Override
                                public void onClick(DialogInterface dialog, int which) {
                                    dialog.dismiss();
                                }
                            })
                            .setPositiveButton(R.string.pushpole_missing_google_play_services_install, new DialogInterface.OnClickListener() {
                                public void onClick(DialogInterface dialog, int id) {
                                    Intent intent = new Intent(Intent.ACTION_VIEW);
                                    intent.setData(Uri.parse("market://details?id=com.google.android.gms"));
                                    try {
                                        context.startActivity(intent);
                                    } catch (Exception ex) {
                                        Logger.error("Google-Play-Service is missing. No market app is installed on this device to launch it for installing Google-Play-Service.", ex);
                                        android.util.Log.e("PushPole", "Google-Play-Service is missing. No market app is installed on this device to launch it for installing Google-Play-Service.\nPlease install/update it manually.");
                                    }
                                }
                            }).create().show(); //assumed that user has at least one market installed
                }
            }
        } catch (Exception e) {
            Logger.error("Error occurred while checking for google play services", e);
            e.printStackTrace();
        }
        return false;
    }
}
