package com.pushpole.sdk.internal.log;

import android.content.Context;

import java.util.Date;
import java.util.Random;

import com.pushpole.sdk.Constants;
import com.pushpole.sdk.internal.db.KeyStore;

public class ExceptionCatcher implements Thread.UncaughtExceptionHandler {
    private volatile static ExceptionCatcher mInstance;
    private Context mContext;
    private Thread.UncaughtExceptionHandler defaultUncaughtEH;

    public ExceptionCatcher(Context context) {
        mContext = context;
    }

    public static ExceptionCatcher getInstance(Context context) {
        if (mInstance == null) {
            synchronized (ExceptionCatcher.class) {
                if (mInstance == null) {
                    mInstance = new ExceptionCatcher(context.getApplicationContext());
                }
            }
        }
        return mInstance;
    }

    public static void makePushPoleDefaultExceptionCatcher(Context context) {
        Thread.UncaughtExceptionHandler currentHandler = Thread.getDefaultUncaughtExceptionHandler();
        if (!(currentHandler instanceof ExceptionCatcher)) {
            Thread.setDefaultUncaughtExceptionHandler(getInstance(context));
        }
    }


    @Override
    public void uncaughtException(Thread thread, Throwable throwable) {
        catchException(thread, throwable);

    }

    public void setDefaultUncaughtExcepHandler(Thread.UncaughtExceptionHandler uceh) {
        if (!(uceh instanceof ExceptionCatcher))
            mInstance.defaultUncaughtEH = uceh;
    }

    public void catchException(Thread thread, Throwable throwable) {
        int percent = KeyStore.getInstance(mContext).getInt(Constants.getVal(Constants.USER_CRASH_REPORT_PERCENT_KEYSTORE), 100);
        if (canSendCrash(percent)) {
            if (isRelatedToPushPoleLib(throwable.getStackTrace())) {
                catchException(throwable, true);
                Logger.debug("exception is related to pushpole, sending it with pushpole tag");
            } else {
                //error is related to user app, report it to her/his sentry project
                catchException(throwable, false);
                Logger.debug("exception is NOT related to pushpole.");
                defaultUncaughtEH.uncaughtException(thread, throwable);
            }
        }
        else{
            Logger.debug("Report rate is set to "+percent+" . This error log will not send to server.");
        }
    }

    private boolean canSendCrash(int percent) {
        boolean send = false;
        if (percent > 0 && percent != 100) {//sending this crash with 'percent' probability
            int rand = new Random(System.currentTimeMillis()).nextInt(100);
            send = (rand <= percent);
        }
        return (send || percent == 100);
    }

    public void catchException(Throwable throwable, boolean pushpoleError) {
        android.util.Log.d("PushPole", "Exception caught " + Logger.getInstance().getHandlers().size());
        android.util.Log.wtf("PushPole", "Exception caught ", throwable);
        Logger.initialize(mContext);
        Log log = createLogFromError(throwable);
        log.setPushPoleError(pushpoleError);
        Logger.fatal(log);
    }

    public void catchException(Throwable throwable) {
        catchException(throwable, true);
    }

    private Log createLogFromError(Throwable throwable) {
        String culprit = null;
        String message = null;

        StackTraceElement goodStack = getGoodStacktrace(throwable.getStackTrace());

        culprit = throwable.getClass().getSimpleName();
        if (throwable.getStackTrace().length > 0) {
            culprit += String.format(": %s.%s", goodStack.getFileName(), goodStack.getLineNumber());
        }

        if (throwable.getMessage() != null) {
            message = throwable.getMessage();
            if (goodStack != null) {
                message += "\n" + String.format(": %s.%s:%s | ", goodStack.getClassName(), goodStack.getMethodName(), goodStack.getLineNumber());
            }
        } else {
            message = throwable.getClass().getName();
            if (goodStack != null) {
                message += "\n" + String.format(": %s.%s", goodStack.getClassName(), goodStack.getMethodName());
            }
        }
        return new Log()
                .setCulprit(culprit)
                .setMessage(message + "\n -- PushPole ExceptionCatcher -- ")
                .setException(throwable)
                .setTimestamp(new Date().getTime());
    }

    private StackTraceElement getGoodStacktrace(StackTraceElement[] elements) {
        for (int i = 0; i < elements.length; i++) {
            String className = elements[i].getClassName();
            if (!className.startsWith("android.")
                    && !className.startsWith("java.")
                    && !className.startsWith("dalvik.")
                    && !className.startsWith("com.android.")
                    && !className.startsWith("com.pushpole.sdk.internal.log.")) {
                return elements[i];
            }
        }

        if (elements.length > 0) {//if above for statement did not return anything, return element[0]
            return elements[0];
        }
        return null;
    }

    public boolean isRelatedToPushPoleLib(StackTraceElement[] elements) {
        for (int i = 0; i < elements.length; i++) {
            String className = elements[i].getClassName();
            if (className.startsWith("com.pushpole.sdk.") || className.startsWith("com.evernote.android.job.")) {
                return true; //it is related to ourlib
            }
        }
        return false;

    }
}
