package com.pushpole.sdk.util;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import com.pushpole.sdk.Constants;
import com.pushpole.sdk.internal.log.Logger;

/***
 * Customized HashMap class
 */
public class Pack extends HashMap<String, Object> {
    /***
     * empty constructor
     */
    public Pack() {
    }

    /***
     * Constructor
     *
     * @param pack the pack
     */
    public Pack(Map<String, Object> pack) {
        super(pack);
    }

    /***
     * convert JSON string to {@link Pack} object
     *
     * @param jsonStr JSON string
     * @return {@link Pack} object
     * @throws InvalidJsonException
     */
    public static Pack fromJson(String jsonStr) throws InvalidJsonException {
        try {
            JSONObject root = new JSONObject(jsonStr);
            return fromJsonObject(root);
        } catch (JSONException | NullPointerException e) {
            throw new InvalidJsonException(e);
        }
    }

    /***
     * convert JSON object to {@link Pack} object
     *
     * @param jsonObject JSON object
     * @return {@link Pack} object
     * @throws InvalidJsonException
     */
    public static Pack fromJsonObject(JSONObject jsonObject) throws InvalidJsonException {
        try {
            Pack pack = new Pack();
            Iterator<String> iter = jsonObject.keys();
            while (iter.hasNext()) {
                String key = iter.next();
                Object value = jsonObject.get(key);

                if (value instanceof JSONObject) {
                    pack.putPack(key, Pack.fromJsonObject((JSONObject) value));
                } else if (value instanceof JSONArray) {
                    pack.putListPack(key, ListPack.fromJsonArray((JSONArray) value));
                } else {
                    pack.put(key, value);
                }
            }
            return pack;
        } catch (JSONException | NullPointerException e) {
            throw new InvalidJsonException(e);
        }
    }

    /***
     * Returns the {@link String} object of the mapping with the specified key.
     *
     * @param key the key
     * @return the value of the mapping with the specified key, or {@code null}
     * if no mapping for the specified key is found.
     */
    public String getString(String key) {
        //return (String) get(key); //to pprevent ClassCastException
        return getString(key, null);
    }

    /***
     * Returns the {@link String} object of the mapping with the specified key.
     *
     * @param key      the key
     * @param defValue the default value
     * @return the value of the mapping with the specified key, or {@code defValue}
     * if no mapping for the specified key is found.
     */
    public String getString(String key, String defValue) {
        Object ret = get(key);
        if (ret == null) {
            return defValue;
        }
        if (!(ret instanceof String)) {
            if(!(key.equals(Constants.getVal(Constants.F_STATUS)) && ret.toString().equals("0")))
                Logger.warning("attempt to use getString on non-string value. key=" + key + " string.valueOf(value): " + ret);
        }
        return String.valueOf(ret);
    }

    /***
     * Returns the integer value of the mapping with the specified key.
     *
     * @param key the key
     * @return the value of the mapping with the specified key, or {@code 0}
     * if no mapping for the specified key is found.
     */
    public int getInt(String key) {
        return getInt(key, 0);
    }

    /***
     * Returns the integer value of the mapping with the specified key.
     *
     * @param key      the key
     * @param defValue the default value
     * @return the value of the mapping with the specified key, or {@code defValue}
     * if no mapping for the specified key is found.
     */
    public int getInt(String key, int defValue) {
        Object ret = get(key);
        if (ret == null) {
            return defValue;
        }
        if(ret instanceof String)
            return Integer.parseInt((String)ret);
        return (Integer) ret;
    }

    /**
     * Returns the long value of the mapping with the specified key.
     * @param key
     * @return the value of the mapping with the specified key, or {@code 0}
     * if no mapping for the specified key is found.
     */
    public long getLong(String key){
        return getLong(key, 0);
    }

    /**
     * Returns the long value of the mapping with the specified key.
     * @param key the key
     * @param defValue the default value returned in case key doesn't exist
     * @return the value of the mapping with the specified key, or {@code defValue}
     * if no mapping for the specified key is found.
     */
    public long getLong(String key, long defValue) {
        Object ret = get(key);
        if (ret == null) {
            return defValue;
        }
        if(ret instanceof String)
            return Long.parseLong((String)ret);
        return (Long) ret;
    }

    /***
     * Returns the boolean value of the mapping with the specified key.
     *
     * @param key the key
     * @return the value of the mapping with the specified key, or {@code null}
     * if no mapping for the specified key is found.
     */
    public boolean getBool(String key) {
        return getBool(key, false);
    }

    /***
     * Returns the boolean value of the mapping with the specified key.
     *
     * @param key      the key
     * @param defValue the default value
     * @return the value of the mapping with the specified key, or {@code defValue}
     * if no mapping for the specified key is found.
     */
    public boolean getBool(String key, boolean defValue) {
        Object ret = get(key);
        if (ret == null) {
            return defValue;
        }
        if (ret instanceof String)
            return Boolean.parseBoolean((String) ret);
        return (Boolean) ret;
    }

    /***
     * Returns the {@link Pack} object of the mapping with the specified key.
     *
     * @param key the key
     * @return the value of the mapping with the specified key, or {@code null}
     * if no mapping for the specified key is found.
     */
    public Pack getPack(String key) {
        return getPack(key, null);
    }

    /***
     * Returns the {@link Pack} object of the mapping with the specified key.
     *
     * @param key      the key
     * @param defValue the default value
     * @return the value of the mapping with the specified key, or {@code defValue}
     * if no mapping for the specified key is found.
     */
    public Pack getPack(String key, Pack defValue) {
        Object ret = get(key);
        if (ret == null) {
            return defValue;
        }
        if(ret instanceof String && ((String)ret).length()== 0) //value is an empty string
            return defValue;
        try{
           return (Pack)ret;
        }catch (Exception e) {
            Logger.error("getPack raised ClassCastException. key is " + key +" value is " + ret);
            return null;
        }
    }

    /***
     * Returns the {@link ListPack} object of the mapping with the specified key.
     *
     * @param key the key
     * @return the value of the mapping with the specified key, or {@code null}
     * if no mapping for the specified key is found.
     */
    public ListPack getListPack(String key) {
        return getListPack(key, null);
    }

    /***
     * Returns the {@link ListPack} object of the mapping with the specified key.
     *
     * @param key      the key
     * @param defValue the default value
     * @return the value of the mapping with the specified key, or {@code defValue}
     * if no mapping for the specified key is found.
     */
    public ListPack getListPack(String key, ListPack defValue) {
        Object ret = get(key);
        if (ret == null) {
            return defValue;
        }
        try {
            return (ListPack) ret;
        }catch (Exception e){
            Logger.error("getListPack raised ClassCastException. key is " + key +" value is " + ret);
            return null;
        }
    }

    /***
     * Maps the specified key to the specified String object.
     *
     * @param key   the key
     * @param value the value
     */
    public void putString(String key, String value) {
        put(key, value);
    }

    /***
     * Maps the specified key to the specified int value.
     *
     * @param key   the key
     * @param value the value
     */
    public void putInt(String key, int value) {
        put(key, value);
    }

    /*
     * Maps the string key to long value
     * @param key
     * @param value
     */
    public void putLong(String key, long value) {
        put(key, value);
    }

    /***
     * Maps the specified key to the specified value.
     *
     * @param key   the key
     * @param value the value
     */
    public void putBool(String key, boolean value) {
        put(key, value);
    }

    /***
     * Maps the specified key to the specified {@link Pack} object
     *
     * @param key  the key
     * @param pack the {@link Pack} object
     */
    public void putPack(String key, Pack pack) {
        put(key, pack);
    }

    /***
     * Maps the specified key to the specified {@link ListPack} object.
     *
     * @param key      the key
     * @param listPack the {@link ListPack} object
     */
    public void putListPack(String key, ListPack listPack) {
        put(key, listPack);
    }

    /***
     * Get the instance
     *
     * @return instance of {@link Pack}
     */
    public Map<String, Object> toMap() {
        return this;
    }

    /***
     * Convert {@link Pack} object to JSON string
     *
     * @return JSON string
     */
    public String toJson() {
        return toJsonObject().toString();
    }

    /***
     * Convert {@link Pack} object to JSON object
     *
     * @return JSON object
     */
    public JSONObject toJsonObject() {
        JSONObject root = new JSONObject();
        for (String key : keySet()) {
            Object value = get(key);
            try {
                if (value instanceof Pack) {
                    root.put(key, ((Pack) value).toJsonObject());
                } else if (value instanceof ListPack) {
                    root.put(key, ((ListPack) value).toJsonArray());
                } else {
                    root.put(key, value);
                }
            } catch (JSONException e) {
                Logger.warning(null, "Error rendering json string from pack");
            }
        }
        return root;
    }
}
