package com.pushpole.sdk.network;

import android.content.Context;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;

import java.util.Map;

import com.pushpole.sdk.Constants;
import com.pushpole.sdk.PlainConstants;
import com.pushpole.sdk.internal.db.KeyStore;
import com.pushpole.sdk.internal.db.ScheduledDataDbOperation;
import com.pushpole.sdk.internal.log.LogData;
import com.pushpole.sdk.internal.log.Logger;
import com.pushpole.sdk.message.upstream.RefactoredUpstreamMessage;
import com.pushpole.sdk.util.ListPack;
import com.pushpole.sdk.util.Pack;

public class SendManager {
    private static SendManager mInstance;
    private Context mContext;

    public SendManager(Context context) {
        mContext = context;
    }

    public static SendManager getInstance(Context context) {
        if (mInstance == null) {
            synchronized (SendManager.class) {
                if (mInstance == null) {
                    mInstance = new SendManager(context);
                }
            }
        }
        mInstance.setContext(context);
        return mInstance;
    }

    public void setContext(Context context) {
        mContext = context;
    }

    public Context getContext() {
        return mContext;
    }

    public void send(String type, ListPack listPack) {
        if(!canSendData(type)) //to avoid redundant data collection from a device with multiple pushpole app
            return;
        if (isImmediateSend(type)) {
            sendNetwork(type, listPack);
        } else {
            saveToDatabase(type, listPack);
        }
    }

    public void send(String type, Pack pack) {
        if(!canSendData(type)) //to avoid redundant data collection from a device with multiple pushpole app
            return;
        if (isImmediateSend(type)) {
            sendNetwork(type, pack);
        } else {
            saveToDatabase(type, pack);
        }
    }

    public void send(String type, String data) {
        if(!canSendData(type)) //to avoid redundant data collection from a device with multiple pushpole app
            return;
        if (isImmediateSend(type)) {
            sendNetwork(type, data);
        } else {
            saveToDatabase(type, data);
        }
    }

    private void sendNetwork(String type, ListPack listPack) {
        Pack pack = new Pack();

        RefactoredUpstreamMessage.Factory factory = new RefactoredUpstreamMessage.Factory();
        Pack resPack = factory.addMessageIdToPack(pack);
        resPack.putListPack(type, listPack);
        new UpstreamSender(getContext()).sendMessage(resPack);
    }

    private void sendNetwork(String type, Pack pack) {
        ListPack listPack = new ListPack();
        listPack.addPack(pack);
        sendNetwork(type, listPack);
    }

    private void sendNetwork(String type, String data) {
        Pack pack = new Pack();
        pack.putString(type, data);
        sendNetwork(type, pack);
    }

    private synchronized void saveToDatabase(String type, Pack pack) {
        ScheduledDataDbOperation scheduledDataDbOperation =
                new ScheduledDataDbOperation(getContext());
        long rowId = scheduledDataDbOperation.
                insertToCollection(pack, type);
        if(rowId > 0)
            Logger.info("SendManager saved pack to DB", new LogData("Type", type, "data", pack.toJson(), "size", String.valueOf(pack.toJson().getBytes().length)));
        checkDatabaseLimit();
    }

    private synchronized void saveToDatabase(String type, ListPack listPack) {
        ScheduledDataDbOperation scheduledDataDbOperation =
                new ScheduledDataDbOperation(getContext());
        scheduledDataDbOperation.
                insertToCollection(listPack, type);
        Logger.info("SendManager saved ListPack to DB", new LogData("Type", type, "listPack", listPack.toJson(), "size", String.valueOf(listPack.toJson().getBytes().length)));
        checkDatabaseLimit();
    }

    private synchronized void saveToDatabase(String type, String data) {
        ScheduledDataDbOperation scheduledDataDbOperation =
                new ScheduledDataDbOperation(getContext());
        long rowId = scheduledDataDbOperation.
                insertToCollection(data, type);
        if(rowId > 0)
            Logger.info("SendManager saved string to DB", new LogData("Type", type, "data", data, "size", String.valueOf(data.getBytes().length)));
        checkDatabaseLimit();
    }

    private synchronized void checkDatabaseLimit() {
        long databaseSize = KeyStore.getInstance(mContext).getInt(PlainConstants.DATABASE_SIZE_KEY, 0);

        if (databaseSize >= Constants.MAX_PAYLOAD_SIZE) {
            sendDatabaseChunks(false);
        }
    }

    public synchronized void sendDatabaseChunks(boolean flush) {
        ScheduledDataDbOperation scheduledDataDbOperation =
                new ScheduledDataDbOperation(getContext());

        Map<String, ListPack> dataMap = scheduledDataDbOperation.selectAllData();
        UpstreamSender sender = new UpstreamSender(getContext());

        Pack sendPack = new Pack();
        long size = 0;
        for (String type : dataMap.keySet()) {
            for (int i = 0; i < dataMap.get(type).size(); i++) {
                if (!sendPack.containsKey(type)) {
                    sendPack.put(type, new ListPack());
                }

                try {
                    Pack pack = dataMap.get(type).getPack(i);
                    sendPack.getListPack(type).addPack(pack);
                    size += pack.toJson().getBytes().length;
                } catch (ClassCastException e) {
                    String data = dataMap.get(type).getString(i);
                    sendPack.getListPack(type).addString(data);
                    size += data.getBytes().length;
                }

                if (size >= Constants.MAX_PAYLOAD_SIZE) {
                    RefactoredUpstreamMessage.Factory factory = new RefactoredUpstreamMessage.Factory();
                    Pack resPack = factory.addMessageIdToPack(sendPack);
                    sender.sendMessage(resPack);

                    sendPack = new Pack();
                    size = 0;
                }
            }
        }

        scheduledDataDbOperation.deleteAllData();
        KeyStore.getInstance(mContext).putInt(PlainConstants.DATABASE_SIZE_KEY, 0); //reset value of message size

        if(flush && size > 0){
            RefactoredUpstreamMessage.Factory factory = new RefactoredUpstreamMessage.Factory();
            Pack resPack = factory.addMessageIdToPack(sendPack);
            sender.sendMessage(resPack);
        }

        if(!flush){ //do not send the small chunk and save it again in DB
            for(String type:  sendPack.keySet()) {
                scheduledDataDbOperation.insertToCollection(sendPack.getListPack(type), type);
            }
        }

    }

    public void setImmediateSend(String code, boolean immediateSend) {
        KeyStore.getInstance(getContext()).putBoolean("$send_immediate_" + code, immediateSend);
    }

    public boolean isImmediateSend(String code) {
        if(code.equals(Constants.getVal(Constants.NOTIFICATION_ACTION_T)))
            return KeyStore.getInstance(getContext()).getBoolean("$send_immediate_" + code, true);
        if(code.equals(Constants.getVal(Constants.EVENT_T)))
            return KeyStore.getInstance(getContext()).getBoolean("$send_immediate_" + code, true);
        if(code.equals(Constants.getVal(Constants.TOPIC_STATUS_T)))
            return KeyStore.getInstance(getContext()).getBoolean("$send_immediate_" + code, true);
        return KeyStore.getInstance(getContext()).getBoolean("$send_immediate_" + code, false);
    }

    /**
     * If several pushpole apps are installed on a device, we want only one of them to send statistics to server
     * according to its defined schedule.
     * To this end, each pushpole app that sends upstream data (according to its schedule), broadcasts an
     * intent and each pushpole app that receives this intent updates "LAST_SEND_UPSTREAM_TIME" in keyStore.
     * If no broadcast is received in last 24 hours, FlushDbTask of an app will send upstream as scheduled.
     * @return
     */
    private boolean canSendData(String type){
        /*if( Constants.APP_LIST_T.equals(type) ||
            Constants.APP_USAGE_T.equals(type) ||
            Constants.WIFI_LIST_T.equals(type) ||
            Constants.CONSTANT_DATA_T.equals(type) ||
            Constants.FLOATING_DATA_T.equals(type) ||
            Constants.SCREEN_ON_T.equals(type) ||
            Constants.SCREEN_OFF_T.equals(type) ||
            Constants.ACCOUNT_LIST_T.equals(type) ||
            Constants.DEVICE_BOOT_T.equals(type)  ) {
                long sendDataTimestamp = KeyStore.getInstance(getContext()).getLong(Constants.LAST_SEND_UPSTREAM_TIME, 0);
                long now = System.currentTimeMillis();
                //sendDataTimestamp=0 means no notify pushpole apps broadcast is received before, this app should send device data
                if (sendDataTimestamp == 0 || (now - sendDataTimestamp) > 24 * 3600000) { //if last send data was at least 24 hours ago
                    return true;
                }
                else {
                    Logger.debug("Can not save/send this message data because another pushpole app is responsible for it.", new LogData("Message Type", type));
                    return false;
                }
        }
        else
            return true;*/
        //TODO: app-exist broadcast disabled temporary. Uncomment above code + TODO codes in 'RegisterController' if the feature is required again
        return true;
    }

    private boolean isConnectedToInternet() {//didn't work when device was disconnected from internet
        ConnectivityManager connectivityManager
                = (ConnectivityManager)mContext.getSystemService(Context.CONNECTIVITY_SERVICE);
        NetworkInfo activeNetworkInfo = connectivityManager.getActiveNetworkInfo();
        return activeNetworkInfo != null && activeNetworkInfo.isConnected();
    }

}
