package com.pushpole.sdk.message.upstream;

import com.pushpole.sdk.Constants;
import com.pushpole.sdk.util.Crypto;
import com.pushpole.sdk.util.Pack;

import static com.pushpole.sdk.Constants.F_ORIGINAL_MESSAGE_ID;
import static com.pushpole.sdk.Constants.F_STATUS;
import static com.pushpole.sdk.Constants.F_RESPONSE_BUTTON_ID;
import static com.pushpole.sdk.Constants.TIMESTAMP;

/***
 * A class that represent upstream message for notification,
 * generate appropriate message to pushpole-server when notification response action occur,
 * response actions are : RESPONSE_ACTION_CLICK, RESPONSE_ACTION_DISMISS
 * the generated message must contains same ID as related {@link com.pushpole.sdk.message.downstream.NotificationDownstreamMessage}
 */
public class NotificationUpstreamMessage extends UpstreamMessage {
    private String mOriginalMessageId;
    private String mResponseAction;
    private Integer mResponseButtonId;


    /***
     * return original message ID
     *
     * @return the ID
     */
    public String getOriginalMessageId() {
        return mOriginalMessageId;
    }

    /***
     * set original message ID
     *
     * @param messageId the ID
     */
    public void setOriginalMessageId(String messageId) {
        mOriginalMessageId = messageId;
    }

    /***
     * return response action
     *
     * @return the action
     */
    public String getResponseAction() {
        return mResponseAction;
    }

    /***
     * set response action
     *
     * @param responseAction the action
     */
    public void setResponseAction(String responseAction) {
        mResponseAction = responseAction;
    }

    /***
     * return response button ID
     *
     * @return the button ID
     */
    public int getResponseButtonId() {
        return mResponseButtonId;
    }

    /***
     * set response button ID
     *
     * @param responseButtonId the ID
     */
    public void setResponseButtonId(Integer responseButtonId) {
        mResponseButtonId = responseButtonId;
    }

    /***
     * store {@link NotificationUpstreamMessage} data in to a {@link Pack} object
     *
     * @return the {@link Pack} object
     */
    @Override
    public Pack toPack() {
        Pack pack = new Pack();
        pack.putString(Constants.getVal(F_ORIGINAL_MESSAGE_ID), mOriginalMessageId);
        //pack.putString(F_RESPONSE_ACTION, mResponseAction);
        pack.putString(Constants.getVal(F_STATUS), mResponseAction);
        if (mResponseButtonId != null) {
            pack.putString(Constants.getVal(F_RESPONSE_BUTTON_ID), String.valueOf(mResponseButtonId));
        }
        pack.putString(Constants.getVal(TIMESTAMP), String.valueOf(System.currentTimeMillis()));
        return pack;
    }

    /***
     * return message type
     *
     * @return {@code Type.NOTIFICATION}
     */
    @Override
    public UpstreamMessage.Type getMessageType() {
        return Type.NOTIFICATION;
    }

    /***
     * Factory class to generate {@link NotificationUpstreamMessage} instance
     */
    public static class Factory extends UpstreamMessageFactory {
        /***
         * generate {@link NotificationUpstreamMessage} instance
         *
         * @param data the pack that contains data for populating message
         * @return {@link NotificationUpstreamMessage} instance
         */
        @Override
        public UpstreamMessage buildMessage(Pack data) {
            NotificationUpstreamMessage message = new NotificationUpstreamMessage();
            populateMessage(message, data);

            message.setOriginalMessageId(data.getString(Constants.getVal(F_ORIGINAL_MESSAGE_ID), ""));
            message.setResponseAction(data.getString(Constants.getVal(F_STATUS), ""));
            Integer responseId = null;
            try {
                responseId = Integer.parseInt(data.getString(Constants.getVal(F_RESPONSE_BUTTON_ID), ""));
            } catch (NumberFormatException e) {
                responseId = null;
            }
            message.setResponseButtonId(responseId);
            return message;
        }

        /***
         * generate {@link NotificationUpstreamMessage} instance
         *
         * @param originalMessageId the ID
         * @param responseAction    the action
         * @param responseButton    the button ID
         * @return
         */
        public NotificationUpstreamMessage buildMessage(String originalMessageId, String responseAction, Integer responseButton) {
            NotificationUpstreamMessage message = new NotificationUpstreamMessage();
            populateMessage(message);
            message.setOriginalMessageId(originalMessageId);
            message.setResponseAction(responseAction);
            message.setResponseButtonId(responseButton);
            return message;
        }
    }
}
