package com.pushpole.sdk.message.upstream;

import android.content.Context;

import com.pushpole.sdk.Constants;
import com.pushpole.sdk.internal.log.Logger;
import com.pushpole.sdk.util.InsufficientPermissionsException;
import com.pushpole.sdk.util.NetworkHelper;
import com.pushpole.sdk.util.Pack;

/***
 * A class that represent upstream message for location message
 * the message contains data about client IP address, ISP name and network type
 */
public class LocationMessage extends UpstreamMessage {
    private boolean mAppOpened;
    private String mIpAddress;
    //private String mIsp;
    private String mNetworkType;

    /***
     * store {@link LocationMessage} data in to a {@link Pack} object
     *
     * @return the {@link Pack} object
     */
    @Override
    public Pack toPack() {
        Pack pack = super.toPack();
        pack.putString(Constants.getVal(Constants.OPEN_APP_T), String.valueOf(mAppOpened));
        if (mIpAddress != null) {
            pack.putString(Constants.getVal(Constants.F_IP_ADDRESS), mIpAddress);
        }

        if (mNetworkType != null) {
            pack.putString(Constants.getVal(Constants.F_NETWORK_TYPE), mNetworkType);
        }
        return pack;
    }

    /***
     * return message type
     *
     * @return {@code Type.QUERY_LOCATION}
     */
    @Override
    public Type getMessageType() {
        return Type.OPEN_APP;
    }

    /***
     * Factory class to generate {@link LocationMessage} instance
     */
    public static class Factory extends UpstreamMessageFactory {

        /***
         * generate {@link LocationMessage} instance
         *
         * @param pack the pack that contains data for populating message
         * @return the {@link LocationMessage} message
         */
        @Override
        public LocationMessage buildMessage(Pack pack) {
            LocationMessage message = new LocationMessage();
            populateMessage(message, pack);

            String appOpenedStr = pack.getString(Constants.getVal(Constants.OPEN_APP_T), "false");
            message.mAppOpened = "true".equals(appOpenedStr);

            message.mIpAddress = pack.getString(Constants.getVal(Constants.F_IP_ADDRESS), null);
            message.mNetworkType = pack.getString(Constants.getVal(Constants.F_NETWORK_TYPE));

            return message;
        }

        /**
         * Must be called from worker thread, contains a http request for obtaining public ip info.
         *
         * @param appOpened Is this message being built as a result of app open event
         * @return new LocationMessages
         */
        public LocationMessage buildMessage(Context context, boolean appOpened) {
            LocationMessage message = new LocationMessage();
            populateMessage(message);
            message.mAppOpened = appOpened;

            NetworkHelper.PublicIpInfo info = NetworkHelper.getPublicIpAddressInfo();
            if (info != null) {
                message.mIpAddress = info.ip;
            } else {
                Logger.warning("Getting public ip info failed");
            }

            try {
                message.mNetworkType = NetworkHelper.getNetworkTypeName(context);
            } catch (InsufficientPermissionsException e) {
            }

            return message;
        }
    }
}
