package com.pushpole.sdk.internal.db;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.SQLException;
import android.database.sqlite.SQLiteDatabase;

import java.util.HashMap;
import java.util.Map;

import com.pushpole.sdk.PlainConstants;
import com.pushpole.sdk.internal.log.LogData;
import com.pushpole.sdk.internal.log.Logger;
import com.pushpole.sdk.util.InvalidJsonException;
import com.pushpole.sdk.util.ListPack;
import com.pushpole.sdk.util.Pack;

/**
 * Created on 16-06-26, 5:42 PM.
 * A class for CRUD operation on a database table which is used to save collected data
 *
 * @author Akram Shokri
 */
public class ScheduledDataDbOperation {

    // Database fields
    private SQLiteDatabase database;
    private DatabaseHelper dbHelper;
    private Context mContext;


    public ScheduledDataDbOperation(Context context) {
        dbHelper = DatabaseHelper.getInstance(context);
        mContext = context;
        open();
    }

    private void open() throws SQLException {
        if (database == null || (database != null && !database.isOpen()))
            database = dbHelper.getWritableDatabase();
    }

    public void close() {
        dbHelper.close();
    }

    public long insertToCollection(Pack pack, String type) {
        if (pack == null)
            return 0;
        String jsonData = pack.toJson();
        ContentValues values = new ContentValues();
        values.put(DatabaseHelper.COLUMN_COLL_JSON, jsonData);
        values.put(DatabaseHelper.COLUMN_COLL_TYPE, type);
        long rowId = database.insert(DatabaseHelper.TABLE_COLLECTION, null, values);
        if (rowId == -1) {
            Logger.warning("Inserting data to CollectionDB failed.", new LogData("collection-type", type, "Data", jsonData));
        } else
            updateDatabaseSize(jsonData.getBytes().length);
        return rowId;
    }

    public long insertToCollection(ListPack listPack, String type) {
        if(listPack == null)
            return 0;
        database.beginTransaction();
        for (int i = 0; i < listPack.size(); i++) {
            String jsonData;
            try {
                jsonData = listPack.getPack(i).toJson();
            } catch (ClassCastException e) {
                jsonData = listPack.getString(i);
            }
            ContentValues values = new ContentValues();
            values.put(DatabaseHelper.COLUMN_COLL_JSON, jsonData);
            values.put(DatabaseHelper.COLUMN_COLL_TYPE, type);
            long rowId = database.insert(DatabaseHelper.TABLE_COLLECTION, null, values);
            if (rowId == -1) {
                Logger.warning("Insert Transaction: Inserting data to CollectionDB failed.", new LogData("collection-type", type, "Data", jsonData));
            } else
                updateDatabaseSize(jsonData.getBytes().length);
        }
        database.setTransactionSuccessful();
        database.endTransaction();
        return 1;
    }

    public long insertToCollection(String data, String type) {
        if(data == null || data.isEmpty())
            return 0;
        ContentValues values = new ContentValues();
        values.put(DatabaseHelper.COLUMN_COLL_JSON, data);
        values.put(DatabaseHelper.COLUMN_COLL_TYPE, type);
        long rowId = database.insert(DatabaseHelper.TABLE_COLLECTION, null, values);
        updateDatabaseSize(data.getBytes().length);
        return rowId;
    }

    private void updateDatabaseSize(int msgSize) {
        int size = msgSize + KeyStore.getInstance(mContext).getInt(PlainConstants.DATABASE_SIZE_KEY, 0);
        KeyStore.getInstance(mContext).putInt(PlainConstants.DATABASE_SIZE_KEY, size);
    }

    public String findDataById(long id, String columnName) {
        String query = "select * from " + DatabaseHelper.TABLE_COLLECTION + " where " + DatabaseHelper.COLUMN_ID + " = " + id;
        Cursor cursor = database.rawQuery(query, null);
        String data = "";
        try {
            if (cursor.getCount() > 0) {
                cursor.moveToFirst();
                data = cursor.getString(cursor.getColumnIndex(DatabaseHelper.COLUMN_COLL_JSON));
            }
        } finally {
            cursor.close();
        }
        return data;
    }

    public Map<String, ListPack> selectAllData() {
        String query = "select * from " + DatabaseHelper.TABLE_COLLECTION;
        Cursor cursor = database.rawQuery(query, null);
        String data = "";
        String type = "";
        Pack allDataPack = new Pack();

        Map<String, ListPack> dataMap = new HashMap<>();

        try {
            if (!cursor.moveToFirst())
                return dataMap;

            for (int i = 0; i < cursor.getCount(); i++) { //for each row of data in table
                data = cursor.getString(cursor.getColumnIndex(DatabaseHelper.COLUMN_COLL_JSON));
                type = cursor.getString(cursor.getColumnIndex(DatabaseHelper.COLUMN_COLL_TYPE));

                if (!dataMap.containsKey(type)) {
                    dataMap.put(type, new ListPack());
                }

                Pack dataPack = null;
                try {
                    dataPack = Pack.fromJson(data);
                    dataMap.get(type).addPack(dataPack);
                } catch (InvalidJsonException e) {
                    dataMap.get(type).addString(data);
                }

                cursor.moveToNext();
            }
            return dataMap;
        } finally {
            cursor.close();
        }
    }

    public void deleteAllData() {
        int result = database.delete(DatabaseHelper.TABLE_COLLECTION, null, null); //deletes all rows
        database.execSQL("VACUUM"); //TODO: added for release 1.2.0, needs test
    }
}
