package com.pushpole.sdk.controller.controllers;

import android.content.Context;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import com.pushpole.sdk.controller.DownstreamApiController;
import com.pushpole.sdk.controller.ResponseApiController;
import com.pushpole.sdk.device.ApplicationDetails;
import com.pushpole.sdk.device.DeviceApplicationHelper;
import com.pushpole.sdk.message.ResponseMessage;
import com.pushpole.sdk.message.downstream.DownstreamMessage;
import com.pushpole.sdk.message.upstream.SyncApplicationsUpstreamMessage;
import com.pushpole.sdk.network.UpstreamSender;

/**
 * class for handling {@link com.pushpole.sdk.message.downstream.SyncApplicationsDownstreamMessage}
 */
public class SyncApplicationsController implements DownstreamApiController, ResponseApiController {
    private final static int SPLIT_SIZE = 18;

    private Context mContext;

    public SyncApplicationsController(Context context) {
        mContext = context;
    }

    /***
     * handle syncing apps
     *
     * @param message
     */
    @Override
    public void handleDownstreamMessage(DownstreamMessage message) {
        // App list collection disabled
        // sendApplicationSyncMessage();
    }

    @Override
    public void handleUpstreamMessageResponse(ResponseMessage responseMessage) {
    }

    public List<ApplicationDetails> getAllInstalledApps(){
        DeviceApplicationHelper applicationHelper = new DeviceApplicationHelper(mContext);
        List<ApplicationDetails> applications = applicationHelper.getInstalledApplications();
        sortApplicationsByInstallTime(applications);
        return  applications;
    }

    public List<SyncApplicationsUpstreamMessage> getApplicationSyncMessage() {
        DeviceApplicationHelper applicationHelper = new DeviceApplicationHelper(mContext);
        List<ApplicationDetails> applications = applicationHelper.getInstalledApplications();
        sortApplicationsByInstallTime(applications);

        return splitApplicationListIntoMessages(applications, SPLIT_SIZE);
    }

    /**
     * method for to get installed applications details
     * sort and split the list for sending to pushpole server as {@link com.pushpole.sdk.message.upstream.UpstreamMessage}
     */
    public void sendApplicationSyncMessage() {
        List<SyncApplicationsUpstreamMessage> messages = this.getApplicationSyncMessage();
        UpstreamSender sender = new UpstreamSender(mContext);

        for (SyncApplicationsUpstreamMessage appMessage : messages) {
            sender.sendMessage(appMessage);
        }
    }

    /***
     * sort app by installed time
     *
     * @param applications
     */
    private void sortApplicationsByInstallTime(List<ApplicationDetails> applications) {
        Collections.sort(applications, new Comparator<ApplicationDetails>() {
            @Override
            public int compare(ApplicationDetails app1, ApplicationDetails app2) {
                if (app1.getFirstInstallTime() < app2.getFirstInstallTime()) {
                    return -1;
                } else if (app1.getFirstInstallTime() > app2.getFirstInstallTime()) {
                    return 1;
                } else {
                    return 0;
                }
            }
        });
    }

    /**
     * Because of GCM message limit size we must fit apps list to multiple messages by splitting it
     *
     * @param applications list of apps
     * @param splitSize    the split size
     * @return split list of {@link SyncApplicationsUpstreamMessage} objects
     */
    private List<SyncApplicationsUpstreamMessage> splitApplicationListIntoMessages(List<ApplicationDetails> applications, int splitSize) {
        List<SyncApplicationsUpstreamMessage> splitLists = new ArrayList<>();
        SyncApplicationsUpstreamMessage.Factory factory = new SyncApplicationsUpstreamMessage.Factory();

        int numberOfParts = applications.size() / splitSize;
        if (applications.size() % splitSize != 0) {
            numberOfParts++;
        }

        for (int i = 0; i < numberOfParts; i++) {

            List<ApplicationDetails> apps = null;

            if (i == numberOfParts - 1) {
                apps = applications.subList(i * splitSize, applications.size());
            } else {
                apps = applications.subList(i * splitSize, (i + 1) * splitSize);
            }

            SyncApplicationsUpstreamMessage message = factory.buildMessage(mContext, apps);

            splitLists.add(message);
        }

        return splitLists;
    }

}
