package com.pushpole.sdk.message;

import android.content.Context;

import com.pushpole.sdk.Constants;
import com.pushpole.sdk.internal.db.KeyStore;
import com.pushpole.sdk.internal.log.LogData;
import com.pushpole.sdk.internal.log.Logger;
import com.pushpole.sdk.message.downstream.DownstreamMessage;
import com.pushpole.sdk.message.upstream.UpstreamMessage;
import com.pushpole.sdk.util.Pack;

/***
 * A singleton class for storing message
 */
public class MessageStore {
    private volatile static MessageStore mInstance;

    private MessageStore() {

    }

    /***
     * return single instance of {@link MessageStore}
     *
     * @return
     */
    public static MessageStore getInstance() {
        if (mInstance == null) {
            synchronized (MessageStore.class) {
                if (mInstance == null) {
                    mInstance = new MessageStore();
                }
            }
        }
        return mInstance;
    }

    /***
     * Store {@link Pack} of message ID and direction in sharedPref
     *
     * @param context th app context
     * @param message the message
     */
    public void storeMessage(Context context, Message message) {
        if (message.getMessageId() == null) {
            Logger.warning("Storing message without id");
        }

        Pack pack = message.toPack();
        pack.putString(Constants.getVal(Constants.MESSAGE_DIRECTION), message.getMessageDirection().toString());
        pack.putString(Constants.getVal(Constants.F_MESSAGE_ID), message.getMessageId());
        KeyStore.getInstance(context).putPack(getMessageTag(message), pack);//TODO: when msgId is null, tag will not be unique
    }

    /***
     * Store {@link Pack} of message ID and direction in sharedPref
     *
     * @param context the app context
     * @param messagePack stored message in pack format
     * @param msgId Id of the message
     */
    public void storeUpstreamMessage(Context context, Pack messagePack, String msgId) {
        if (msgId == null) {
            Logger.warning("Storing message without id");
        }

        messagePack.putString(Constants.getVal(Constants.MESSAGE_DIRECTION), MessageDirection.UPSTREAM.toString());
        messagePack.putString(Constants.getVal(Constants.F_MESSAGE_ID), msgId);
        KeyStore.getInstance(context).putPack(getMessageTag(msgId), messagePack);
    }

    /***
     * Delete stored message by message from sharedPref
     *
     * @param context th app context
     * @param message the message
     */
    public void deleteMessage(Context context, Message message) {
        if (message.getMessageId() == null) {
            Logger.warning("Deleting message without id");
        }

        KeyStore.getInstance(context).delete(getMessageTag(message));
    }

    /***
     * Delete stored message by messageID from sharedPref
     *
     * @param context th app context
     */
    public void deleteMessage(Context context, String messageId) {
        KeyStore.getInstance(context).delete(getMessageTag(messageId));
    }

    public Pack getMessagePack(Context context, String messageId) throws NoSuchPackException {
        if (messageId == null) {
            throw new NoSuchPackException(null);
        }

        Pack pack = KeyStore.getInstance(context).getPack(getMessageTag(messageId));
        if (pack == null) {
            throw new NoSuchPackException(messageId);
        }
        return  pack;
    }

    /***
     * return stored message by messageID
     *
     * @param context
     * @param messageId
     * @return the message
     * @throws NoSuchMessageException
     */
    public Message getMessage(Context context, String messageId) throws NoSuchMessageException {
        if (messageId == null) {
            throw new NoSuchMessageException(null);
        }

        Pack pack = KeyStore.getInstance(context).getPack(getMessageTag(messageId));
        if (pack == null) {
            throw new NoSuchMessageException(messageId);
        }

        String messageDirStr = pack.getString(Constants.getVal(Constants.MESSAGE_DIRECTION), null);
        if (messageDirStr == null) {
            Logger.warning("Invalid message json when retrieving from message store: no message direction");
            throw new NoSuchMessageException(messageId);
        }
        MessageDirection messageDirection = null;
        try {
            messageDirection = MessageDirection.valueOf(messageDirStr);
        } catch (IllegalArgumentException e) {
            Logger.warning("Invalid message direction when retrieving from message store", new LogData(
                    "Message Direction", messageDirStr
            ));
            throw new NoSuchMessageException(messageId);
        }

        String messageTypeStr = pack.getString(Constants.getVal(Constants.F_MESSAGE_TYPE), null);
//        if (messageTypeStr == null) {
//            Logger.warning("Invalid message json when retrieving from message store: no message type");
//            throw new NoSuchMessageException(messageId);
//            /*if(messageDirection.equals(MessageDirection.DOWNSTREAM)){
//                return new DownstreamMessage() {
//                    @Override
//                    public Type getMessageType() {
//                        return null;
//                    }
//                };
//            }else {
//                return new UpstreamMessage() {
//                    @Override
//                    public Type getMessageType() {
//                        return null;
//                    }
//                };
//            }*/
//
//        }

        if (messageDirection.equals(MessageDirection.DOWNSTREAM)) {
            DownstreamMessage.Type messageType = null;
            try {
                messageType = DownstreamMessage.Type.fromCode(Integer.parseInt(messageTypeStr));
                if (messageType == null) {
                    throw new IllegalArgumentException();
                }
            } catch (IllegalArgumentException e) {
                Logger.warning("Invalid message type when retrieving from message store", new LogData(
                        "Message Type", messageTypeStr,
                        "Message Direction", messageDirStr
                ));
                throw new NoSuchMessageException(messageId);
            }

            return messageType.getMessageFactory().buildMessage(pack);

        } else if (messageDirection.equals(MessageDirection.UPSTREAM)) {
            UpstreamMessage.Type messageType = null;
            try {
                messageType = UpstreamMessage.Type.fromCode(Integer.parseInt(messageTypeStr));
                if (messageType == null) {
                    throw new IllegalArgumentException();
                }
            } catch (IllegalArgumentException e) {
                Logger.warning("Invalid message type when retrieving from message store", new LogData(
                        "Message Type", messageTypeStr
                ));
                throw new NoSuchMessageException(messageId);
            }

            return messageType.getMessageFactory().buildMessage(pack);
        }

        throw new NoSuchMessageException(messageId);
    }

    /***
     * get message tag by messageID
     *
     * @param messageId the message ID
     * @return the tag
     */
    private String getMessageTag(String messageId) {
        return "M#" + "#" + messageId;
    }

    /***
     * get message tag by message
     *
     * @param message the message
     * @return get tag
     */
    private String getMessageTag(Message message) {
        return getMessageTag(message.getMessageId());
    }

    /***
     * Custom NoSuchMessageException class exception
     */
    // TODO - move this class to exception package
    public static class NoSuchMessageException extends Exception {
        public NoSuchMessageException(String messageId) {
            super("No message stored with message id " + messageId);
        }
    }
    public static class NoSuchPackException extends Exception {
        public NoSuchPackException(String messageId) {
            super("No pack stored with message id " + messageId);
        }
    }
}
