package com.pushpole.sdk.device;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.SharedPreferences;
import android.os.Build;
import android.provider.Settings;
import android.telephony.TelephonyManager;

import com.google.android.gms.ads.identifier.AdvertisingIdClient;
import com.google.android.gms.ads.identifier.AdvertisingIdClient;
import com.google.android.gms.common.GooglePlayServicesNotAvailableException;
import com.google.android.gms.common.GooglePlayServicesRepairableException;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.UUID;

import com.pushpole.sdk.Constants;
import com.pushpole.sdk.internal.db.KeyStore;
import com.pushpole.sdk.internal.log.Logger;
import com.pushpole.sdk.util.PermissionChecker;

public class DeviceIDHelper {
    private final static String NOIMEI_PREFIX = "pid_";

    private static String advertisingId = null;
    private Context mContext;

    /**
     * Constructs a new {@code DeviceIDHelper}.
     *
     * @param context the context to use
     */
    public DeviceIDHelper(Context context) {
        mContext = context;
    }

    public static boolean imeiAccessRefused(Context context) {
        return KeyStore.getInstance(context).getInt(Constants.getVal(Constants.KEY_STORE_USE_IMEI_USER_ANSWER), -1) == 0;
    }

    public static boolean imeiAccessGranted(Context context) {
        return KeyStore.getInstance(context).getInt(Constants.getVal(Constants.KEY_STORE_USE_IMEI_USER_ANSWER), -1) == 1;
    }

    /**
     * Retrieves a unique device id for the current device.
     * <p>
     * Uses the device IMEI code if available and sufficient permissions are given in the application.
     * In the case where it can't obtain the IMEI code, Android ID is used. If neither are present,
     * a random UUID is generated and used. The generated ID is persisted to SharedPreferences so it
     * will remain the same (unless application data is cleared).
     * <p>
     * If the device IMEI code was used as the id, the id will be prefixed with "NO-IMEI_".
     *
     * @return the device unique id
     */
    public String getDeviceId() {
        String deviceId = null;

//        try {
//            deviceId = getIMEI();
//        } catch (InsufficientPermissionsException e) {
//        }

        if (deviceId == null) {
            String customId = getCustomId();
            if (customId.length() > 16) {
                deviceId = NOIMEI_PREFIX + customId.substring(4, 16);
            } else {
                deviceId = NOIMEI_PREFIX + customId.substring(0, 12);
            }
        }

        return deviceId;
    }

    public boolean hasReadPhoneStatePermission() {
        return PermissionChecker.hasPermission(mContext, PermissionChecker.READ_PHONE_STATE);
    }

    @SuppressLint("MissingPermission")
    public String getIMEI() {
        if (imeiAccessGranted(mContext) && hasReadPhoneStatePermission()) {
            TelephonyManager telephonyManager = (TelephonyManager) mContext.
                    getSystemService(Context.TELEPHONY_SERVICE);
            return telephonyManager.getDeviceId();
        }
        return "";
    }

    public String getCustomId() {
        return new DeviceUUIDFactory(mContext).getDeviceUuid().toString();
    }

    public String getAndroidId() {
        return Settings.Secure.getString(mContext.getContentResolver(), Settings.Secure.ANDROID_ID);
    }

    public String getSecondSimIMEI() {
        if (DeviceIDHelper.imeiAccessGranted(mContext) && PermissionChecker.hasPermission(mContext, PermissionChecker.READ_PHONE_STATE)) {
            TelephonyInfo telephonyInfo = TelephonyInfo.getInstance(mContext);
            return telephonyInfo.getImsiSIM2();
        }
        return null;
    }

    public static String getAdvertisingId(Context context) throws GooglePlayServicesNotAvailableException, IOException, GooglePlayServicesRepairableException {
        if (advertisingId != null) {
            return advertisingId;
        }
        advertisingId = AdvertisingIdClient.getAdvertisingIdInfo(context).getId();
        return advertisingId;
    }

    /**
     * Helper class for retrieving device UUID
     *
     * @author Some guy in Stack Overflow
     * @see <a href="here">http://stackoverflow.com/questions/2785485/is-there-a-unique-android-device-id</a>
     */
    private static class DeviceUUIDFactory {

        protected static final String PREFS_FILE = "device_id.xml";
        protected static final String PREFS_DEVICE_ID = "device_id";
        protected volatile static UUID uuid = null;

        @SuppressLint("MissingPermission")
        public DeviceUUIDFactory(Context context) {
            if (uuid == null) {
                synchronized (DeviceUUIDFactory.class) {
                    if (uuid == null) {
                        final SharedPreferences prefs = context
                                .getSharedPreferences(PREFS_FILE, 0);
                        final String id = prefs.getString(PREFS_DEVICE_ID, null);
                        if (id != null) {
                            // Use the ids previously computed and stored in the
                            // prefs file
                            uuid = UUID.fromString(id);
                        }//For android 8+ where android-id is changed per installation and is not unique per device
                        else if (android.os.Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                            try {
                                String seed = getAdvertisingId(context);

                                uuid = (seed != null && !seed.isEmpty()) ?
                                        UUID.nameUUIDFromBytes(seed.getBytes("utf8")) :
                                        UUID.randomUUID();

                            } catch (Exception e) {
                                /*
                                    If PushPole Id is being generated in the main thread, we can't access
                                    the advertisement id so use a random id instead
                                */
                                uuid = UUID.randomUUID();
//                                Logger.error("Failed to build proper PushPole ID, assigning random ID. Error: " + e.getMessage());
                            }

                            prefs.edit()
                                    .putString(PREFS_DEVICE_ID, uuid.toString())
                                    .apply();
                        } else {
                            final String androidId = Settings.Secure.getString(
                                    context.getContentResolver(), Settings.Secure.ANDROID_ID);
                            // Use the Android ID unless it's broken, in which case
                            // fallback on deviceId,
                            // unless it's not available, then fallback on a random
                            // number which we store to a prefs file
                            try {
                                if (!"9774d56d682e549c".equals(androidId)) {
                                    uuid = UUID.nameUUIDFromBytes(androidId
                                            .getBytes("utf8"));
                                } else if (PermissionChecker.hasPermission(context,
                                        PermissionChecker.READ_PHONE_STATE)) {
                                    final String deviceId = (
                                            (TelephonyManager) context
                                                    .getSystemService(Context.TELEPHONY_SERVICE))
                                            .getDeviceId();
                                    uuid = deviceId != null ? UUID
                                            .nameUUIDFromBytes(deviceId
                                                    .getBytes("utf8")) : UUID
                                            .randomUUID();
                                } else {
                                    uuid = UUID.randomUUID();
                                }
                            } catch (UnsupportedEncodingException e) {
                                throw new RuntimeException(e);
                            }
                            // Write the value out to the prefs file
                            prefs.edit()
                                    .putString(PREFS_DEVICE_ID, uuid.toString())
                                    .apply();
                        }
                    }
                }
            }
        }

        /**
         * Returns a unique UUID for the current android device. As with all UUIDs,
         * this unique ID is "very highly likely" to be unique across all Android
         * devices. Much more so than ANDROID_ID is.
         * <p>
         * The UUID is generated by using ANDROID_ID as the base key if appropriate,
         * falling back on TelephonyManager.getDeviceID() if ANDROID_ID is known to
         * be incorrect, and finally falling back on a random UUID that's persisted
         * to SharedPreferences if getDeviceID() does not return a usable value.
         * <p>
         * In some rare circumstances, this ID may change. In particular, if the
         * device is factory reset a new device ID may be generated. In addition, if
         * a user upgrades their phone from certain buggy implementations of Android
         * 2.2 to a newer, non-buggy version of Android, the device ID may change.
         * Or, if a user uninstalls your app on a device that has neither a proper
         * Android ID nor a Device ID, this ID may change on reinstallation.
         * <p>
         * Note that if the code falls back on using TelephonyManager.getDeviceId(),
         * the resulting ID will NOT change after a factory reset. Something to be
         * aware of.
         * <p>
         * Works around a bug in Android 2.2 for many devices when using ANDROID_ID
         * directly.
         *
         * @return a UUID that may be used to uniquely identify your device for most
         * purposes.
         * @see <a href="here">http://code.google.com/p/android/issues/detail?id=10603</a>
         */
        public UUID getDeviceUuid() {
            return uuid;
        }
    }
}
