/*
 * Copyright (C) 2013 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.philliphsu.bottomsheetpickers.date;

import android.content.Context;
import android.content.res.Resources;
import android.graphics.drawable.StateListDrawable;
import android.view.View;
import android.view.ViewGroup;
import android.view.accessibility.AccessibilityEvent;
import android.widget.AdapterView;
import android.widget.AdapterView.OnItemClickListener;
import android.widget.ArrayAdapter;
import android.widget.ListView;
import android.widget.TextView;

import com.philliphsu.bottomsheetpickers.R;
import com.philliphsu.bottomsheetpickers.date.DatePickerDialog.OnDateChangedListener;

import java.util.ArrayList;
import java.util.List;

/**
 * Displays a selectable list of years.
 */
public class YearPickerView extends ListView implements OnItemClickListener, OnDateChangedListener {
    private static final String TAG = "YearPickerView";

    private final DatePickerController mController;
    private YearAdapter mAdapter;
    private int mViewSize;
    private int mChildSize;
    private TextViewWithIndicator mSelectedView;
    private final DateRangeHelper mDateRangeHelper;

    private boolean mThemeDark;
    private int mAccentColor;

    /**
     * @param context
     */
    public YearPickerView(Context context, DatePickerController controller) {
        super(context);
        mController = controller;
        mController.registerOnDateChangedListener(this);
        mDateRangeHelper = new DateRangeHelper(controller);
        ViewGroup.LayoutParams frame = new ViewGroup.LayoutParams(LayoutParams.MATCH_PARENT,
                LayoutParams.WRAP_CONTENT);
        setLayoutParams(frame);
        Resources res = context.getResources();
        mViewSize = res.getDimensionPixelOffset(R.dimen.bsp_date_picker_view_animator_height);
        mChildSize = res.getDimensionPixelOffset(R.dimen.bsp_year_label_height);
        setVerticalFadingEdgeEnabled(true);
        setFadingEdgeLength(mChildSize / 3);
        init(context);
        setOnItemClickListener(this);
        setSelector(new StateListDrawable());
        setDividerHeight(0);
        onDateChanged();
    }

    private void init(Context context) {
        ArrayList<String> years = new ArrayList<String>();
        for (int year = mController.getMinYear(); year <= mController.getMaxYear(); year++) {
            years.add(String.format("%d", year));
        }
        mAdapter = new YearAdapter(context, R.layout.bsp_year_label_text_view, years);
        setAdapter(mAdapter);
    }

    void setTheme(Context context, boolean themeDark) {
        mThemeDark = themeDark;
    }

    void setAccentColor(int color) {
        mAccentColor = color;
    }

    @Override
    public void onItemClick(AdapterView<?> parent, View view, int position, long id) {
        TextViewWithIndicator clickedView = (TextViewWithIndicator) view;
        if (clickedView != null && clickedView.isEnabled()) {
            if (clickedView != mSelectedView) {
                mSelectedView = clickedView;
            }
            mController.tryVibrate();
            mController.onYearSelected(getYearFromTextView(clickedView));
        }
    }

    private static int getYearFromTextView(TextView view) {
        return Integer.parseInt(view.getText().toString());
    }

    private class YearAdapter extends ArrayAdapter<String> {

        public YearAdapter(Context context, int resource, List<String> objects) {
            super(context, resource, objects);
        }

        @Override
        public View getView(int position, View convertView, ViewGroup parent) {
            TextViewWithIndicator v = (TextViewWithIndicator)
                    super.getView(position, convertView, parent);
            v.setTheme(parent.getContext(), mThemeDark);
            if (v instanceof TextViewWithHighlightIndicator && mAccentColor != 0) {
                ((TextViewWithHighlightIndicator) v).setHighlightIndicatorColor(mAccentColor);
            }
            v.requestLayout();
            int year = getYearFromTextView(v);
            CalendarDay selectedDay = mController.getSelectedDay();
            if (mDateRangeHelper != null && mDateRangeHelper.isOutOfRange(
                    year, selectedDay.month, selectedDay.day)) {
                v.setEnabled(false);
            } else {
                v.setEnabled(true);
                boolean selected = selectedDay.year == year;
                v.drawIndicator(selected);
                if (selected) {
                    mSelectedView = v;
                }
            }
            return v;
        }
    }

    public void postSetSelectionCentered(final int position) {
        postSetSelectionFromTop(position, mViewSize / 2 - mChildSize / 2);
    }

    public void postSetSelectionFromTop(final int position, final int offset) {
        post(new Runnable() {

            @Override
            public void run() {
                setSelectionFromTop(position, offset);
                requestLayout();
            }
        });
    }

    public int getFirstPositionOffset() {
        final View firstChild = getChildAt(0);
        if (firstChild == null) {
            return 0;
        }
        return firstChild.getTop();
    }

    @Override
    public void onDateChanged() {
        mAdapter.notifyDataSetChanged();
        postSetSelectionCentered(mController.getSelectedDay().year - mController.getMinYear());
    }

    @Override
    public void onInitializeAccessibilityEvent(AccessibilityEvent event) {
        super.onInitializeAccessibilityEvent(event);
        if (event.getEventType() == AccessibilityEvent.TYPE_VIEW_SCROLLED) {
            event.setFromIndex(0);
            event.setToIndex(0);
        }
    }
}
