package com.paystack.android.ui.paymentchannels

import androidx.compose.foundation.border
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.lazy.grid.GridCells
import androidx.compose.foundation.lazy.grid.LazyVerticalGrid
import androidx.compose.foundation.lazy.grid.items
import androidx.compose.material3.Icon
import androidx.compose.material3.LocalContentColor
import androidx.compose.material3.LocalTextStyle
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.CompositionLocalProvider
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.unit.dp
import com.paystack.android.ui.R
import com.paystack.android.ui.models.PaymentChannelMenuItem
import com.paystack.android.ui.theme.paymentChannelItemSpacing
import com.paystack.android.ui.theme.paymentChannelMenuItemWidth
import com.paystack.android.ui.theme.shapes
import com.paystack.androiddesignsystem.PaystackTheme

@Composable
internal fun PaymentChannelUi(
    channels: List<PaymentChannelMenuItem>,
    onChannelClick: (PaymentChannelMenuItem) -> Unit,
    modifier: Modifier = Modifier
) {
    Column(
        modifier = modifier.padding(horizontal = PaystackTheme.dimensions.spacing.double)
    ) {
        Text(
            text = stringResource(R.string.pstk_choose_payment_method_title),
            style = PaystackTheme.typography.body.body16M,
            textAlign = TextAlign.Center,
            color = PaystackTheme.colors.primaryColors.stackBlue,
            modifier = Modifier.fillMaxWidth()
        )

        Spacer(modifier = Modifier.height(PaystackTheme.dimensions.spacing.triple))

        CompositionLocalProvider(
            LocalContentColor provides PaystackTheme.colors.primaryColors.navy02,
            LocalTextStyle provides PaystackTheme.typography.body.body14M
        ) {
            LazyVerticalGrid(
                columns = GridCells.Adaptive(PaystackTheme.dimensions.paymentChannelMenuItemWidth),
                horizontalArrangement = Arrangement.spacedBy(PaystackTheme.dimensions.paymentChannelItemSpacing)
            ) {
                items(channels, { it.providerKey }) { channelOption ->
                    PaymentChannelMenuItem(
                        channelOption = channelOption,
                        onClick = { onChannelClick(channelOption) }
                    )
                }
            }
        }
    }
}

@Composable
private fun PaymentChannelMenuItem(
    channelOption: PaymentChannelMenuItem,
    onClick: () -> Unit,
    modifier: Modifier = Modifier
) {
    Column(
        horizontalAlignment = Alignment.Start,
        verticalArrangement = Arrangement.Center,
        modifier = modifier
            .border(
                width = MenuItemBorderWidth,
                color = PaystackTheme.colors.primaryColors.navy05,
                shape = PaystackTheme.shapes.large
            )
            .clip(PaystackTheme.shapes.large)
            .clickable(
                onClickLabel = stringResource(
                    R.string.pstk_action_pay_with_channel,
                    channelOption.name
                ),
                onClick = onClick
            )
            .padding(
                horizontal = PaystackTheme.dimensions.spacing.double,
                vertical = MenuItemVerticalPadding
            )
    ) {
        Icon(
            painter = painterResource(channelOption.iconRes),
            contentDescription = null
        )
        Text(text = channelOption.name)
    }
}

private val MenuItemBorderWidth = 1.dp
private val MenuItemVerticalPadding = 12.dp
