package com.paystack.android.ui.components.views.inputs.validators

import kotlinx.datetime.Clock
import kotlinx.datetime.TimeZone
import kotlinx.datetime.toLocalDateTime

/**
 * Singleton class that identifies the card that's been entered into the UI
 */
internal object CardValidationUtils {

    const val DEFAULT_CARD_LENGTH = 14

    /**
     * Method that identified the given card number and computers the card provider using
     * regex
     *
     * @param cardNumber
     * @return The Card Provider of the given card number
     */

    fun checkCardScheme(cardNumber: String): CardType {
        val visa = "^4[0-9]{0,15}$".toRegex()
        val masterCard =
            ("^5[1-5][0-9]{0,14}$|^2(22[1-9][0-9]{0,12}|2[3-9][0-9]{0,13}|[3-6][0-9]{0,14}" +
                "|7[01][0-9]{0,13}|720[0-9]{0,12})$").toRegex()
        val maestro = "^(5018|5020|5038|5893|6304|6390|67[0-9]{2})[0-9]{5,14}$".toRegex()
        val amex = "^3[47][0-9]{0,13}$".toRegex()
        val verve = "^((506([01]))|(507([89]))|(6500))[0-9]{0,19}$".toRegex()
        val diners = "^3(?:0[0-5]|[68][0-9])[0-9]{0,11}$".toRegex()
        val discover = "^6(?:011|5[0-9]{2}|4[4-9][0-9]|22[1-9])\\d*".toRegex()
        val jcb = "^35(?:2[89]|[3-8][0-9])\\d*".toRegex()

        val unformattedCardNumber = cardNumber.replace(" ", "")

        return when {
            unformattedCardNumber.matches(visa) -> CardType.VISA
            unformattedCardNumber.matches(masterCard) -> CardType.MASTERCARD
            unformattedCardNumber.matches(maestro) -> CardType.MAESTRO
            unformattedCardNumber.matches(amex) -> CardType.AMEX
            unformattedCardNumber.matches(diners) -> CardType.DINERS_CLUB
            unformattedCardNumber.matches(discover) -> CardType.DISCOVER
            unformattedCardNumber.matches(jcb) -> CardType.JCB
            unformattedCardNumber.matches(verve) -> CardType.VERVE
            else -> CardType.UNKNOWN
        }
    }

    /**
     * Method that gets the correct card length for the type of card identified
     * @param cardType
     * @return
     */
    fun getCardNumberLength(cardType: CardType): Int {
        return when (cardType) {
            CardType.AMEX -> 15
            CardType.VERVE, CardType.MAESTRO, CardType.JCB -> 19
            CardType.DINERS_CLUB -> 14
            CardType.UNKNOWN -> 20
            else -> 16
        }
    }

    /**
     * Function that returns the card CVV length for the card type. i.e American Express
     * has a CVV length of four. Every other card has a CVV length of three
     *
     * @param cardType
     * @return the correct CVV length for the given card
     */

    fun getCardCvvLength(cardType: CardType): Int {
        return when (cardType) {
            CardType.AMEX -> 4
            else -> 3
        }
    }

    /**
     * This method uses the Luhn's  algorithm (https://en.wikipedia.org/wiki/Luhn_algorithm)
     * in order to validate a credit card number.
     * It does NOT check if the card number is active but checks to make sure the checksum of the
     * number is accurate and can be used to submit payment information
     *
     * @param cardNumber
     * @return's whether the checksum is valid or not if FALSE the card is not valid to be submitted
     * for use, else returns true
     */
    fun isCardValid(cardNumber: String): Boolean {
        val sanitizedCardNumber = cardNumber.replace(" ", "")
        if (cardNumber.length < DEFAULT_CARD_LENGTH) return false

        var checksum = 0
        for (i in sanitizedCardNumber.length - 1 downTo 0 step 2) {
            checksum += sanitizedCardNumber[i] - '0'
        }
        for (i in sanitizedCardNumber.length - 2 downTo 0 step 2) {
            val digit: Int = (sanitizedCardNumber[i] - '0') * 2
            checksum += if (digit > 9) digit - 9 else digit
        }
        return checksum % 10 == 0
    }

    /**
     * Method that returns whether the card expiry date is valid for the entered data.
     *
     * @param enteredDate  the date entered as a string in the card details form
     * @return whether the given date is valid.
     */
    fun isCardExpiryDateValid(enteredDate: String): Boolean {
        val parts = enteredDate.split("/")
        if (parts.size != 2) return false

        val month = parts[0].toIntOrNull()
        val year = parts[1].toIntOrNull()

        if (month == null || year == null) return false
        if (month < 1 || month > 12) return false

        val currentYear = Clock.System.now().toLocalDateTime(TimeZone.currentSystemDefault()).year
        val currentMonth =
            Clock.System.now().toLocalDateTime(TimeZone.currentSystemDefault()).monthNumber

        return when {
            year < currentYear % 100 -> false
            year == currentYear % 100 && month < currentMonth -> false
            else -> true
        }
    }
}
