package com.paystack.android_sdk.ui.components.views.inputs

import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.heightIn
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.material3.Icon
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.unit.dp
import com.paystack.android_sdk.ui.R
import com.paystack.android_sdk.ui.components.views.inputs.textfield.StackTextField
import com.paystack.android_sdk.ui.components.views.inputs.validators.CardNumberFormatter
import com.paystack.android_sdk.ui.components.views.inputs.validators.CardType
import com.paystack.android_sdk.ui.components.views.inputs.validators.CardValidationUtils.DEFAULT_CARD_LENGTH
import com.paystack.android_sdk.ui.components.views.inputs.validators.CardValidationUtils.checkCardScheme

@Composable
internal fun CardNumberInputField(
    cardText: String,
    maxInputLength: Int,
    onCardTextChanged: (String) -> Unit,
    isValidCard: Boolean = false,
    modifier: Modifier = Modifier,
) {
    val isError = !isValidCard && cardText.length >= DEFAULT_CARD_LENGTH
    val cardType = checkCardScheme(cardText)

    StackTextField(
        modifier = modifier.fillMaxWidth(),
        value = cardText,
        onValueChange = { updatedNumber ->
            if (updatedNumber.length <= maxInputLength) {
                onCardTextChanged(updatedNumber)
            }
        },
        label = { Text(text = stringResource(id = R.string.pstk_card_payment_card_number_label)) },
        placeholder = { Text(text = stringResource(id = R.string.pstk_card_payment_card_number_placeholder)) },
        supportingText = {
            if (isError) {
                Text(text = stringResource(id = R.string.pstk_card_payment_card_number_error))
            }
        },
        trailingIcon = {
            Box(
                modifier = Modifier
                    .padding(end = 16.dp)
                    .heightIn(max = 24.dp)
            ) {
                Icon(
                    painter = painterResource(id = getCardTypeDrawable(cardType)),
                    contentDescription = "",
                    tint = Color.Unspecified
                )
            }
        },
        keyboardOptions = KeyboardOptions.Default.copy(keyboardType = KeyboardType.Number),
        singleLine = true,
        visualTransformation = { updatedNumber ->
            CardNumberFormatter.formatCardNumberByType(
                cardType = cardType,
                cardNumber = updatedNumber
            )
        },
        isError = isError
    )
}

private fun getCardTypeDrawable(cardType: CardType): Int {
    return when (cardType) {
        CardType.VISA -> R.drawable.pstk_visa
        CardType.AMEX -> R.drawable.pstk_amex
        CardType.DINERS_CLUB -> R.drawable.pstk_diners
        CardType.JCB -> R.drawable.pstk_jcb
        CardType.DISCOVER -> R.drawable.pstk_discover
        CardType.MAESTRO -> R.drawable.pstk_maestro
        CardType.MASTERCARD -> R.drawable.pstk_mastercard
        CardType.VERVE -> R.drawable.pstk_verve
        CardType.UNKNOWN -> R.drawable.pstk_unknown
    }
}
